;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Classes.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)

(in-package "DTP")

;;;----------------------------------------------------------------------------

(defclass dtp-object () ()
  (:documentation "Superclass of all DTP objects") )

;;;----------------------------------------------------------------------------

(defclass dtp-proof-node (dtp-object) ()
  (:documentation "Node in the proof space") )

;;;----------------------------------------------------------------------------

(defclass dtp-subgoal (dtp-proof-node)
	  ((literal
	    :initarg :literal
	    :initform nil
	    :type (or literal-node null) )
	   (answers
	    :initform nil
	    :type list )
	   (conjuncts-to-propogate-to
	    :initarg :propogate-to
	    :initform nil
	    :type list )
	   (ancestor-subgoals
	    :initform :uninitialized
	    :type (or list (eql :uninitialized))
	    :documentation "Used for model elimination" )
	   (inferences
	    :initform :uninitialized
	    :type (or list (eql :uninitialized))
	    :documentation "List of conjunction nodes" )
           (forward-inference-conjunctions
            :initform nil
            :type list
            :documentation "Conjunction nodes that are waiting for propogation"
            )
	   (used-inferences
	    :initform nil
	    :type list
	    :documentation "Only used for tracing" ))
  (:documentation "Subgoal node in proof space") )

(defmethod print-object ((object dtp-subgoal) stream)
  (with-slots (literal answers inferences) object
    (format stream "#<DTP Subgoal ")
    (if (literal-node-p literal)
	(print-literal-node literal :s stream)
      (format stream "?") )
    (format stream " with ~D answer~:P" (length answers))
    (when (listp inferences)
      (if inferences
	  (format stream " [~D task~:P pending]" (length inferences))
	(format stream " [complete]") ))
    (format stream ">") ))

;;;----------------------------------------------------------------------------

(defclass dtp-conjunction (dtp-proof-node)
	  ((list
	    :initarg :list
	    :initform nil
	    :type list
	    :documentation "List of conjuncts" )
	   (answers :initform nil)
	   (parent-subgoal
	    :initarg :parent
	    :initform nil
	    :type (or dtp-subgoal null)
	    :documentation "Parent of NIL means this is a query-conjunction" )
	   (stack-pointer
	    :initform 0
	    :type (integer -1 *)
	    :documentation "number of current conjunct" )
	   (binding-list
	    :initarg :binding-list
	    :initform nil
	    :type binding-list
	    :documentation "From inference" )
	   (label
	    :initarg :label
	    :initform nil
	    :type (or label null)
	    :documentation "From inference" )
	   (ae-binding-list
	    :initarg :ae-binding-list
	    :initform nil
	    :type binding-list
	    :documentation
	    "Needed for disjunctive answers via answer extraction" )
	   (origin
	    :initarg :origin
	    :type symbol
	    :documentation "KB node ID of parent rule, used for tracing" ))
  (:documentation "Conjunction (from inference)") )

(defmethod print-object ((object dtp-conjunction) stream)
  (with-slots (list) object
    (format stream "#<DTP Conjunction")
    (when list
      (format stream ":")
      (dolist (conjunct list)
	(when (typep conjunct 'dtp-conjunct)
	  (with-slots (literal answer-count) conjunct
	    (format stream " ")
	    (if (literal-node-p literal)
		(print-literal-node literal :s stream)
	      (format stream "?") )
	    (format stream "/~D" answer-count) ))))
    (format stream ">") ))

;;;----------------------------------------------------------------------------

(defclass dtp-conjunct (dtp-proof-node)
	  ((literal
	    :initarg :literal
	    :initform nil
	    :type (or literal-node null) )
	   (parent-conjunction
	    :initarg :parent
	    :type dtp-conjunction
	    :documentation "Must be initialized when first created" )
	   (binding-list
	    :initform nil
	    :type list
	    :documentation "Apply to literal, then search for subgoal" )
	   (transform-binding-list
	    :initform nil
	    :type list
	    :documentation "Apply to answers from subgoal before valid" )
	   (answer-count
	    :initform 0
	    :type (integer 0 *) )
	   (subgoal
	    :initform :uninitialized )
	   (used-subgoals
	    :initform nil
	    :type list
	    :documentation "For tracing only" ))
  (:documentation "Conjunct") )

(defmethod print-object ((object dtp-conjunct) stream)
  (with-slots (literal answer-count) object
    (format stream "#<DTP Conjunct ")
    (if (literal-node-p literal)
	(print-literal-node literal :s stream)
      (format stream "?") )
    (format stream " with ~D answer~:P>" answer-count) ))

;;;----------------------------------------------------------------------------
