;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Tracing.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)

(in-package "DTP")

(eval-when (compile load eval)
  (export
   '(trace-options) ))

;;;----------------------------------------------------------------------------

(defun reset-tracing ()
  (setq *tracemap* (make-tracemap)) )

;;;----------------------------------------------------------------------------

(defun show-trace (node)
  (let ((did-prologue nil))
    (when (trace-goals *tracemap*)
      (setq did-prologue t)
      (if (agenda-node-p node)
	  (progn
	    (prologue-trace node)
	    (format t "~{~:(~A~)~}~{ of ~A~}~{ and ~A~}~%"
		    (when (anode-origin node) (list (anode-origin node)))
		    (when (anode-parents node)
		      (list (first (anode-parents node))) )
		    (cdr (anode-parents node)) ))
	(format t "~A [KB]~%" (node-id node)) ))
    (when (and (trace-assumptions *tracemap*)
	       (agenda-node-p node)
	       (not (anode-pruned-p node))
	       (eq (anode-origin node) 'assumption) )
      (let (my-assumpt parent parent-assumpt plug-parent-assumpt new-assumpt)
	(setq my-assumpt (anode-assumptions node))
	(setq parent (first (anode-parents node)))
	(setq parent-assumpt
	  (when parent (anode-assumptions (id-to-node parent))) )
	(setq plug-parent-assumpt
	  (plug parent-assumpt (anode-binding-list node)) )
	(setq new-assumpt
	  (set-difference my-assumpt plug-parent-assumpt :test #'equal) )
	(when new-assumpt
	  (setq did-prologue t)
	  (if (trace-goals *tracemap*)
	      (format t "~7T")
	    (prologue-trace node) ))
	(when new-assumpt
	  (format t "Assuming ~A~{, ~A~} [~D object~:P]~%"
		  (car new-assumpt) (cdr new-assumpt)
		  (assumption-count node) ))
	))
    (when (trace-clauses *tracemap*)
      (unless did-prologue (prologue-trace node))
      (setq did-prologue t)
      (format t "~7T")
      (print-clause-node (node-clause node))
      (format t "~%") )
    (when (and (trace-designs *tracemap*)
	       (proof-assumables *proof*) ; Insure that this is a residue proof
	       (not (anode-pruned-p node))
	       (only-designs node) )
      (unless did-prologue (prologue-trace node))
      (setq did-prologue t)
      (format t "Design +~{ ~A~}~%"
	      (mapcar #'(lambda (x) (literal-terms x))
		      (clause-literals (node-clause node))) )
      (format t "~7T~A~%" (anode-assumptions node)) )
    ))

;;;----------------------------------------------------------------------------

(defun prologue-trace (node)
  (if (anode-pruned-p node)
      (format t "~A" (anode-pruned-p node))
    (format t "~A" (node-id node)) )
  (format t "~7T~2,'0D " (anode-depth node))
  (dotimes (i (anode-depth node))
    (format t ".") )
  (format t " ~D " (anode-breadth node))
  )

;;;----------------------------------------------------------------------------

(defun only-designs (node)
  (dolist (lit (clause-literals (node-clause node)))
    (unless (and (literal-negated-p lit)
		 (eq (literal-relation lit) 'design-works) )
      (return-from only-designs nil) ))
  t )

;;;----------------------------------------------------------------------------

(defun trace-options
    (&key
     (everything nil ev)
     (nothing nil no)

     (answers nil ans)
     (goals nil g)
     (assumptions nil a)
     (clauses nil c)
     (designs nil d)
     (subsumptions nil s)
     (cache-answers nil ca)
     (cache-postponing nil cp)
     (failures nil f)
     (pure-literals nil pl)

     (load nil ld)
     (tests nil tt)

     (output t) )
    (when (and ev everything)
      (setf (trace-answers *tracemap*) t
	    (trace-goals *tracemap*) t
	    (trace-assumptions *tracemap*) t
	    (trace-clauses *tracemap*) t
	    (trace-designs *tracemap*) t
	    (trace-subsumptions *tracemap*) t
	    (trace-cache-answers *tracemap*) t
	    (trace-cache-postponing *tracemap*) t
	    (trace-failures *tracemap*) t
	    (trace-pure-literals *tracemap*) t
	    (trace-load *tracemap*) t
	    (trace-tests *tracemap*) t ))
    (when (and no nothing)
      (setf (trace-answers *tracemap*) nil
	    (trace-goals *tracemap*) nil
	    (trace-assumptions *tracemap*) nil
	    (trace-clauses *tracemap*) nil
	    (trace-designs *tracemap*) nil
	    (trace-subsumptions *tracemap*) nil
	    (trace-cache-answers *tracemap*) nil
	    (trace-cache-postponing *tracemap*) nil
	    (trace-failures *tracemap*) nil
	    (trace-pure-literals *tracemap*) nil
	    (trace-load *tracemap*) nil
	    (trace-tests *tracemap*) nil ))
    (when ans (setf (trace-answers *tracemap*) answers))
    (when g (setf (trace-goals *tracemap*) goals))
    (when a (setf (trace-assumptions *tracemap*) assumptions))
    (when c (setf (trace-clauses *tracemap*) clauses))
    (when d (setf (trace-designs *tracemap*) designs))
    (when s (setf (trace-subsumptions *tracemap*) subsumptions))
    (when ca (setf (trace-cache-answers *tracemap*) cache-answers))
    (when cp (setf (trace-cache-postponing *tracemap*) cache-postponing))
    (when f (setf (trace-failures *tracemap*) failures))
    (when pl (setf (trace-pure-literals *tracemap*) pure-literals))
    (when ld (setf (trace-load *tracemap*) load))
    (when tt (setf (trace-tests *tracemap*) tests))
    (unless output (return-from trace-options))
    (format t "~&Answers ~A, Goals ~A, Assump ~A, Clauses ~A~%"
	    (trace-answers *tracemap*) (trace-goals *tracemap*)
	    (trace-assumptions *tracemap*) (trace-clauses *tracemap*) )
    (format t "Designs ~A, Subsump ~A, Cache [Answers ~A, Postponing ~A]~%"
	    (trace-designs *tracemap*) (trace-subsumptions *tracemap*)
	    (trace-cache-answers *tracemap*)
	    (trace-cache-postponing *tracemap*) )
    (format t "Failure ~A, Pure literals ~A, Load ~A, Tests ~A~%"
	    (trace-failures *tracemap*) (trace-pure-literals *tracemap*)
	    (trace-load *tracemap*) (trace-tests *tracemap*) )
    (values) )

;;;----------------------------------------------------------------------------
