;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		TopLevel.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)

(in-package "DTP")

(eval-when (compile load eval)
  (export
   '(prove show-contents all-theories reset-dtp
     make-theory-from-sentences save-sentence-in-theory
     drop-sentence-from-theory
     options iteration-options output-form-options
     justify id-to-node sentences-in )))

;;;----------------------------------------------------------------------------
;;;
;;;	Returns three values: Bound QUERY, label, assumptions
;;;	(Or, if more than one answer requested, each value is a list)
;;;	If RETURN-FORM is non-nil, then the first value is RETURN-FORM bound
;;;	  rather than QUERY bound, using answers from QUERY.
;;;
;;;	The latest proof state is available in *proof*

(defun prove (query &key (all-answers nil) (max-answers nil) (return-form nil))
  (declare (type integer max-answers))
  (loop
      with solutions = nil
      with solution
      until (or (and max-answers (= max-answers (length solutions)))
		(and (null max-answers) solutions) )

      do
	(do-iteration proof-node-search-bound *node-iteration*)
	(do-iteration proof-assumption-search-bound *assumption-iteration*)

	(when (or *node-iteration* *assumption-iteration*)
	  (format t "~%____________________________________________________~%")
	  (when *node-iteration*
	    (format t "Node search bound is ~D~%"
		    (proof-node-search-bound *proof*) ))
	  (when *assumption-iteration*
	    (format t "Assumption search bound is ~D~%"
		    (proof-node-search-bound *proof*) ))
	  (format t "~%") )

	(cond
	 (max-answers
	  (setq solutions (prove-n-answers query max-answers)) )
	 (all-answers
	  (setq solutions (prove-all-answers query))
	  (loop-finish) )
	 (t
	  (setq solutions (list (prove-first-answer query))) ))

	(unless (or *node-iteration* *assumption-iteration*)
	  (loop-finish) )
	
      finally
	(when return-form
	  (mapc
	   #'(lambda (x)
	       (when x
		 (setf (answer-logic x)
		   (alter-form (answer-logic x) query return-form) )))
	   solutions ))
	(cond
	 ((or all-answers max-answers)
	  (return (values (mapcar #'answer-logic solutions)
			  (mapcar #'answer-visible-label solutions)
			  (mapcar
			   #'(lambda (x) (mapcar #'literal-to-list x))
			   (mapcar #'answer-assumptions solutions) ))))
	 ((setq solution (first solutions))
	  (return (values (answer-logic solution)
			  (answer-visible-label solution)
			  (mapcar #'literal-to-list
				  (answer-assumptions solution) ))))
	 (t
	  (return (values nil nil nil)) ))
	))

;;;----------------------------------------------------------------------------

(defun show-contents (&optional (theory nil))
  (format t "~&")
  (dolist (th (if theory (list theory) (included-active-theory-names)))
    (dolist (node (theory-contents th))
      (format t "~A~15T" (node-id node))
      (print-clause-node (node-clause node) :as-rule t)
      (format t "~%") ))
  (values) )

;;;----------------------------------------------------------------------------

(defun all-theories ()
  (reduce
   #'union
   (list
    (proof-active-theories *proof*)
    (all-include-theories)
    (all-kb-theories) )))

;;;----------------------------------------------------------------------------

(defun reset-dtp ()
  (setq *proof* (make-proof))
  (setq *node-iteration* nil)
  (setq *assumption-iteration* nil)
  (setq *output-form-map* (make-output-form-map))
  (reset-tracing)
  (reset-hierarchy)
  (reset-database) )

;;;----------------------------------------------------------------------------

(defun make-theory-from-sentences (theory-name sentence-label-pairs)
  (let (cnf-label-pairs literal-lists nodes)
    (setq cnf-label-pairs
      (mapcar #'(lambda (slp)
		  (let ((label (cdr slp)))
		    (mapcar #'(lambda (s) (cons s label))
			    (sentence-to-cnf (car slp)) )))
	      sentence-label-pairs ))
    (setq cnf-label-pairs (apply #'append cnf-label-pairs))
    (setq literal-lists
      (mapcar #'(lambda (x)
		  (list (mapcar #'list-to-literal (car x)) (cdr x)) )
	      cnf-label-pairs ))
    (setq nodes
      (loop
	  for (literal-list label) in literal-lists
	  for count from 1
	  collect
	    (make-kb-node
	     :id (make-new-id theory-name count)
	     :clause (make-clause-node :literals literal-list :label label) )))
    (make-theory-from-nodes nodes theory-name)
    theory-name ))

;;;----------------------------------------------------------------------------

(defun save-sentence-in-theory
    (sentence &key (theory-name 'global) (label nil))
  (loop
      with cnf = (sentence-to-cnf sentence)
      with literal-lists =
	(mapcar #'(lambda (x) (mapcar #'list-to-literal x)) cnf)
      for literal-list in literal-lists
      for count from (1+ (last-id-count theory-name))
      for id = (make-new-id theory-name count)
      for new-node =
	(make-kb-node
	 :id id
	 :clause (make-clause-node :literals literal-list :label label) )
      collect id
      do (save-node-in-theory new-node theory-name) ))

;;;----------------------------------------------------------------------------

(defun drop-sentence-from-theory
    (sentence &key (theory-name 'global) (test #'equal))
  "Locate the node(s) in the theory corresponding to SENTENCE and remove them"
  (loop
      with theory = (get-theory-structure theory-name)
      while theory
      for dnf in (sentence-to-cnf sentence)
      collect
	(loop
	    for node in (theory-nodes theory)
	    when (clause-list-equal-p (node-clause node) dnf :test test)
	    do (drop-node-from-theory node theory)
	       (return (node-id node)) )
      into success
      finally (return (remove nil success)) ))

;;;----------------------------------------------------------------------------

(defun options
    (&key
     (active-theories nil at-p)
     (assumables nil a-p)
     (advise nil ad-p)
     
     (output t) )
  (setq assumables
    (mapcar #'nliteral-rename-all-variables
	    (mapcar #'list-to-literal assumables) ))
  (when at-p (setf (proof-active-theories *proof*) active-theories))
  (when a-p (setf (proof-assumables *proof*) assumables))
  (when ad-p (setf (proof-advise *proof*) advise))
  (unless output (return-from options))
  (format t "~&Active theor~@P: ~A~{, ~A~}~%"
	  (length (proof-active-theories *proof*))
	  (car (proof-active-theories *proof*))
	  (cdr (proof-active-theories *proof*)) )
  (if (proof-assumables *proof*)
      (format t "Assumables: ~A~{, ~A~}~%"
	      (first (proof-assumables *proof*))
	      (rest (proof-assumables *proof*)) )
    (format t "No assumables~%") )
  (if (proof-advise *proof*)
      (format t "Manual search control [Advise]~%")
    (format t "Automatic search control [Advising off]~%") )
  (values) )

;;;----------------------------------------------------------------------------

(defun iteration-options
    (&key
     (unrestricted-nodes nil)
     (node-max nil nm-p)
     (node-start nil ns-p)
     (node-increment nil ni-p)
     (node-depth-factor nil ndf-p)
     (node-breadth-factor nil nbf-p)
     (unrestricted-assumptions nil)
     (assumption-max nil am-p)
     (assumption-start nil as-p)
     (assumption-increment nil ai-p)
     
     (output t) )

  (when unrestricted-nodes (setf *node-iteration* nil))
  (when (or nm-p ns-p ni-p ndf-p nbf-p)
    (setf *node-iteration* (make-iteratemap)) )
  (when nm-p (setf (iterate-max-bound *node-iteration*) node-max))
  (when ns-p (setf (iterate-start *node-iteration*) node-start))
  (when ni-p (setf (iterate-increment *node-iteration*) node-increment))
  (when ndf-p (setf (iterate-depth-factor *node-iteration*) node-depth-factor))
  (when nbf-p
    (setf (iterate-breadth-factor *node-iteration*) node-breadth-factor) )

  (when unrestricted-assumptions (setf *assumption-iteration* nil))
  (when (or am-p as-p ai-p)
    (setf *assumption-iteration* (make-iteratemap)) )
  (when am-p (setf (iterate-max-bound *assumption-iteration*) assumption-max))
  (when as-p (setf (iterate-start *assumption-iteration*) assumption-start))
  (when ai-p
    (setf (iterate-increment *assumption-iteration*) assumption-increment) )
  
  (unless output (return-from iteration-options))
  
  (format t "Nodes: ")
  (if *node-iteration*
      (format t "Max: ~A, Start: ~A, Inc: ~A, DF: ~A, BF: ~A~%"
	      (iterate-max-bound *node-iteration*)
	      (iterate-start *node-iteration*)
	      (iterate-increment *node-iteration*)
	      (iterate-depth-factor *node-iteration*)
	      (iterate-breadth-factor *node-iteration*) )
    (format t "Unrestricted~%") )
  (format t "Assumptions: ")
  (if *assumption-iteration*
      (format t "Max: ~A, Start: ~A, Inc: ~A~%"
	      (iterate-max-bound *assumption-iteration*)
	      (iterate-start *assumption-iteration*)
	      (iterate-increment *assumption-iteration*) )
    (format t "Unrestricted~%") )
  (values) )

;;;----------------------------------------------------------------------------

(defun output-form-options
    (&key
     (display-as-lists nil dal)
     (show-renamed-variables nil srv)
     (show-answer-literals nil sal)
     (output t) )
  (when dal
    (setf (output-form-display-as-lists *output-form-map*) display-as-lists) )
  (when srv
    (setf (output-form-show-renamed-variables *output-form-map*)
      show-renamed-variables ))
  (when sal
    (setf (output-form-show-answer-literals *output-form-map*)
      show-answer-literals ))
  (unless output (return-from output-form-options))
  (format t "Clauses displayed as ")
  (if (output-form-display-as-lists *output-form-map*)
      (format t "lists, ")
    (format t "english, ") )
  (format t "Renamed variables ")
  (if (output-form-show-renamed-variables *output-form-map*)
      (format t "shown, ")
    (format t "hidden, ") )
  (format t "Answer literals ")
  (if (output-form-show-answer-literals *output-form-map*)
      (format t "shown~%")
    (format t "hidden~%") )
  (values) )

;;;----------------------------------------------------------------------------

(defun justify (node-id &key (start-id nil))
  (let ((all-nodes nil))
    (loop
	for to-go = (list node-id)
	then (append (anode-parents node) (cdr to-go))
	until (null to-go)
	for id = (first to-go)
	for node = (id-to-node id)
	when node
	do (push node all-nodes) )
    (setq all-nodes (remove-duplicates all-nodes))
    (format t "~&")
    (dolist (node all-nodes)
      (when (and start-id (eq start-id (node-id node)))
	(setq start-id nil) )
      (unless start-id
	(show-trace node) ))
    (values) ))

;;;----------------------------------------------------------------------------

(defun id-to-node (id)
  (let ((node (find-kb-node-with-id id)))
    (if node
	node
      (find-agenda-node-with-id id) )))

;;;----------------------------------------------------------------------------

(defun sentences-in (theory-name &key (with-atom nil))
  "Return list of sentences in theory named THEORY-NAME"
  (loop
      for node in (theory-contents theory-name)
      for clause = (node-clause node)
      collect (simplify-dnf (cons 'or (clause-to-list clause)))
      into sentences
      finally
	(if with-atom
	    (return
	      (remove-if-not
	       #'(lambda (x) (tree-find with-atom x))
	       sentences ))
	  (return sentences) )))

;;;----------------------------------------------------------------------------
