;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Residue.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)

(in-package "DTP")

;;;----------------------------------------------------------------------------

(defun assumption-count (node)
  (if (proof-fn-object-count *proof*)
      (funcall (proof-fn-object-count *proof*) (anode-assumptions node))
    (length (anode-assumptions node)) ))

;;;----------------------------------------------------------------------------

(defun consistent-assumptions (node)
  "Returns t, nil, or a binding list (which also modifies NODE)"
  (let ((assumptions (anode-assumptions node))
        (binding nil) )
    (cond
     ((eq (anode-origin node) 'factor)
      (setq binding t) )
     ((eq (anode-origin node) 'assumption)
      (if (proof-fn-unify-assumption *proof*)
	  (setq binding
	    (funcall (proof-fn-unify-assumption *proof*)
		     (car assumptions) (cdr assumptions) ))
	(setq binding '((t . t))) ))
     (t
      (let (remaining bl)
	(dolist (a assumptions)
	  (setq remaining (cdr (member a assumptions :test #'equal)))
	  (if (proof-fn-unify-assumption *proof*)
	      (setq bl
		(funcall (proof-fn-unify-assumption *proof*) a remaining) )
	    (setq bl '((t . t))) )
	  (when (null bl)
	    (setq binding nil)
	    (return) )
	  (when (listp bl)
	    (setq binding (append binding bl)) ))
	(unless binding (setq binding t))
	)))
    (when (listp binding)
      (setq binding (remove-duplicates binding :test #'equal))
      (clause-plug (node-clause node) binding)
      (setf (anode-assumptions node)
	(remove-duplicates
	 (plug (anode-assumptions node) binding)
	 :test #'equal :from-end t ))
      (setf (anode-binding-list node)
	(remove-duplicates
	 (append (anode-binding-list node) binding) :test #'equal )))
    binding
    ))

;;;----------------------------------------------------------------------------

(defun same-assumptions (node1 node2)
  (null
   (set-exclusive-or
    (assumptions-of-node node1)
    (assumptions-of-node node2)
    :test #'samep )))

(defun subset-assumptions (node1 node2)
  (subsetp
   (assumptions-of-node node1)
   (assumptions-of-node node2)
   :test #'samep ))

;;;----------------------------------------------------------------------------

(defun assumption-resolution (node)
  (loop
      with clause = (node-clause node)
      with assumptions = nil
      with (unifier new-assumption)
      for literal in (clause-literals clause)
      for matching-assumption =
	(find literal (proof-assumables *proof*)
	      :test #'literal-negated-pair-p )
      when matching-assumption
      do
	(setq unifier (literal-mgu literal matching-assumption :ignore-sign t))
	(setq new-assumption (literal-plug matching-assumption unifier))
	(cond
     
	 ((groundp (literal-terms new-assumption))
		 
	  (let ((new-bl
		 (merge-binding-lists unifier (anode-binding-list node)) ))
	    (push (make-agenda-node
		   :clause (clause-plug (clause-remove literal clause)
					new-bl )
		   :binding-list new-bl
		   :assumptions (remove-duplicates
				 (cons new-assumption
				       (assumptions-of-node node) )
				 :test #'equal )
		   :parents (list (node-id node))
		   :origin 'assumption )
		  assumptions )))

	 ((and (varp (first (literal-terms new-assumption)))
	       (groundp (second (literal-terms new-assumption))) )
	  ;; (type $var <gate-type>) or (drives $var <input>)
	       
	  (let ((doing-type (eq (first new-assumption) 'type))
		(new-bindings nil)
		(new-var (first (literal-terms new-assumption)))
		(fixed (second (literal-terms new-assumption))) )
	    (when doing-type
	      (push (cons new-var (make-new-gate fixed)) new-bindings) )
	   
	    (loop
		for assumpt in (assumptions-of-node node)
		when (and (eq (literal-relation assumpt)
			      (literal-relation new-assumption) )
			  (equal (second (literal-terms assumpt)) fixed) )
		do (push (cons new-var (first (literal-terms assumpt)))
			 new-bindings ))
	      
	    (loop
		with short-clause = (clause-remove literal clause)
		for new-b in new-bindings
		for new-bl =
		  (merge-binding-lists
		   (list new-b) unifier (anode-binding-list node) )
		with (new-clause new-assumpt)
		do (setq new-clause (clause-plug short-clause new-bl))
		   (setq new-assumpt
		     (remove-duplicates
		      (cons (plug new-assumption new-bl)
			    (assumptions-of-node node) )
		      :test #'equal ))
		   (push (make-agenda-node
			  :clause new-clause
			  :binding-list new-bl
			  :assumptions new-assumpt
			  :parents (list (node-id node))
			  :origin 'assumption )
			 assumptions ))
	    )))

	(return-from assumption-resolution (reverse assumptions)) ))

;;;----------------------------------------------------------------------------

(defun make-new-gate (str)
  (intern (format nil "~A-~D" str (incf (proof-gate-count *proof*)))) )

;;;----------------------------------------------------------------------------
