;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Literals.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)

(in-package "DTP")

;;;----------------------------------------------------------------------------
;;;
;;;	Data Structures

(defstruct (literal-node
	    (:conc-name literal-)
	    (:print-function literal-node-print-function) )
  "A literal"
  negated-p
  relation
  terms )

;;;----------------------------------------------------------------------------

(defun literal-node-print-function (structure stream depth)
  (declare (ignore depth))
  (format stream "<")
  (print-literal-node structure :s stream)
  (format stream ">") )

(defun print-literal-node (node &key (s t) (flip-negation nil))
  (cond
   ((output-form-display-as-lists *output-form-map*)
    (print-literal-node-as-list node :s s :flip-negation flip-negation) )
   (t
    (print-literal-node-as-logic node :s s :flip-negation flip-negation) )))

(defun print-literal-node-as-list (node &key (s t) (flip-negation nil))
  (unless (eq (literal-negated-p node) flip-negation)
    (format s "(not ") )
  (format s "(")
  (format s "~:(~A~)" (literal-relation node))
  (when (literal-terms node)
    (format s "~{ ~S~}" (literal-terms node)) )
  (format s ")")
  (unless (eq (literal-negated-p node) flip-negation)
    (format s ")") ))

(defun print-literal-node-as-logic (node &key (s t) (flip-negation nil))
  (unless (eq (literal-negated-p node) flip-negation)
    (format s "~~") )
  (format s "~:(~A~)" (literal-relation node))
  (when (literal-terms node)
    (let ((term-strings (mapcar #'term-to-string (literal-terms node))))
      (format s "(~A~{,~A~})" (car term-strings) (cdr term-strings)) )))

(defun term-to-string (term &optional (s nil))
  "Variable terms -> lowercase string, Contant terms -> capitalized string"
  (cond
   ((varp term)
    (format s "~(~A~)"
	    (if (output-form-show-renamed-variables *output-form-map*)
		term
	      (subseq (symbol-name term)
		      0 (position #\_ (symbol-name term)) ))
	    ))
   ((consp term)
    (with-output-to-string (s)
      (format s "~:(~A~)" (first term))
      (when (rest term) (format s "("))
      (loop
	  for (subterm . rest-subterms) on (rest term)
	  do (term-to-string subterm s)
	  when rest-subterms
	  do (format s ",") )
      (when (rest term) (format s ")")) ))
   ((stringp term)
    (format s "~S" term) )
   (t
    (format s "~:(~A~)" term) )))

;;;----------------------------------------------------------------------------

(defun literal-match-p (lit1 lit2)
  (and (eq (literal-negated-p lit1) (literal-negated-p lit2))
       (eq (literal-relation lit1) (literal-relation lit2))
       (samep (literal-terms lit1) (literal-terms lit2)) ))

;;;----------------------------------------------------------------------------

(defun list-to-literal (list &optional (replacement-bindings nil))
  (let ((lit (make-literal-node)))
    (when (eq 'not (first list))
      (setf (literal-negated-p lit) t)
      (setq list (second list)) )
    (setf (literal-relation lit) (first list))
    (if replacement-bindings
	(setf (literal-terms lit) (plug (cdr list) replacement-bindings))
      (setf (literal-terms lit) (cdr list)) )
    lit ))

;;;----------------------------------------------------------------------------

(defun literal-to-list (literal &key (ignore-negation nil))
  (let ((new-list (cons (literal-relation literal) (literal-terms literal))))
    (unless ignore-negation
      (when (literal-negated-p literal)
	(setq new-list (list 'not new-list)) ))
    new-list ))

;;;----------------------------------------------------------------------------

(defun literal-list-equal-p (literal list &key (test #'equal))
  "True iff the literal is equivalent to the list representation"
  (let ((list-negation (eq (first list) 'not)))
    (when list-negation (setq list (second list)))
    (and (eq (literal-negated-p literal) list-negation)
	 (eq (literal-relation literal) (first list))
	 (funcall test (literal-terms literal) (rest list)) )))

;;;----------------------------------------------------------------------------

(defun literal-plug (literal binding-list)
  "Return a new literal, which is a copy of LITERAL with BINDING-LIST applied"
  (let ((copy (copy-literal-node literal)))
    (setf (literal-terms copy) (plug (literal-terms copy) binding-list))
    copy ))

(defun nliteral-plug (literal binding-list)
  "Destructively modify LITERAL by applying BINDING-LIST"
  (setf (literal-terms literal) (plug (literal-terms literal) binding-list))
  literal )

;;;----------------------------------------------------------------------------

(defun literal-flip-negation (literal)
  "Return a copy of LITERAL with opposite sign"
  (let ((copy (copy-literal-node literal)))
    (setf (literal-negated-p copy) (not (literal-negated-p copy)))
    copy ))

(defun nliteral-flip-negation (literal)
  "Destructively modify LITERAL to have opposite sign"
  (setf (literal-negated-p literal) (not (literal-negated-p literal)))
  literal )

;;;----------------------------------------------------------------------------

(defun literal-negated-pair-p (lit1 lit2 &key (test #'my-unifyp))
  (and lit1 lit2
       (not (eq (literal-negated-p lit1) (literal-negated-p lit2)))
       (eq (literal-relation lit1) (literal-relation lit2))
       (funcall test (literal-terms lit1) (literal-terms lit2)) ))

;;;----------------------------------------------------------------------------

(defun literal-mgu (lit1 lit2 &key (ignore-sign nil))
  (when (and (or ignore-sign
		 (eq (literal-negated-p lit1) (literal-negated-p lit2)) )
	     (eq (literal-relation lit1) (literal-relation lit2)) )
    (my-unifyp (literal-terms lit1) (literal-terms lit2)) ))

;;;----------------------------------------------------------------------------

(defun literal-instance (lit1 lit2 &optional (old-binding-list nil))
  "True iff LIT2 is an instance of LIT1"
  (and (eq (literal-negated-p lit1) (literal-negated-p lit2))
       (eq (literal-relation lit1) (literal-relation lit2))
       (my-instp (literal-terms lit1) (literal-terms lit2) old-binding-list) ))

;;;----------------------------------------------------------------------------

(defun literal-equal-p (lit1 lit2)
  (and (eq (literal-negated-p lit1) (literal-negated-p lit2))
       (eq (literal-relation lit1) (literal-relation lit2))
       (equal (literal-terms lit1) (literal-terms lit2)) ))

;;;----------------------------------------------------------------------------

(defun query-to-answer-literal (query)
  (make-literal-node :relation 'answer_ :terms (find-vars query)) )

;;;----------------------------------------------------------------------------

(defun nliteral-rename-all-variables (literal)
  (let ((bl (literal-rename-binding-list literal)))
    (nliteral-plug literal bl) ))

(defun literal-rename-binding-list (literal)
  (mapcar
   #'(lambda (x) (cons x (make-new-variable x)))
   (remove-duplicates
    (find-vars (literal-terms literal)) )))

;;;----------------------------------------------------------------------------
