;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Internals.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)

(in-package "DTP")

;;;----------------------------------------------------------------------------

(defmacro add-to-end (item list)
  `(if ,list
       (rplacd (last ,list) (list ,item))
     (setf ,list (list ,item)) ))

(defmacro add-new-to-end (item list)
  `(unless (find ,item ,list)
     (add-to-end ,item ,list) ))

;;;----------------------------------------------------------------------------

(defmacro do-iteration (bound map)
  "E.g. (do-iteration proof-node-search-bound *node-iteration*)"
  `(when ,map
     (if (,bound *proof*)
	 (setf (,bound *proof*) (+ (,bound *proof*) (iterate-increment ,map)))
       (setf (,bound *proof*) (iterate-start ,map)) )
     (when (and (iterate-max-bound ,map)
		(> (,bound *proof*) (iterate-max-bound ,map)) )
       (loop-finish) )))

;;;----------------------------------------------------------------------------

(defun find-vars (list)
  (cond
   ((consp list)
    (append (find-vars (car list))
	    (find-vars (cdr list)) ))
   ((varp list)
    (list list) )))

(defun binding-list-vars (binding-list)
  (remove-duplicates
   (loop
       for binding in binding-list
       collect (car binding)
       collect (cdr binding) )))

;;;----------------------------------------------------------------------------

(defun make-new-variable (var)
  (setq var (subseq (symbol-name var) 0 (position #\_ (symbol-name var))))
  (intern (format nil "~A_~D" var (incf (proof-gensym-count *proof*)))
	  *dtp-package* ))

;;;----------------------------------------------------------------------------

(defun merge-binding-lists (binding-list &rest more-binding-lists)
  (if more-binding-lists
      (let (bl2 merge)
	(setq bl2 (apply #'merge-binding-lists more-binding-lists))
	(when bl2
	  (setq binding-list (remove '(t . t) binding-list :test #'equal))
	  (setq bl2 (remove '(t . t) bl2 :test #'equal))
	  (if (or binding-list bl2)
	      (progn
		(setq merge (append-binding-lists binding-list bl2))
		(when merge (append merge '((t . t)))) )
	    '((t . t)) )))
    binding-list ))

;;;----------------------------------------------------------------------------

(defun unify-collection (sexp &rest more-sexps)
  (if more-sexps
      (let ((other-sexp (apply #'unify-collection more-sexps))
	    bl )
	(setq bl (my-unifyp sexp other-sexp))
	(when bl (plug sexp bl)) )
    sexp ))

;;;----------------------------------------------------------------------------

(defun make-new-id (str &optional (num nil))
  (if num
      (intern (format nil "~A-~D" str num) *dtp-package*)
    (intern
     (format nil "~A-~D"
	     str (incf (proof-node-id-count *proof*)) )
     *dtp-package* )))

;;;----------------------------------------------------------------------------

(defun list-rename-variables (list)
  (let ((vars (find-vars list))
	bl )
    (setq bl (mapcar #'(lambda (x) (cons x (make-new-variable x))) vars))
    (plug list bl) ))

;;;----------------------------------------------------------------------------

(defun permutations (list-of-items)
  (if (= (length list-of-items) 1)
      (list list-of-items)
    (loop
	for item in list-of-items
	for remaining = (remove item list-of-items :test #'equal)
	appending
	  (loop
	      for perm in (permutations remaining)
	      collect (cons item perm) ))
    ))

;;;----------------------------------------------------------------------------

(defun tree-find (item tree)
  (cond
   ((or (null tree) (atom tree))
    nil )
   ((find item tree)
    t )
   (t
    (some #'(lambda (x) (tree-find item x)) tree) )))

;;;----------------------------------------------------------------------------
