;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Inference.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)

(in-package "DTP")

;;;----------------------------------------------------------------------------

(defun prove-next-answer ()
  "Find next answer using the state in *proof*, return (1) answer (2) proof"
  (loop
      with success = nil
      with children
      with slave-res
      for goal = (first (proof-agenda *proof*))
      until (or (null (proof-agenda *proof*)) success)
      finally
	(unless success
	  (when (trace-answers *tracemap*)
	    (format t "~&==> Proof failed.~%") ))
	(return (values success *proof*))
      do
	(unless (eq (anode-origin goal) 'negated-goal)
	  (setf (node-id goal) (make-new-id "SG"))
	  (possibly-prune goal) )
	(show-trace goal)
	(unless (or (anode-pruned-p goal)
		    (eq (anode-origin goal) 'negated-goal) )
	  (when (only-designs goal)
	    (push goal (proof-design-goals *proof*)) )
	  (push goal (proof-subgoals *proof*)) )
	(multiple-value-setq (children slave-res)
	  (inference-children goal) )
	(when (proof-advise *proof*)
	  (setq children (advise-choose children)) )
	(when (and (refutation-clause-p (node-clause goal))
		   (not (anode-pruned-p goal)) )
	  (setq success
	    (make-answer
	     :logic
	     (construct-logic-answer
	      (proof-query *proof*)
	      (clause-answer-literals (node-clause goal)) )	       
	     :label (clause-label (node-clause goal))
	     :assumptions (reverse (assumptions-of-node goal)) ))
	  (if (find success (proof-answers *proof*) :test #'equal-answer-p)
	      (progn
		(when (trace-answers *tracemap*)
		  (format t "==> Duplicate answer [~A]: ~A~2%"
			  (node-id goal) success ))
		(setq success nil) )
	    (progn
	      (incf (proof-answer-count *proof*))
	      (when (trace-answers *tracemap*)
		(format t "==> Proof succeeded!  [~A]: ~A~%"
			(node-id goal) success ))
	      (push success (proof-answers *proof*)) )))
	(setf (proof-agenda *proof*)
	  (agenda-merge children (cdr (proof-agenda *proof*)) slave-res) )
	))

;;;----------------------------------------------------------------------------

(defun construct-logic-answer (original-query answer-literals)
  (loop
      with original-ans-terms =
	(literal-terms (query-to-answer-literal original-query))
      with answers = nil
      for ans-lit in answer-literals
      for unifier = (my-unifyp original-ans-terms (literal-terms ans-lit))
      do (push (plug original-query unifier) answers)
      finally
	(if (cdr answers)
	    (let ((collapse (apply #' unify-collection answers)))
	      (if collapse
		  (return collapse)
		(return (cons 'or answers) )))
	  (return (first answers)) )))

;;;----------------------------------------------------------------------------

(defun agenda-merge (children old-agenda slave-children)
  (append children old-agenda slave-children) )

(defun inference-children (parent)
  "Returns: (1) useful resolutions of parent, (2) slave resolutions"
  (when (and (agenda-node-p parent)
	     (not (anode-pruned-p parent)) )
    (let (children slave-resolutions)
      (setq slave-resolutions
	(unless (proof-assumables *proof*) ; Caching fails with residue now
	  (cache-any-new-answers parent) ))
      (setq children (prune-children parent (resolve parent)))
      (loop
	  for child in children
	  for breadth from 1
	  do (setf (anode-depth child) (1+ (anode-depth parent)))
	     (setf (anode-breadth child) breadth)
	     (when (proof-assumables *proof*)
	       (setf (anode-assumptions child)
		 (remove-duplicates
		  (plug (anode-assumptions child) (anode-binding-list child))
		  :test #'equal :from-end t )))
	  finally (return (values children slave-resolutions)) ))))

;;;----------------------------------------------------------------------------

(defun prune-children (parent children)
  "Shrink the search space if possible"
  (loop
      for child in children
      when (subsumes child parent)
      do (setf (anode-pruned-p parent) 'subsum)
	 (when (trace-subsumptions *tracemap*)
	   (format t "[~A subsumed by child]~%" (node-id parent)) )
	 (return (list child)) )
  children )

(defun subsumes (node1 node2)
  "If assmpt subset, and some subset of clause2 is an instance of clause1"
  (unless (and (clause-literals (node-clause node1))
	       (clause-literals (node-clause node2)) )
    (return-from subsumes nil) )
  (when (subset-assumptions node1 node2)
    (loop
	named subsumes-loop
	with clause1 = (node-clause node1)
	with clause2 = (node-clause node2)
	with bl = '((t . t))
	for lit1 in (clause-literals clause1)
	finally
	  (return-from subsumes-loop (remove-duplicates bl :test #'equal))
	unless (find (literal-plug lit1 bl) (clause-literals clause2)
		     :test #'literal-equal-p )
	do (loop
	       named c2-lits
	       for lit2 in (clause-literals clause2)
	       for new-bl = (literal-instance lit1 lit2 bl)
	       when new-bl
	       do (setq bl new-bl)
		  (return-from c2-lits)
	       finally (return-from subsumes-loop nil) )
	   )))

;; (defun conjunct-subsumes (c1 c2 bl)
;;   (if c1
;;       (let ((lit1 (car c1)))
;; 	(some #'(lambda (lit2)
;; 		  (let ((new-bl (my-instp lit1 lit2 bl)))
;; 		    (when new-bl (conjunct-subsumes (cdr c1) c2 new-bl)) ))
;; 	      c2 ))
;;     bl ))

;;; Ginsberg's MVL subsumption check

;; (defun conj-subsumes (c1 c2) (cj-1 c1 c2 (list nil)))

;; (defun cj-1 (c1 c2 bdgs &aux b)
;;   (if c1
;;       (some #'(lambda (y) (when (setq b (inst-1 (car c1) y bdgs nil))
;; 			    (cj-1 (cdr c1) c2 b)))
;; 	    c2)
;;     t))

;;;----------------------------------------------------------------------------

(defun possibly-prune (node)
  (cond
   ((cutoff-p node)
    (setf (anode-pruned-p node) 'cutoff) )
   ((repeat-p node)
    (setf (anode-pruned-p node) 'repeat) )
   ((not (consistent-assumptions node))
    (setf (anode-pruned-p node) 'nogood) )
   ((tautological node)
    (setf (anode-pruned-p node) 'istrue) )
   ((subsumed node)
    (setf (anode-pruned-p node) 'subsum) )
   ))

(defun cutoff-p (node)
  (or (and (proof-node-search-bound *proof*)
	   (> (+ (* (anode-depth node)
		    (iterate-depth-factor *node-iteration*) )
		 (* (anode-breadth node)
		    (iterate-breadth-factor *node-iteration*) ))
	      (proof-node-search-bound *proof*) ))
      (and (proof-assumption-search-bound *proof*)
	   (> (funcall (proof-fn-object-count *proof*)
		       (anode-assumptions node) )
	      (proof-assumption-search-bound *proof*) ))
      ))

(defun repeat-p (node)
  (let ((goals (proof-subgoals *proof*))
	(clause (node-clause node))
	(bl (anode-binding-list node)) )
    (some #'(lambda (goal)
	      (and (samep clause (node-clause goal))
		   (same-assumptions node goal)
		   (consistent-binding-lists bl (anode-binding-list goal)) ))
	  goals )
    ))

(defun subsumed (node)
  (dolist (goal (proof-design-goals *proof*))
    (when (subsumes goal node)
      (when (trace-subsumptions *tracemap*)
	(format t "[~A subsumed by ~A]~%" (node-id node) (node-id goal)) )
      (return t) )))

;;;----------------------------------------------------------------------------

(defun consistent-binding-lists (bl1 bl2)
  (my-unifyp
   (append (mapcar #'car bl1) (mapcar #'car bl2))
   (append (mapcar #'cdr bl1) (mapcar #'cdr bl2)) ))

;;;----------------------------------------------------------------------------

(defun tautological (node)
  (loop
      with clause-lits = (clause-literals (node-clause node))
      for lit1 in clause-lits
      for all-after-lit1 on (cdr clause-lits)
      finally (return nil)
      do (dolist (lit2 all-after-lit1)
	   (when (literal-negated-pair-p lit1 lit2 :test #'samep)
	     (return t) ))
	 ))

;;;----------------------------------------------------------------------------

(defun resolve (node)
  "Computes resolutions of node"
  (when (refutation-clause-p (node-clause node))
    (return-from resolve nil) )
  
  (let (literal answers resolutions factors cid)

    (setq literal (clause-goal (node-clause node)))
    (setq answers (find-cache-answers literal node))
    (unless answers (cache-new-literal literal node))

    (cond
     ((setq resolutions (procedural-attachment node))
	 ;; Return just the attachments
	 )
     ((setq resolutions (assumption-resolution node))
	 ;; Return just the assumptions
	 )
     (t
      (setq factors (factor node))
      (cond
       ((and answers (eq answers :none))
	(when (trace-cache-postponing *tracemap*)
	  (setq cid
	    (node-id (first (last (cla-nodes (cache-lookup literal))))) )
	  (format t "Cache: Postponing until ~A finds answers~%" cid) )
	(setq resolutions factors) )
       (answers
	(setq resolutions
	  (append factors (resolve-with-answers node answers)) ))
       (t
	(setq resolutions (append factors (resolve-with-database node))) ))
      ))
    
    (when (and (trace-failures *tracemap*)
	       (null resolutions)
	       (not (eq answers :none)) )
      (format t "       [Branch ended in failure]~%") )
    resolutions ))

;;;----------------------------------------------------------------------------

(defun resolve-with-answers (node answers)
  (loop
      with clause = (node-clause node)
      with bl = (anode-binding-list node)
      with terms = (literal-terms (clause-goal clause))
      with new-clause =
	(make-clause-node
	 :literals (clause-remaining-literals clause)
	 :answer-literals (clause-answer-literals clause) )
      for answer in answers
      for mgu = (my-unifyp terms answer)
      collect
	(make-agenda-node
	 :clause (clause-plug new-clause mgu)
	 :binding-list (merge-binding-lists bl mgu)
	 :assumptions (anode-assumptions node)
	 :parents (list (node-id node))
	 :origin 'cache-lookup )
	))

;;;----------------------------------------------------------------------------

(defun resolve-with-database (node)
  "Actually resolve literals in NODE with database & previous subgoals"
  (let* ((c1 (node-clause node))
	 (source-literal (clause-goal c1))
	 (goal-index (proof-subgoals *proof*))
	 (kb-index
	  (append
	   (remove-pure-literal-nodes
	    (active-theory-contents
	     :index-on (literal-relation source-literal) ))
	   (proof-negated-goals *proof*) ))
	 resolutions rest subsums others )

    (setq resolutions
      (append
       (loop
	   for other-node in goal-index
	   for other-clause = (node-clause other-node)
	   appending
	     (loop
		 for other-literal in (clause-literals other-clause)
		 with br
		 when (and (literal-negated-pair-p
			    source-literal other-literal )
			   (setq br
			     (binary-resolve
			      source-literal c1 node
			      other-literal other-clause other-node t )))
		 collect br ))
       (loop
	   for other-node in kb-index
	   for other-clause =
	     (clause-rename-all-variables
	      (node-clause other-node)
	      :except
	      (when (agenda-node-p other-node)
		(binding-list-vars (anode-binding-list other-node)) ))
	   appending
	     (loop
		 for other-literal in (clause-literals other-clause)
		 with br
		 when (and (literal-negated-pair-p
			    source-literal other-literal )
			   (setq br
			     (binary-resolve
			      source-literal c1 node
			      other-literal other-clause other-node nil )))
		 collect br ))
       ))
    
    (setq rest (make-clause-node :literals (clause-remaining-literals c1)))
    (setq subsums
      (remove-if-not
       #'(lambda (x)
	   (and (clause-equal-p rest (node-clause x))
		(same-assumptions node x) ))
       resolutions ))
    (setq others
      (remove subsums resolutions
	      :test #'(lambda (set item) (find item set)) ))

    (append subsums others) ))

;;;----------------------------------------------------------------------------

(defun binary-resolve (sl c n ol oc on is-goal-goal-p)
  (let (mgu nc new-node)
    (setq mgu
      (merge-binding-lists
       (literal-mgu ol sl :ignore-sign t)
       (binding-list-of-node n) (binding-list-of-node on) ))
    (unless mgu (return-from binary-resolve nil))
    (setq nc (clause-merge ol oc sl c mgu))
    (nclause-rename-all-variables
     nc :except (remove-duplicates (find-vars mgu)) )
    (setq new-node
      (make-agenda-node
       :clause nc
       :binding-list mgu
       :assumptions
       (union (assumptions-of-node n) (assumptions-of-node on) :test #'equal)
       :parents (list (node-id n) (node-id on))
       ))
    (if is-goal-goal-p
	(setf (anode-origin new-node) 'goal-goal)
      (setf (anode-origin new-node) 'goal-kb) )
    new-node ))

;;;----------------------------------------------------------------------------

(defun factor (node)
  (loop
      with clause = (node-clause node)
      for lit1 in (clause-literals clause)
      for all-after-lit1 on (clause-remaining-literals clause)
      appending
	(loop
	    for lit2 in all-after-lit1
	    for unifier = (my-unifyp lit1 lit2)
	    when unifier
	    collect
	      (make-agenda-node
	       :clause (clause-plug clause unifier)
	       :binding-list
	       (merge-binding-lists unifier (anode-binding-list node))
	       :assumptions (anode-assumptions node)
	       :parents (list (node-id node))
	       :origin 'factor ))
	))

;;;----------------------------------------------------------------------------

(defun procedural-attachment (node)
  "Call Lisp for an Eval relation"
  (let (literal lisp lisp-result)
    (setq literal (clause-goal (node-clause node)))
    (unless (eq (literal-relation literal) 'eval)
      (return-from procedural-attachment nil) )
    (setq lisp (first (literal-terms literal)))
    (unless (eq (first lisp) 'list)
      (return-from procedural-attachment nil) )
    (setq lisp (term-attachment-map (cdr lisp)))
    (setq lisp-result (apply (car lisp) (cdr lisp)))
    (when (eq lisp-result (literal-negated-p literal)) ; Virtual negated pair
      (list (make-agenda-node
	     :clause (clause-remove literal (node-clause node))
	     :binding-list (anode-binding-list node)
	     :assumptions (anode-assumptions node)
	     :parents (list (node-id node))
	     :origin 'attachment )))
    ))

(defun term-attachment-map (list)
  "Replace (name ?x) with lisp name"
  (mapcar #'(lambda (x)
	      (cond
	       ((and (consp x) (eq (car x) 'name))
		(second x) )
	       ((consp x)
		(term-attachment-map x) )
	       (t
		x )))
	  list ))

;;;----------------------------------------------------------------------------
;;;
;;;	Pure Literal Elimination

(defun remove-pure-literal-nodes (nodes)
  (loop
      for node in nodes
      unless (pure-literal-node-p node)
      collect node ))

(defun pure-literal-node-p (node)
  (let (pure-p)
    (setq pure-p
      (gethash (node-id node) (proof-pure-literal-nodes *proof*) :unknown) )
    (when (eq pure-p :unknown)
      (setq pure-p
	(loop
	    named check-pure
	    for literal in (clause-literals (node-clause node))
	    unless (can-find-matching-literal literal :except node)
	    do (when (trace-pure-literals *tracemap*)
		 (format t "Pure literal detected in ~A...removing~%~7T"
			 (node-id node) )
		 (print-clause-node (node-clause node) :as-rule t)
		 (format t "~%for duration of proof~%") )
	       (return-from check-pure t)
	    finally (return-from check-pure nil) ))
      (setf (gethash (node-id node) (proof-pure-literal-nodes *proof*))
	pure-p ))
    pure-p ))

(defun can-find-matching-literal (literal &key (except nil))
  (or
   ;; Matching in query or (active) database...
   (loop
       for kb-node in
	 (append
	  (proof-negated-goals *proof*)
	  (remove
	   except
	   (active-theory-contents :index-on (literal-relation literal)) ))
       do
	 (loop
	     with test-clause =
	       (clause-rename-all-variables (node-clause kb-node))
	     for kb-literal in (clause-literals test-clause)
	     when (literal-negated-pair-p literal kb-literal)
	     do (return-from can-find-matching-literal kb-node) )
       finally (return nil) )
   ;; ...or assumptions
   (find literal (proof-assumables *proof*) :test #'literal-negated-pair-p)
   ))

;;;----------------------------------------------------------------------------
