;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Informant.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)
;;;	Purpose		To help with replacing Epikit by DTP

(in-package "DTP")

(eval-when (compile load eval)
  (export
   '(knownp proval remval prologp prologx prologs
     save drop empty facts contents brf )))

;;;----------------------------------------------------------------------------

(defmacro with-theory-active (&body body)
  `(progn
     (setq theory (list theory))
     (rotatef (proof-active-theories *proof*) theory)
     (prog1
	 (progn ,@body)
       (rotatef theory (proof-active-theories *proof*)) )))

;;;----------------------------------------------------------------------------

(defun knownp (fact theory)		; matchp (DB can be more specific)
  "Actually only called with FACT of form (rel ?v1 ?v2 ...)"
  (with-theory-active
      (not (null (active-theory-contents :index-on (car fact)))) ))

(defun proval (fact theory)
  "Actually, no term inference, only equality lookup"
  (with-theory-active
      (prove (list '= fact '?x) :return-form '?x) ))

(defun remval (fact theory)
  (drop-sentence-from-theory
   (list '= fact (proval fact theory)) :theory-name theory ))

(defun prologp (fact theory)
  (with-theory-active
      (prove fact) ))

(defun prologx (expr fact theory)
  (with-theory-active
      (prove fact :return-form expr) ))

(defun prologs (expr fact theory)
  (with-theory-active
      (prove fact :all-answers t :return-form expr) ))

(defun save (fact theory)		; samep
  (save-sentence-in-theory fact :theory-name theory) )

(defun drop (fact theory)		; samep
  (drop-sentence-from-theory fact :theory-name theory) )

(defun empty (theory)
  (empty-theory theory) )

(defun facts (atom theory)
  (sentences-in theory :with-atom atom) )

(defun contents (theory)
  (sentences-in theory) )

(defun brf (fact)
  (let (cnf)
    (setq cnf
      (mapcar
       #'(lambda (dnf)
	   (cons
	    `<=
	    (cons
	     (first dnf)
	     (mapcar
	      #'(lambda (lit)
		  (if (eq 'not (first lit))
		      (second lit)
		    (list 'not lit) ))
	      (rest dnf) ))))
       (cnf fact) ))
    (if (cdr cnf)
	(cons 'and cnf)
      (first cnf) )
    ))

;;;----------------------------------------------------------------------------
