;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Definitions.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)

#+lucid (in-package "USER")

(defpackage "DTP"
  #-lucid (:use "COMMON-LISP")
  (:import-from
   "USER"				; Defined in dtp.lisp
   "*DTP-DIRECTORY*" "*DTP-LOGIC-DIRECTORY*" "*DTP-FILES*" "LOAD-DTP-SYSTEM" ))

(in-package "DTP")

;;;----------------------------------------------------------------------------
;;;
;;;	Structures

(defstruct (proof
	    (:print-function proof-print-function) )
  "State of a proof effort"

  ;; Semantics
  query
  (active-theories '(global))
  (max-answers 1)
  
  ;; State
  answers
  agenda
  negated-goals
  subgoals
  (answer-count 0)
  (gensym-count 0)
  (rename-gensym-count 0)
  (node-id-count 0)
  (gate-count 0)
  (design-goals nil)
  
  ;; Residue
  (assumables nil)
  (fn-unify-assumption nil)
  (fn-object-count nil)

  ;; Iteration / Syntactic cutoffs
  (node-search-bound nil)
  (assumption-search-bound nil)

  ;; Caching
  cached-answers			; Hash table: relation -> cache list
  pure-literal-nodes			; Hash table: Id -> t, nil, or unknown

  ;; User interaction
  (advise nil) )

(defun proof-print-function (structure stream depth)
  (declare (ignore depth))
  (format stream "<")
  (if (and (proof-query structure) (null (proof-agenda structure)))
      (format stream "Completed proof")
    (format stream "Proof") )
  (when (proof-query structure)
    (format stream " of ~A" (proof-query structure)) )
  (format stream ">") )

;;;----------------------------------------------------------------------------

(defstruct (kb-node
	    (:conc-name node-)
	    (:print-function kb-node-print-function) )
  id
  (clause nil) )

(defun kb-node-print-function (structure stream depth)
  (declare (ignore depth))
  (format stream "<~A:~A>" (node-id structure) (node-clause structure)) )

;;;----------------------------------------------------------------------------

(defstruct (agenda-node
	    (:include kb-node)
	    (:conc-name anode-) )
  (assumptions nil)
  (binding-list '((t . t)))

  (pruned-p nil)
  
  (depth 1)
  (breadth 1)
  (origin nil)
  (parents nil) )

;;;----------------------------------------------------------------------------

(defstruct (iteratemap
	    (:conc-name iterate-) )
  "From START by INCREMENT until MAX"

  (max-bound nil)			; Optional termination value
  (start 2)
  (increment 1)
  (depth-factor 1)			; Appropriate for 2D iterations
  (breadth-factor 1) )			; Appropriate for 2D iterations

;;;----------------------------------------------------------------------------

(defstruct (output-form-map
	    (:conc-name output-form-) )
  "User preferences for the style of showing output"
  
  (display-as-lists nil)
  (show-renamed-variables nil)
  (show-answer-literals nil) )

;;;----------------------------------------------------------------------------

(defstruct (tracemap
	    (:conc-name trace-) )
  "User preferences for how much tracing information to show"
  
  (answers nil)
  (goals nil)
  (clauses nil)
  (assumptions nil)
  (designs nil)
  (subsumptions nil)
  (cache-answers nil)
  (cache-postponing nil)
  (failures nil)
  (pure-literals nil)

  (load t)
  (tests t) )

;;;----------------------------------------------------------------------------
;;;
;;;	Global variables

(defvar *dtp-package* (find-package "DTP"))

(defvar *proof* (make-proof))
(defvar *node-iteration* nil)		; Or (make-iteratemap)
(defvar *assumption-iteration* nil)	; Or (make-iteratemap)
(defvar *output-form-map* (make-output-form-map))

(defvar *tracemap* (make-tracemap))

;;;----------------------------------------------------------------------------
