;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Database.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)

(in-package "DTP")

(eval-when (compile load eval)
  (export
   '(empty-theory) ))

;;;----------------------------------------------------------------------------

(defvar *kb* nil "List of THEORY structures")
(defvar *id-index* (make-hash-table :test #'eq))

;;;----------------------------------------------------------------------------

(defstruct (theory
	    (:print-function theory-print-function) )
  name
  (nodes nil)
  (indexed-nodes nil) )

(defstruct (node-index
	    (:print-function node-index-print-function) )
  key
  (nodes nil) )

;;;----------------------------------------------------------------------------

(defun theory-print-function (structure stream depth)
  (declare (ignore depth))
  (format stream "<Theory ~A with ~D node~:P>"
	  (theory-name structure) (length (theory-nodes structure)) ))

(defun node-index-print-function (structure stream depth)
  (declare (ignore depth))
  (format stream "<Index ~A with ~D node~:P>"
	  (node-index-key structure) (length (node-index-nodes structure)) ))

;;;----------------------------------------------------------------------------

(defun reset-database ()
  (setq *id-index* (make-hash-table :test #'eq))
  (setq *kb* nil) )

;;;----------------------------------------------------------------------------

(defun empty-theory (theory-name)
  (let ((theory (get-theory-structure theory-name)))
    (when theory
      (dolist (node (theory-nodes theory))
	(unindex-id (node-id node)) )
      (setq *kb* (remove theory *kb*))
      theory-name )))

;;;----------------------------------------------------------------------------

(defun all-kb-theories ()
  (mapcar #'theory-name *kb*) )

;;;----------------------------------------------------------------------------

(defun get-theory-structure (theory-name)
  (find theory-name *kb* :key #'theory-name) )

;;;----------------------------------------------------------------------------

(defun active-theory-contents (&key (index-on nil))
  (loop
      for theory-name in (included-active-theory-names)
      append (theory-contents theory-name index-on) ))

;;;----------------------------------------------------------------------------

(defun theory-contents (theory-name &optional (index nil))
  (let ((theory (get-theory-structure theory-name)))
    (cond
     ((null theory)
      nil )
     (index
      (let (ni)
	(setq ni
	  (find index (theory-indexed-nodes theory) :key #'node-index-key) )
	(when ni (node-index-nodes ni)) ))
     (t
      (theory-nodes theory) ))
    ))

;;;----------------------------------------------------------------------------

(defun make-theory-from-nodes (nodes theory-name)
  (empty-theory theory-name)
  (add-to-end
   (make-theory
    :name theory-name :nodes nodes :indexed-nodes (make-index nodes) )
   *kb* )
  (mapc #'(lambda (node) (index-id (node-id node) node))
	nodes )
  theory-name )

;;;----------------------------------------------------------------------------

(defun save-node-in-theory (node theory-name)
  (let ((old-theory (get-theory-structure theory-name)))
    (if old-theory
	(progn
	  (setf (theory-indexed-nodes old-theory)
	    (make-index (list node) (theory-indexed-nodes old-theory)) )
	  (add-to-end node (theory-nodes old-theory))
	  (index-id (node-id node) node) )
      (make-theory-from-nodes (list node) theory-name) )
    (node-id node) ))

;;;----------------------------------------------------------------------------

(defun drop-node-from-theory (node theory-name)
  (unindex-id (node-id node))
  (setf (theory-nodes theory-name) (remove node (theory-nodes theory-name)))
  (dolist (index (theory-indexed-nodes theory-name))
    (setf (node-index-nodes index)
      (remove node (node-index-nodes index)) ))
  (setf (theory-indexed-nodes theory-name)
    (remove-if-not #'node-index-nodes (theory-indexed-nodes theory-name)) )
  (node-id node) )

;;;----------------------------------------------------------------------------

(defun make-index (nodes &optional (indices nil))
  "Return a list of index nodes with a key for each relation in NODES"
  (loop
      for node in nodes
      do (loop
	     for literal in (clause-literals (node-clause node))
	     for relation = (literal-relation literal)
	     for index = (find relation indices :key #'node-index-key)
	     do (if index
		    (add-new-to-end node (node-index-nodes index))
		  (add-to-end
		   (make-node-index :key relation :nodes (list node))
		   indices )))
      finally (return indices) ))

;;;----------------------------------------------------------------------------

(defun find-kb-node-with-id (id)
  (gethash id *id-index*) )

(defun index-id (id node)
  (when (gethash id *id-index*)
    (error "Trying to remap id ~A from ~A to ~A"
	   id (gethash id *id-index*) node ))
  (setf (gethash id *id-index*) node) )

(defun unindex-id (id)
  (remhash id *id-index*) )

;;;----------------------------------------------------------------------------

(defun last-id-count (theory-name)
  (let ((theory (get-theory-structure theory-name))
	id str )
    (cond
     (theory
      (setq id (node-id (first (last (theory-nodes theory)))))
      (setq str (symbol-name id))
      (setq str (subseq str (1+ (length (symbol-name theory-name)))))
      (values (read-from-string str)) )
     (t
      0 ))))	

;;;----------------------------------------------------------------------------
