;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Circuits.Lisp
;;;	System		Don's Theorem Prover
;;;	Status		Example of use of residue
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)

(in-package "DTP")

(eval-when (load eval)
  (pushnew :dtp-circuits *features*) )

;;;----------------------------------------------------------------------------

(defvar *module-inputs* nil)
(defvar *module-outputs* nil)
(defvar *gate-sizes* nil "A-list of gate types & # of inputs")

;;; Use: To construct a full-adder, call with
;;; (build '(previous-carry input-1 input-2)
;;;        '((sum   (0 1 1 0 1 0 0 1))
;;;          (carry (0 0 0 1 0 1 1 1)) ))

(defun build (input-list output-desc &optional (basic-components 'standard))
  (let ((*proof* (proof-with-design-settings)))
    (set-circuit-theory basic-components)
    (setq *module-inputs* input-list)
    (setq *module-outputs* (mapcar #'first output-desc))
    (describe-design input-list output-desc)
    (prove '(design)) ))

;;;----------------------------------------------------------------------------

(defun proof-with-design-settings ()
  (let ((new-proof (copy-proof *proof*)))
    (setf (proof-active-theories new-proof) nil)
    (setf (proof-assumables new-proof)
      '((type ?gate ?type) (drives ?driver ?drivee)) )
    (setf (proof-fn-unify-assumption new-proof) #'ok-assumption)
    (setf (proof-fn-object-count new-proof) #'count-gates)
    new-proof ))

;;;----------------------------------------------------------------------------
;;;
;;; E.g. for a full adder, would save something like
;;; (<= (design)
;;;     (design-works 0 0 0) (design-works 0 0 1) ... (design-works 1 1 1) )
;;; (<= (design-works 0 0 0)
;;;     (<= (and (module-output sum 0) (module-output carry 0))
;;;         (module-input previous-carry 0) ... (module-input input-2 0) ))
;;; into theory AUX

(defun describe-design (input-list output-desc)
  (empty-theory 'aux)
  
  (loop
      with num-of-digits = (length input-list)
      for input-combo from 0 to (1- (expt 2 num-of-digits))
      for binary-sequence = (list-of-bits input-combo num-of-digits)
      for outputs =
	(mapcar #'(lambda (out)
		    (list
		     'module-output
		     (first out)
		     (nth input-combo (second out)) ))
		output-desc )
      for inputs =
	(loop
	    for input in binary-sequence
	    for label in input-list
	    collect (list 'module-input label input) )
      collect (cons 'design-works binary-sequence) into working-design
      do
	(save-sentence-in-theory
	 `(<= (design-works ,@binary-sequence)
	      (<= (and ,@outputs)
		  ,@inputs ))
	 'aux
	 :all-orders t )
      finally
	(save-sentence-in-theory `(<= (design) ,@working-design) 'aux) )
  
  (pushnew 'aux (proof-active-theories *proof*)) )

;;;----------------------------------------------------------------------------

(defun list-of-bits (number digits)
  (let ((list nil))
    (dotimes (digit digits)
      (push (logand number 1) list)
      (setq number (ash number -1)) )
    list ))

;;;----------------------------------------------------------------------------

(defun ok-assumption (new-assumption old-assumptions)
  "(type ?gate ?gate-type) and (drives ?output ?input)"
  
  (let ((all-assumptions (cons new-assumption old-assumptions))
	(port-types
	 (append
	  (mapcar #'(lambda (inp) `(type ,inp input)) *module-inputs*)
	  (mapcar #'(lambda (out) `(type ,out output)) *module-outputs*) ))
	typed-gates )

    (cond

     ;; Already there
     ((find new-assumption old-assumptions :test #'equal)
      t )

     ;; Gate with two types
     ((progn
	(setq typed-gates
	  (append (remove 'type all-assumptions :test-not #'eq :key #'first)
		  port-types ))
	(not (equal typed-gates
		    (remove-duplicates typed-gates :key #'second) )))
       nil )

     ;; Gate with too many inputs
     ((some
       #'(lambda (typed-gate)
	   (let ((gate (second typed-gate))
		 (type (third typed-gate)) )
	     (> (count gate all-assumptions
		       :test
		       #'(lambda (g a)
			   (let ((rel (first a))
				 (end (third a)) )
			     (and (eq rel 'drives)
				  (eq (if (listp end) (second end) end) g) )))
		       )
		(number-of-drivers type) )))
       typed-gates )
      nil )
      
     ;; Output cycle
     ((and (eq (first new-assumption) 'drives)
	   (driving-cycle
	    (second new-assumption)
	    (cddr new-assumption)
	    (remove 'drives old-assumptions :test-not #'eq :key #'first) ))
      nil )

     ;; Symmetry: Drivers must be in lexicographic order
     ((do* ((out-of-order nil)
	    (last-driver nil this-driver)
	    (last-drivee nil this-drivee)
	    (drives
	     (remove 'drives all-assumptions :key #'first :test-not #'eq)
	     (cdr drives) )
	    (drive (car drives) (car drives))
	    (this-driver (second drive) (second drive))
	    (this-drivee (third drive) (third drive)) )
	  ((null drives)
	   out-of-order )
	(when (listp this-drivee) (setq this-drivee (second this-drivee)))
	(when (and (eq last-drivee this-drivee)
		   (string> this-driver last-driver) )
	  (setq out-of-order t) ))
      nil )

     ;; Otherwise ok
     (t
      t ))
    ))

(defun driving-cycle (key-gate start-gates drivings)
  (do* ((set (mapcar #'gate-of start-gates)
	     (apply #'append
		    (mapcar
		     #'(lambda (gate)
			 (let ((new-gates nil))
			   (dolist (driving drivings)
			     (when (eq (second driving) gate)
			       (push (gate-of (third driving)) new-gates) ))
			   new-gates ))
			    set )))
	(count 0 (1+ count)) )
      ((or (null set) (find key-gate set))
       (when set count) )
    ))

(defun number-of-drivers (gate-type)
  (case gate-type
    (input 0)
    (output 1)
    (otherwise (cdr (assoc gate-type *gate-sizes*))) ))

(defun gate-of (label)
  (if (listp label)
      (second label)
    label ))

;;;----------------------------------------------------------------------------

(defun order-agenda (agenda)
  (let ((new-agenda (copy-list agenda)))
    (sort new-agenda #'< :key #'count-gates)
    (unless (equal new-agenda agenda)
      (when (or (trace-goals *tracemap*)
		(trace-clauses *tracemap*) )
	(format t "[Agenda reordered.  ")
	(format t "Minimum gates ~D, maximum ~D.  Size = ~D.]~%"
		(count-gates (first new-agenda))
		(count-gates (first (last new-agenda)))
		(length new-agenda) )))
    new-agenda ))

(defun count-gates (assumptions)
  (let ((count 0))
    (dolist (assumpt assumptions)
      (when (eq (car assumpt) 'type)
	(setq count (1+ count)) ))
    count ))

;;;----------------------------------------------------------------------------

(defun set-circuit-theory (type)
  "NAND and DW (xor, and, or)"
  (pushnew type (proof-active-theories *proof*))
  (case type
    (nand (setq *gate-sizes* '((nand . 2))))
    (dw (setq *gate-sizes* '((xor . 2) (and . 2) (or . 2)))) ))

;;;----------------------------------------------------------------------------
