;;; -*- Mode: LISP; Syntax: Common-lisp; Package: DTP; Base: 10 -*-

;;;----------------------------------------------------------------------------
;;;
;;;	File		Cache.Lisp
;;;	System		Don's Theorem Prover
;;;
;;;	Written by	Don Geddis (Geddis@CS.Stanford.Edu)
;;;
;;;	Based on	"Controlling Recursive Inference"
;;;			D. E. Smith, M. R. Genesereth and M. L. Ginsberg
;;;			AIJ Vol 30 No 3 Dec 1986
;;;
;;;	Summary		Keep a cache of answers to literals

(in-package "DTP")

;;;----------------------------------------------------------------------------
;;;
;;;	Data structures

(defstruct (cached-literal-answers
	    (:conc-name cla-)
	    (:print-function cached-literal-answers-print-function) )
  "Cached value of answers for literals encountered during theorem proving"
  literal
  answers				; Just terms
  nodes )				; Places to propogate the answer to

(defun cached-literal-answers-print-function (structure stream depth)
  (declare (ignore depth))
  (format stream "<Cache of ")
  (print-literal-node (cla-literal structure) :s stream :flip-negation t)
  (format stream " with ~D answer~:P>" (length (cla-answers structure))) )

;;;----------------------------------------------------------------------------

(defun cache-new-literal (literal node)
  (push
   (make-cached-literal-answers
    :literal literal
    :nodes (list node) )
   (gethash
    (literal-relation literal)
    (proof-cached-answers *proof*)
    nil )))

;;;----------------------------------------------------------------------------

(defun cache-lookup (literal)
  (find
   literal
   (gethash (literal-relation literal) (proof-cached-answers *proof*))
   :key #'cla-literal :test #'literal-match-p ))

;;;----------------------------------------------------------------------------

(defun find-cache-answers (literal &optional (node nil))
  "Returns nil (not in cache), :none (no answers), or list of answers"
  (let ((cache (cache-lookup literal)))
    (when cache
      (when (and node (clause-literals (node-clause node)))
	(push node (cla-nodes cache)) )
      (or (cla-answers cache)
	  :none ))
    ))

;;;----------------------------------------------------------------------------

(defun cache-any-new-answers (node)
  "If clause is an instance of a subset of cdr of any parents, new ans"
  (unless (eq (anode-origin node) 'factor)
    (loop
	with new-answer
	for parent in (agenda-goal-ancestors node)
	for literal = (clause-goal (node-clause parent))
	for bl =
	  (clause-is-instance-of-subset-p
	   (node-clause node) (node-clause parent) :cdr t )
	when (and bl
		  (setq new-answer
		    (plug (literal-terms literal) (anode-binding-list node)) ))
	appending
	  (add-new-answer new-answer literal :do-not-resolve-with parent)
	  )))

;;;----------------------------------------------------------------------------

(defun add-new-answer (answer literal &key (do-not-resolve-with nil))
  "Adds ANSWER to LITERAL's cache, returns new resolutions if new answer"
  (let (cache slaves)
    (setq cache (cache-lookup literal))
    (unless cache
      (format t "Error: Can't find ~A to cache ~A~%" literal answer) )
    (if (find answer (cla-answers cache) :test #'equal)
	nil
      (progn
	(push answer (cla-answers cache))
	(setq slaves (remove do-not-resolve-with (cla-nodes cache)))
	(when (trace-cache-answers *tracemap*)
	  (format t "Cache: Added answer ~A to " answer)
	  (print-literal-node (cla-literal cache) :flip-negation t)
	  (when slaves
	    (format t "~%       Propagating to ~A~{, ~A~}"
		    (node-id (car slaves))
		    (mapcar #'node-id (cdr slaves)) ))
	  (format t "~%") )
	(apply #'append
	       (mapcar #'(lambda (x)
			   (resolve-with-answers x (list answer)) )
		       slaves ))
	))				; if
    ))

;;;----------------------------------------------------------------------------
