/* ---------------------------------------------------------- 
%   (C)1992 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
generate(_/Arty,Clauses,Works,Susps,Info0,Info,Code) :-
    separate_cond(Clauses,Cond),
    make_index_tree(Cond,e,Tree),
    arg_reg_assoc(0,Arty,AL),
    gen_code(Tree,interrupt,0,Works,0,_,0,AL,0,Susps,Info0,Info,Code,[]).

separate_cond([],[]).
separate_cond([Clause|CT],[Clause|CondT]) :- atom(Clause),!,
    separate_cond(CT,CondT).
separate_cond([clause(ID,H,G,B)|CT],[Cond-body(ID,B,N3)|CondT]) :-
    An = _,
    numbervars((H,G,B,An),0,N),
    normalize_head(H,Cond,Cond1,An,N,N2),
    normalize_guard(G,Cond1,[],An,N2,N3),
    separate_cond(CT,CondT).

/*
  Normalize Head Unification and Guard Builtin

  a list of:
       Position-Pattern
       guard(Modes,Name/Arity,Args)
       guard_unify(PosPats,PosPats)

  "Position" is one of:
    [Pos]		% argument position
    [car|Pos]		% car of "Pos"
    [Index|Pos]		% argument "Index" of "Pos"

  "Pattern" is one of:
    '$VAR'(K)		% variable numbered "K"
    a(Atom)		% atom "Atom"
    list		% list
    funct(Funct,Arity)	% functor structure Funct/Arity
    an                  % anonymous variable

   "PosPats" is a list of:
     Position-Pattern

   "Modes" is a list of:
     int(Mode)          % integer
     atom(Mode)         % atom
     list(Mode)         % list
     vector(Mode)       % vector
     string(Mode)       % string
     functor(Mode)      % functor
     undef              % undefined

   "Mode" is one of (in, out)

   "Name" is a builtin function name
   "Arity is an arity of a guard builtin
   "Args" is an argument list s.t. [PosPats, ...]

*/

normalize_head(H,N0,N,An,VN0,VN) :-
    H =.. [_|Args],
    normalize_args(Args,[],0,N0,N,An,VN0,VN).

normalize_args([],_,_,N,N,_,VN,VN):- !.

normalize_args([H|T],Pos,K,N0,N,An,VN0,VN) :-
    normalize_arg(H,[K|Pos],N0,N1,An,VN0,VN1),
    K1 is K+1,
    normalize_args(T,Pos,K1,N1,N,An,VN1,VN).

normalize_arg(An,Pos,[Pos-an|N],N,An,VN,VN) :- !.
normalize_arg('$VAR'(K),Pos,[Pos-'$VAR'(K)|N],N,_,VN,VN) :- !.
normalize_arg({},Pos,[Pos-funct('VECT',1),
		     [0|Pos]-a(0)|N],N,_,VN,VN):- !.
normalize_arg(A,Pos,[Pos-a(A)|N],N,_,VN,VN) :- atomic(A), !.
normalize_arg([H|T],Pos,[Pos-list|N0],N,An,VN0,VN) :- !,
    normalize_arg(H,[car|Pos],N0,N1,An,VN0,VN1),
    normalize_arg(T,[cdr|Pos],N1,N,An,VN1,VN).
normalize_arg(S,Pos,[Pos-funct('VECT',A1),
                     [0|Pos]-a(A)|N0],N,An,VN0,VN) :-
    functor(S,{},1),!,S =.. [_|Args],
    normalize_vector(Args,A,Pos,N0,N,An,VN0,VN),
    A1 is A + 1.
/*    normalize_arg(A,Pos,N0,N1,An,VN1,VN). */
normalize_arg(S,Pos,[Pos-NewVar|N0],N,An,VN0,VN) :-
    functor(S,~,1),S =.. [_|Exp],!,
    numbervars(NewVar,VN0,VN1),
    normalize_guard(NewVar:=Exp,N0,N,An,VN1,VN).
normalize_arg(S,Pos,N0,N,An,VN0,VN) :-
    functor(S,#,_),
    macro_expand(S,NewS),!,
    normalize_arg(NewS,Pos,N0,N,An,VN0,VN).
normalize_arg(S,Pos,[Pos-funct(F,A)|N0],N,An,VN0,VN) :-
    functor(S,(`),1), S =.. [_,NewS],
    functor(NewS,F,A),S=.. [_|Args],!,
    normalize_args(Args,Pos,0,N0,N,An,VN0,VN).
normalize_arg(S,Pos,[Pos-funct(F,A)|N0],N,An,VN0,VN) :- !,
    functor(S,F,A),S =.. [_|Args],
    normalize_args(Args,Pos,0,N0,N,An,VN0,VN).

normalize_vector([],0,_,N,N,_,VN,VN):- !.
normalize_vector([A0],M,Pos,N0,N,An,VN0,VN):-
    functor(A0,(','),2),!, A0 =.. [_,A1,A2],
    normalize_vector2(A1,A2,0,M,Pos,N0,N,An,VN0,VN).
normalize_vector([A0],1,Pos,N,N0,An,VN0,VN):-
    normalize_arg(A0,[1|Pos],N,N0,An,VN0,VN).

normalize_vector2(A0,Arg,M0,M,Pos,N0,N,An,VN0,VN):-
    functor(Arg,(','),2),!, Arg =.. [_,A1,A2],
    M1 is M0 + 1,
    normalize_arg(A0,[M1|Pos],N0,N1,An,VN0,VN1),
    normalize_vector2(A1,A2,M1,M,Pos,N1,N,An,VN1,VN).
normalize_vector2(A0,A1,M0,M,Pos,N0,N,An,VN0,VN):-
    M1 is M0 + 1,
    M is M0 + 2,
    normalize_arg(A0,[M1|Pos],N0,N1,An,VN0,VN1),
    normalize_arg(A1,[M|Pos],N1,N,An,VN1,VN).

normalize_guard(true,N,N,_,VN,VN) :- !.
normalize_guard((X,Y),N0,N,An,VN0,VN) :- !,
    normalize_guard(X,N0,N1,An,VN0,VN1),
    normalize_guard(Y,N1,N,An,VN1,VN).
normalize_guard((X = Y),N0,N,An,VN0,VN):- !,
    normalize_guard_unify(X,Y,N0,N,An,VN0,VN).
normalize_guard(X,N0,N,An,VN0,VN) :-
    functor(X,Op,2),
    is_guard_assign(Op,2,_,_),!,
    X =.. [_,LHS,RHS],
    normalize_guard_arg(LHS,LHS1,An,N0,N1,VN0,VN1),
    normalize_guard_exp(LHS1,RHS,An,N1,N,VN1,VN).
normalize_guard(X,N0,N,An,VN0,VN) :-
    functor(X,Op,2),
    is_guard_comp(Op,2,Func,Modes),!,
    X =.. [_,LHS,RHS],
    normalize_guard_exp2(LHS,Y,An,N0,N1,VN0,VN1),
    normalize_guard_exp2(RHS,Z,An,N1,[guard(Modes,Func/2,[Y,Z])|N],VN1,VN).
normalize_guard(X,N0,N,An,VN0,VN):-
    functor(X,F,A), 
    is_guard_builtin(F,A,Func,Modes),!,
    X =.. [_|Args0],
    normalize_guard_args(Args0,Args,An,N0,[guard(Modes,Func/A,Args)|N],VN0,VN).

/* temporary
normalize_guard((X := Y),[guard([int-out|Modes],Func/A,[X|Args])|N],N,
                                                            An,VN0,VN):-
    functor(Y,F,A0), F \== ('.'), Y =.. [_|Args0], !,
    A is A0 + 1,
    is_guard_builtin(F,A0,Func,Modes),!,
    normalize_guard_exp2(Args0,Args,An,N0,N,VN0,VN).
normalize_guard(X,[guard(Modes,Func/A,Args)|N],N,An,VN0,VN):-
    functor(X,F,A), F \== ('.'),X =.. [_|Args0],!,
    is_guard_builtin(F,A,Func,Modes),!,
    normalize_guard_args(Args0,Args,An,VN0,VN).
*/
normalize_guard(X,N,N,_,VN,VN) :-
    format(user_error,"Guard not implemented yet: ~w~n",[X]).

normalize_guard_unify('$VAR'(K),Y,
                     [guard_unify(Pos,'$VAR'(K),N1)|N],N,An,VN0,VN):- !,
    normalize_arg(Y,Pos,N1,[],An,VN0,VN).
normalize_guard_unify(X,'$VAR'(K),
                     [guard_unify(Pos,'$VAR'(K),N1)|N],N,An,VN0,VN):- !,
    normalize_arg(X,Pos,N1,[],An,VN0,VN).

/* normalize_guard_unify(X,Y,N,N,_,VN,VN):- !. */
normalize_guard_unify(Atom,Atom,N,N,_,VN,VN):- atom(Atom),!.
normalize_guard_unify([CAR|CDR],[CAR2|CDR2],N0,N,An,VN0,VN):- !,
    normalize_guard_unify(CAR,CAR2,N0,N1,An,VN0,VN1),
    normalize_guard_unify(CDR,CDR2,N1,N ,An,VN1,VN).
normalize_guard_unify(S,S2,N0,N,An,VN0,VN):-
    functor(S,F,A),S =.. [_|Args],
    functor(S2,F,A),S =.. [_|Args2],!,
    normalize_guard_unifies(Args,Args2,N0,N,An,VN0,VN).

normalize_guard_unifies([],[],N,N,_,VN,VN):- !.
normalize_guard_unifies([One|Rest1],[Two|Rest2],N0,N,An,VN0,VN):-
    normalize_guard_unify(One,Two,N0,N1,An,VN0,VN1),
    normalize_guard_unifies(Rest1,Rest2,N1,N,An,VN1,VN).

normalize_guard_arg(An,an,An,N,N,VN,VN):- !.
normalize_guard_arg('$VAR'(K),'$VAR'(K),_,N,N,VN,VN):- !.
normalize_guard_arg(A,a(A),_,N,N,VN,VN) :- atomic(A),!.
normalize_guard_arg([A|B],list(AA,BB),An,N0,N,VN0,VN) :- !,
    normalize_guard_arg(A,AA,An,N0,N1,VN0,VN1),
    normalize_guard_arg(B,BB,An,N1,N,VN1,VN).
normalize_guard_arg(X,Y,An,N0,N,VN0,VN):-
    functor(X,(~),1),!, X =.. [_,Exp],
    numbervars(Y,VN0,VN1),
    normalize_guard_exp(Y,Exp,An,N0,N,VN1,VN).
normalize_guard_arg(X,funct(F,A,Args),An,N0,N,VN0,VN):- !,
    functor(X,F,A), X =.. [_|Args0],
    normalize_guard_args(Args0,Args,An,N0,N,VN0,VN).

normalize_guard_args([],[],_,N,N,VN,VN):- !.
normalize_guard_args([X|Args0],[Y|Args],An,N0,N,VN0,VN):-
    normalize_guard_arg(X,Y,An,N0,N1,VN0,VN),
    normalize_guard_args(Args0,Args,An,N1,N,VN0,VN).

normalize_guard_exp(X,Exp,An,N0,N,VN0,VN):-
    functor(Exp,Op,1),
    is_guard_builtin(Op,2,Funct,Modes),!,
    Exp =.. [_,ExpA],
    NewExp = guard(Modes,Funct/2,[NewExpA,X]),
    normalize_guard_exp2(ExpA,NewExpA,An,N0,[NewExp|N],VN0,VN).
normalize_guard_exp(X,Exp,An,N0,N,VN0,VN):-
    functor(Exp,Op,2),
    is_guard_builtin(Op,3,Funct,Modes),!,
    Exp =.. [_,ExpA,ExpB],
    NewExp = guard(Modes,Funct/3,[NewExpA,NewExpB,X]),
    normalize_guard_exp2(ExpA,NewExpA,An,N0,N1,VN0,VN1),
    normalize_guard_exp2(ExpB,NewExpB,An,N1,[NewExp|N],VN1,VN).
normalize_guard_exp(X,Exp,An,N0,N,VN0,VN):-
    normalize_guard_arg(Exp,Exp1,An,N0,
                  [guard([int-out,int-in],gblt_assign/2,[X,Exp1])|N],VN0,VN).
normalize_guard_exp2(Exp,NewExp,An,N0,N,VN0,VN) :-
    functor(Exp,Op,1),
    is_guard_builtin(Op,2,Funct,Modes),!,
    Exp =.. [_,ExpA],
    NewExp2 = guard(Modes,Funct/2,[NewExpA,NewExp]),
    normalize_guard_exp2(ExpA,NewExpA,An,N0,[NewExp2|N],VN0,VN1),
    numbervars((NewExpA,NewExp),VN1,VN).
normalize_guard_exp2(Exp,NewExp,An,N0,N,VN0,VN) :-
    functor(Exp,Op,2),
    is_guard_builtin(Op,3,Funct,Modes),!,
    Exp =.. [_,ExpA,ExpB],
    NewExp2 = guard(Modes,Funct/3,[NewExpA,NewExpB,NewExp]),
    normalize_guard_exp2(ExpA,NewExpA,An,N0,N1,VN0,VN1),
    normalize_guard_exp2(ExpB,NewExpB,An,N1,[NewExp2|N],VN1,VN2),
    numbervars((NewExpA,NewExpB,NewExp),VN2,VN).
normalize_guard_exp2(Exp,Exp1,An,N0,N,VN0,VN):-
    normalize_guard_arg(Exp,Exp1,An,N0,N,VN0,VN).


/*
  Making Clause Indexing Tree Structure

    n(Pos,			% Non-terminal: Argument position
      ([Atom-Node,...],		%   Atomic cases
       [GBNode,...])
      (ListNode,		%   List cases
       [GBNode,...])
      ([F/A-Node,...],		%   Functor cases
       [GBNode,...])
      [Epos-Node,...],		%   Equational cases
      UncondNode)		%   Unconditional cases

    g(F/A,Args,Deref,Set,Unify,Node,UncondNode)
                                % Guard Builtin node (non-terminal)
    l([Vars=Body,...])		% Leaf node (completely unconditional)
    e				% Empty node

  Argment position is a list of integers such as:
	[N1, N2, ... Nn]
  which means "N1'th element of N2'th element of .. of Nn'th argument".
*/

make_index_tree(CT,T0,T):-
    make_index_tree(CT,T0,T,-1,0,0).

make_index_tree([],T,T,_,_,_):- !.
make_index_tree([Cond-Body|CT],T0,T,GReg0,Ot,Al) :- !,
    ins_index(Cond,T0,[],Body,T1,GReg0,GReg1),
    make_index_tree(CT,T1,T,GReg1,Ot,Al).
make_index_tree([otherwise|CT],T0,other(Ot,T0,T),_,Ot,Al) :- !,
    Ot1 is Ot + 1,
    make_index_tree(CT,e,T,-1,Ot1,Al).
make_index_tree([alternatively|CT],T0,alter(Al,T0,T),_,Ot,Al) :- !,
    Al1 is Al + 1,
    make_index_tree(CT,e,T,-1,Ot,Al1).

ins_index([],T0,Vars,Body,T,G0,G1) :- ins(T0,Vars,Body,T,G0,G1).
ins_index([_-an|CondT],T0,Vars,Body,T,G0,G1) :- !,
    ins_index(CondT,T0,Vars,Body,T,G0,G1).

/*
ins_index([Pos-'$VAR'(K)|CondT],T0,Vars,Body,
	  n(Pos1,([],[]),(e,[]),([],[]),[Pos-T],e),G0,G1) :-
    assoc_with_type(Vars,'$VAR'(K),Pos1,OldType,NewType,NewVars), !,
    update_type(OldType,any-0,NewType,_),
    ins_index(CondT,T0,[('$VAR'(K)=Pos,NewType)|NewVars],Body,T,G0,G1).
*/
ins_index([Pos-'$VAR'(K)|CondT],T0,Vars,Body,T,G0,G1):-
    assoc_with_type(Vars,'$VAR'(K),Pos1,OldType,NewType,NewVars), !,
    update_type(OldType,any-0,NewType,_),
    ins_index_eq(T0,Pos1,Pos,CondT,
                 [('$VAR'(K)=Pos,NewType)|NewVars],Body,T,G0,G1).
ins_index([Pos-'$VAR'(K)|CondT],T0,Vars,Body,T,G0,G1) :- !,
    ins_index(CondT,T0,[('$VAR'(K)=Pos,undef)|Vars],Body,T,G0,G1).
ins_index([Pos-Cond|CondT],T0,Vars,Body,T,G0,G1) :-
    ins_node(T0,Pos,Cond,CondT,Vars,Body,T,G0,G1).
ins_index([guard_unify(Pos,X,Y)|CondT],T0,Vars,Body,T,G0,G1) :-
    ins_guard_unify(X,Y,Pos,CondT,T0,Vars,Body,T,G0,G1).
ins_index([guard(Modes,Blt,Args)|CondT],T0,Vars,Body,T,G0,G1) :-
    ins_guard(Args,Modes,Blt,CondT,T0,Vars,Body,T,G0,G1).

ins_index_eq(n(Pos1,A,L,F,E,C),Pos1,Pos,CondT,Vars,Body,
             n(Pos1,A,L,F,NewE,C),G0,G1):- !,
    ins_index_eq2(E,NewE,Pos,CondT,Vars,Body,G0,G1).
ins_index_eq(n(Pos,A,L,F,E,C),Pos1,Pos,CondT,Vars,Body,
             n(Pos,A,L,F,NewE,C),G0,G1):- !,
    ins_index_eq2(E,NewE,Pos1,CondT,Vars,Body,G0,G1).
ins_index_eq(n(Pos,A,L,F,E,C),Pos1,Pos2,CondT,Vars,Body,
             n(Pos,A,L,F,E,NewC),G0,G1):- !,
    ins_index_eq(C,Pos1,Pos2,CondT,Vars,Body,NewC,G0,G1).
ins_index_eq(T0,Pos1,Pos,CondT,Vars,Body,
             n(Pos1,([],[]),(e,[]),([],[]),[Pos-T],e),G0,G1):-
    ins_index(CondT,T0,Vars,Body,T,G0,G1).

ins_index_eq2([],[Pos-T],Pos,CondT,Vars,Body,G0,G1):- !,
    ins_index(CondT,e,Vars,Body,T,G0,G1).
ins_index_eq2([Pos-T0|E],[Pos-T|E],Pos,CondT,Vars,Body,G0,G1):- !,
    ins_index(CondT,T0,Vars,Body,T,G0,G1).
ins_index_eq2([Pos-T0|E],[Pos-T0|E1],Pos1,CondT,Vars,Body,G0,G1):-
    ins_index_eq2(E,E1,Pos1,CondT,Vars,Body,G0,G1).

ins_node(e,P,Cond,CondT,V,B,n(P,(A,[]),(L,[]),(F,[]),[],e),G0,G1) :- !,
    ins_type(Cond,[],A,e,L,[],F,CondT,V,B,G0,G1).
ins_node(l(BL),P,Cond,CondT,V,B,n(P,(A,[]),(L,[]),(F,[]),[],l(BL)),G0,G1) :- !,
    ins_type(Cond,[],A,e,L,[],F,CondT,V,B,G0,G1).
ins_node(n(P,(A0,AB),(L0,LB),(F0,FB),E,U),P,Cond,CondT,V,B,n(P,(A,AB),(L,LB),(F,FB),E,U),G0,G1) :- !,
    ins_type(Cond,A0,A,L0,L,F0,F,CondT,V,B,G0,G1).
ins_node(n(P0,A,L,F,E,U),P,Cond,CondT,V,B,
                n(P,(AA,[]),(LL,[]),(FF,[]),[],n(P0,A,L,F,E,U)),G0,G1) :-
    P @< P0, !,
    ins_type(Cond,[],AA,e,LL,[],FF,CondT,V,B,G0,G1).
ins_node(n(P0,A,L,F,E,U0),P,Cond,CondT,V,B,n(P0,A,L,F,E,U),G0,G1) :-
    ins_node(U0,P,Cond,CondT,V,B,U,G0,G1).
ins_node(g(Blt,A,D,S,Uni,N,Unc0),P,Cond,CondT,V,B,g(Blt,A,D,S,Uni,N,Unc),G0,G1):-
    ins_node(Unc0,P,Cond,CondT,V,B,Unc,G0,G1).

ins_type(a(Atom),A0,A,L,L,F,F,CondT,V,B,G0,G1) :-
    ins_case(A0,Atom,CondT,V,B,A,G0,G1).
ins_type(list,A,A,L0,L,F,F,CondT,V,B,G0,G1) :-
    ins_index(CondT,L0,V,B,L,G0,G1).
ins_type(funct(Funct,Arty),A,A,L,L,F0,F,CondT,V,B,G0,G1) :-
    ins_case(F0,Funct/Arty,CondT,V,B,F,G0,G1).

ins_case([],V,CondT,Vars,B,[V-T],G0,G1) :-
    ins_index(CondT,e,Vars,B,T,G0,G1).
ins_case([V-T0|Cases],V,CondT,Vars,B,[V-T|Cases],G0,G1) :- !,
    ins_index(CondT,T0,Vars,B,T,G0,G1).
ins_case([H|Cases0],V,CondT,Vars,B,[H|Cases],G0,G1) :-
    ins_case(Cases0,V,CondT,Vars,B,Cases,G0,G1).

ins(e,Vars,B,l([Vars=B]),G,G) :- !.
ins(l(BL),Vars,B,l([Vars=B|BL]),G,G) :- !.
ins(n(P,A,L,F,E,U0),Vars,B,n(P,A,L,F,E,U),G0,G1) :- !,
               ins(U0,Vars,B,U,G0,G1).
ins(g(Blt,A,D,S,Uni,N,Unc0),Vars,B,g(Blt,A,D,S,Uni,N,Unc),G0,G1):-
               ins(Unc0,Vars,B,Unc,G0,G1).


ins_guard(Args,Modes,Blt,Cond,T0,Vars,Body,T,G0,G1):-
 ins_guard(Args,Modes,ArgT,ArgT,[],[],[],Blt,Cond,T0,Vars,Body,T,G0,G1).

ins_guard([],_,Top,[],PSet,Set,Unify,Blt,Cond,T0,Vars,Body,
                     g(Blt,Top,PSet,Set,Unify,T,e),G0,G1):- !,
   ins_index(Cond,T0,Vars,Body,T,G0,G1).
ins_guard([an|Args],[undef|Modes],Top,[(undef,[G0]-an)|Btm],
                 PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G0,G1):- !,
   G2 is G0 - 1,
   ins_guard(Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G2,G1).
ins_guard([an|Args],[Mode|Modes],Top,[(Mode,[G0]-an)|Btm],
                 PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G0,G1):-
   Mode =.. [_,_,out],!,
   G2 is G0 - 1,
   ins_guard(Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G2,G1).
ins_guard(['$VAR'(K)|Args],[Mode|Modes],Top,Btm,
                 PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G0,G1):-
   assoc_with_type(Vars,'$VAR'(K),Pos,OldType,NewType,VarsN),!,
   update_type(OldType,Mode,NewType,M),
   ins_guard_var(OldType,NewType,M,'$VAR'(K),Modes,Pos,Args,
         Top,Btm,PS,Se,Un,Blt,Cond,T0,VarsN,Body,T,G0,G1).
ins_guard(['$VAR'(K)|Args],[Mode|Modes],Top,Btm,
                 PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G0,G1):- !,
   G2 is G0 - 1,
   update_type(new,Mode,NewType,M),
   ins_guard_nvar(NewType,M,'$VAR'(K),Modes,[G0],Args,
         Top,Btm,PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G2,G1).
ins_guard([a(A)|Args],[Mode|Modes],Top,[(Mode,A)|Btm],
                 PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G0,G1):-
   update_type(undef,Mode,atomic-_,in),!,
   ins_guard(Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G0,G1).
ins_guard([a(A)|Args],[Mode|Modes],Top,[(Mode,[G0]-A)|Btm],
                 PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G0,G1):- !,
   G2 is G0 - 1,
   ins_guard(Args,Modes,Top,Btm,[(var([G0]),atom(A))|PS],Se,Un,
                             Blt,Cond,T0,Vars,Body,T,G2,G1).
ins_guard([funct(F,A,Arg)|Args],[Mode|Modes],Top,[(Mode,[G0]-an)|Btm],
                 PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G0,G1):-
   G2 is G0 - 1,
   format(user_error,"Functor Not implement in Guard Builtin: ~w~n",[(Blt,F,A,Arg)]),
   ins_guard(Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G2,G1).


/*
ins_guard_var(OldType,NewType,M,'$VAR'(K),Modes,Pos,Args,
         Top,Btm,PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G0,G1):-
*/
ins_guard_var(undef,undef,M,'$VAR'(K),Modes,Pos,Args,
         Top,[(undef,[G0]-M)|Btm],PS,Se,Un,Blt,
                                Cond,T0,Vars,Body,T,G0,G1):- !,
   G2 is G0 - 1,
   ins_guard(Args,Modes,Top,Btm,PS,Se,[(var([G0]),Pos-'$VAR'(K))|Un],
               Blt,Cond,T0,Vars,Body,T,G2,G1).
ins_guard_var(undef,atomic-Type,in,'$VAR'(_),Modes,Pos,Args,
                   Top,[(atomic-Type,Pos-in)|Btm],PS,Se,Un,
                            Blt,Cond,T0,Vars,Body,T,G0,G1):- !,
   ins_guard_atomic(Pos,Args,Modes,Top,Btm,PS,Se,Un,
                  Blt,Cond,T0,Vars,Body,T,G0,G1).
ins_guard_var(undef,list-Type,in,'$VAR'(_),Modes,Pos,Args,
                   Top,[(list-Type,Pos-in)|Btm],PS,Se,Un,
                            Blt,Cond,T0,Vars,Body,
                            T,G0,G1):- !,
   ins_guard_list(Pos,Args,Modes,Top,Btm,PS,Se,Un,
           Blt,Cond,T0,Vars,Body,T,G0,G1).
ins_guard_var(undef,functor-Type,in,'$VAR'(_),Modes,Pos,Args,
                   Top,[(functor-Type,Pos-in)|Btm],PS,Se,Un,
                            Blt,Cond,T0,Vars,Body,
                            T,G0,G1):- !,
   ins_guard_functor(Pos,Args,Modes,Top,Btm,PS,Se,Un,
           Blt,Cond,T0,Vars,Body,T,G0,G1).
ins_guard_var(_,TypeType,in,'$VAR'(_),Modes,Pos,Args,
                   Top,[(TypeType,Pos-in)|Btm],PS,Se,Un,
                            Blt,Cond,T0,Vars,Body,T,G0,G1):- !,
   ins_guard(Args,Modes,Top,Btm,PS,Se,Un,
           Blt,Cond,T0,Vars,Body,T,G0,G1).
ins_guard_var(_,TypeType,M,'$VAR'(K),Modes,Pos,Args,
                   Top,[(TypeType,[G0]-M)|Btm],PS,Se,Un,
                            Blt,Cond,T0,Vars,Body,T,G0,G1):- !,
   G2 is G0 - 1,
   ins_guard(Args,Modes,Top,Btm,PS,Se,[(var([G0]),Pos-'$VAR'(K))|Un],
           Blt,Cond,T0,Vars,Body,T,G2,G1).

ins_guard_atomic(Pos,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,
           e,Vars,Body,
           n(Pos,([],[T]),(e,[]),([],[]),[],e),G0,G1):- !,
    ins_guard(Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,e,Vars,Body,T,G0,G1).
ins_guard_atomic(Pos,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,
           n(Pos,(A,AB0),L,F,E,U),Vars,Body,
           n(Pos,(A,AB),L,F,E,U),G0,G1):- !,
    ins_guard_case(AB0,AB,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,Vars,Body,G0,G1).
ins_guard_atomic(Pos,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,
           n(Pos0,A,L,F,E,U0),Vars,Body,
           n(Pos0,A,L,F,E,U),G0,G1):- !,
    ins_guard_atomic(Pos,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,U0,Vars,Body,U,G0,G1).

ins_guard_list(Pos,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,
           e,Vars,Body,
           n(Pos,([],[]),(e,[T]),([],[]),[],e),G0,G1):- !,
    ins_guard(Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,e,Vars,Body,T,G0,G1).
ins_guard_list(Pos,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,
           n(Pos,A,(L,LB0),F,E,U),Vars,Body,
           n(Pos,A,(L,LB),F,E,U),G0,G1):- !,
    ins_guard_case(LB0,LB,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,Vars,Body,G0,G1).
ins_guard_list(Pos,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,
           n(Pos0,A,L,F,E,U0),Vars,Body,
           n(Pos0,A,L,F,E,U),G0,G1):- !,
    ins_guard_list(Pos,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,U0,Vars,Body,U,G0,G1).

ins_guard_functor(Pos,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,
           e,Vars,Body,
           n(Pos,([],[]),(e,[]),([],[T]),[],e),G0,G1):- !,
    ins_guard(Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,e,Vars,Body,T,G0,G1).
ins_guard_functor(Pos,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,
           n(Pos,A,L,(F,FB0),E,U),Vars,Body,
           n(Pos,A,L,(F,FB),E,U),G0,G1):- !,
    ins_guard_case(FB0,FB,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,Vars,Body,G0,G1).
ins_guard_functor(Pos,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,
           n(Pos0,A,L,F,E,U0),Vars,Body,
           n(Pos0,A,L,F,E,U),G0,G1):- !,
    ins_guard_functor(Pos,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,U0,Vars,Body,U,G0,G1).

ins_guard_case([],[T],Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,Vars,Body,G0,G1):- !,
    ins_guard(Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,e,Vars,Body,T,G0,G1).
ins_guard_case([VT|Case0],[VT|Case],Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,Vars,Body,G0,G1):- !,
    ins_guard_case(Case0,Case,Args,Modes,Top,Btm,PS,Se,Un,Blt,Cond,Vars,Body,G0,G1).


/*
   ins_guard_nvar(NewType,M,'$VAR'(K),Modes,G0,G1,Args,
         Top,Btm,PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G1).

*/

ins_guard_nvar(undef,_,'$VAR'(K),Modes,Pos,Args,
    Top,[(undef,Pos-0)|Btm],PS,Se,Un,Blt,Cond,T0,Vars,Body,T,G0,G1):- !,
   ins_guard(Args,Modes,Top,Btm,PS,Se,Un,
               Blt,Cond,T0,[('$VAR'(K)=Pos,undef)|Vars],Body,T,G0,G1).
ins_guard_nvar(NewType,out,'$VAR'(K),Modes,Pos,Args,
     Top,[(undef,Pos-out)|Btm],PS,Se,Un,
                   Blt,Cond,T0,Vars,Body,T,G0,G1):- !,
   ins_guard(Args,Modes,Top,Btm,PS,Se,Un,
               Blt,Cond,T0,[('$VAR'(K)=Pos,NewType)|Vars],Body,T,G0,G1).

ins_guard_unify('$VAR'(K),Y,Pos,Cond,T0,Vars,Body,T,G0,G1):-
   assoc_with_type(Vars,'$VAR'(K),Pos,OldType,NewType,NewVars),!,
   update_type(OldType,any-0,NewType,_),
   append(Y,Cond,Cond1),
   ins_index(Cond1,T0,NewVars,Body,T,G0,G1).
ins_guard_unify('$VAR'(K),Y,[G0],Cond,T0,Vars,Body,T,G0,G1):-
   G2 is G0 - 1,
   append(Y,Cond,Cond1),
   ins_index(Cond1,T0,[('$VAR'(K)=[G0],new)|Vars],Body,T,G2,G1).

update_type(undef,any-M,any-M,M):- !.
update_type(new,undef,undef,0):- !.
update_type(_,int-M,atomic-int,M):- !.
update_type(_,atom-M,atomic-atom,M):- !.
update_type(_,string-M,functor-string,M):- !.
update_type(_,vector-M,functor-vector,M):- !.
update_type(_,list-M,list-list,M):- !.
update_type(_,functor-M,functor-functor,M):- !.
update_type(OldMode,any-M,OldMode,M):- !.
update_type(OldMode,undef,OldMode,0):- !.

assoc_with_type([(H=T,Old)|Rest],H,T,Old,M,[(H=T,M)|Rest]):- !.
assoc_with_type([VI|Rest],H,T,Old,M,[VI|NV1]):-
   assoc_with_type(Rest,H,T,Old,M,NV1).

/*
    arg_reg_assoc(0,Arty,AL),
*/
arg_reg_assoc(N,N,[]) :- !.
arg_reg_assoc(K,N,[[K] = a(K)|T]) :-
	K1 is K+1, arg_reg_assoc(K1,N,T).


/*
  Code generation
*/

/*
  gen_code( Source code info to generate object,
	    Label to jump on suspension or failure,
	    First work register number available,
	    Next work register number available,
	    First label number available,
	    Next label number available,
	    Register in which index value should reside,
	    Register association list,
	    Suspension reason index,
	    Max number of suspension reasons,
	    Collected info top,
	    Collected info tail,
	    Top of the generated code,		% implicit argument
	    Tail of the generated code)		% implicit argument
*/

gen_code(other(Oth,SC,Next),Lint,W0,W,L0,L,R,AL,Susp0,Susp,Info0,Info) --> !,
    gen_code(SC,other(Oth),W0,W1,L0,L1,R,AL,Susp0,Susp1,Info0,Info1),
    [label(other(Oth)),otherwise],
    gen_code(Next,Lint,W1,W,L1,L,R,AL,Susp0,Susp2,Info1,Info),
    {max(Susp1,Susp2,Susp)}.
gen_code(alter(Alt,SC,Next),Lint,W0,W,L0,L,R,AL,Susp0,Susp,Info0,Info) --> !,
    gen_code(SC,alter(Alt),W0,W1,L0,L1,R,AL,Susp0,Susp1,Info0,Info1),
    [label(alter(Alt)),alternatively],
    gen_code(Next,Lint,W1,W,L1,L,R,AL,Susp0,Susp2,Info1,Info),
    {max(Susp1,Susp2,Susp)}.
gen_code(e,Lint,W,W,L,L,_,_,Susp,Susp,Info,Info) --> !,[goto(Lint)].
gen_code(l(Bodies),_,W0,W,L0,L,R,AL,Susp,Susp,Info0,Info) --> !,
    gen_bodies(Bodies,R,W0,W,AL,L0,L,Susp,Info0,Info).
gen_code(n(P,A,C,F,E,U),Lint,W0,W,Lloop,L,R0,AL0,Susp0,Susp,Info0,Info) --> !,
    {
	Leq is Lloop+1,
	Luncond is Lloop+2,
	L0 is Lloop+3
    },
    load_elem(P,AL0,AL,R0,R,Reg),
    [label(Lloop), sw_tag(Reg)],
    (
	{ C = (e,[]) } ->
	{ W1 = W0, L1 = L0, Susp1 = 0, Info1 = Info0 }
    ;
	[case_label("CONS")],
	gen_list(C,Leq,W0,W1,L0,L1,Reg,R,AL,Susp0,Susp1,Info0,Info1)
    ),
    (
	{ A = ([],[]) } ->
	{ W2 = W1, L2 = L1, Susp2 = 0, Info2 = Info1 }
    ;
	[case_label("ATOMIC")],
	gen_atoms(A,Leq,W1,W2,L1,L2,Reg,R,AL,Susp0,Susp2,Info1,Info2)
    ),
    [case_label("VARREF")],
    [deref(Reg,Lloop,Luncond)],
    (
	{ F = ([],[]) } ->
	{ W3 = W2, L3 = L2, Susp3 = 0, Info3 = Info2 }
    ;
	[case_label("FUNCTOR")],
	gen_functs(F,Leq,W2,W3,L2,L3,Reg,R,AL,Susp0,Susp3,Info2,Info3)
    ),
    (
	{ C = (e,[]); A = ([],[]); F = ([],[]) } ->
	[case_default, goto(Leq)]
    ;
	[]
    ),
    [end_sw],
    [label(Leq)],
    gen_eq(E,Luncond,W3,W4,L3,L4,Reg,R,AL,Susp0,Susp4,Info3,Info4),
    {
	Susp5 is Susp0+1,
	max_list([Susp1,Susp2,Susp3,Susp4,Susp5], Susp6)
    },
    [label(Luncond)],
    gen_code(U,Lint,W4,W,L4,L,R,AL,Susp6,Susp,Info4,Info).
gen_code(g(B,A,PS,Uni,Se,Node,Unc),Lint,W0,W,Lloop,L,R0,AL0,
                                          Susp0,Susp,Info0,Info) --> !,
    {
        L0 is Lloop + 1
    },
    gen_pset(PS,AL0,AL1,R0,R1),
    gen_gbuilt(B,A,Lint,R1,R2,AL1,AL),
    gen_gb_set(Se,AL),
    gen_gb_unify(Uni,Lloop,L0,L1,AL,Susp0,Susp1),
    gen_code(Node,Lloop,W0,W1,L1,L2,R2,AL,Susp1,Susp2,Info0,Info1),
    [label(Lloop)],
    gen_code(Unc,Lint,W1,W,L2,L,R2,AL,Susp2,Susp,Info1,Info).

gen_pset([],AL,AL,R,R) --> !.
gen_pset([(var(Pos0),atom(A))|Se],AL0,AL,R0,R) -->
    prepare_elem(Pos0,AL0,AL1,R0,R1,Reg0),!,
    [load_atom(Reg0,A)],
    gen_pset(Se,AL1,AL,R1,R).
gen_pset([(var(Pos0),Pos-_)|Se],AL0,AL,R0,R) -->
    prepare_elem(Pos,AL0,AL1,R0,R1,Reg),
    prepare_elem(Pos0,AL1,AL2,R1,R2,Reg0),!,
    [move(Reg,Reg0)],
    gen_pset(Se,AL2,AL,R2,R).

gen_gbuilt(F/A,Args0,Lint,R0,R,AL0,AL) -->
    assoc_gbuilt(Args0,R0,R,AL0,AL,Args),
    [gblt(F,A,Args,Lint)].

assoc_gbuilt([],R,R,AL,AL,[]) --> !.
assoc_gbuilt([(_,Pos-_)|Args0],R0,R,AL0,AL,[Reg|Args]) --> !,
    prepare_elem(Pos,AL0,AL1,R0,R1,Reg),
    assoc_gbuilt(Args0,R1,R,AL1,AL,Args).
assoc_gbuilt([(_,Atom)|Args0],R0,R,AL0,AL,[atom(Atom)|Args]) --> {atomic(Atom)},!,
    assoc_gbuilt(Args0,R0,R,AL0,AL,Args).

gen_gb_set([],_) --> !.
gen_gb_set([(var(Pos0),atom(A))|Se],AL) -->
    {assoc(AL,Pos0,Reg0) },!,
    [load_atom(Reg0,A)],
    gen_gb_set(Se,AL).
gen_gb_set([(var(Pos0),Pos-_)|Se],AL) -->
    { assoc(AL,Pos,Reg),
      assoc(AL,Pos0,Reg0) },
    [move(Reg,Reg0)],
    gen_gb_set(Se,AL).

gen_gb_unify([],_,L,L,_,Susp,Susp) -->  !.
gen_gb_unify([(var(Pos0),Pos-_)|Uni],Luncond,L0,L,AL,Susp0,Susp) --> !,
    { L1 is L + 1,
      Susp1 is Susp0 + 1,
      assoc(AL,Pos,Reg1),
      assoc(AL,Pos0,Reg0)},
    [eq(Reg0,Reg1,L0,Luncond,Luncond),label(L0)],
    gen_gb_unify(Uni,Luncond,L1,L,AL,Susp1,Susp).
gen_gb_unify([(var(Pos0),Value)|Uni],Luncond,L0,L,AL,Susp0,Susp) -->
    { L1 is L + 1,
      Susp1 is Susp0 + 1,
      assoc(AL,Pos0,Reg0)},
    [eq(Reg0,Value,L0,Luncond,Luncond),label(L0)],
    gen_gb_unify(Uni,Luncond,L1,L,AL,Susp1,Susp).

gen_eq([],Lint,W,W,L,L,_,_,_,Susp,Susp,Info,Info) --> !,[goto(Lint)].
gen_eq([Pos-E|T],Lint,W0,W,Leq,L,Reg,R0,AL0,Susp0,Susp,Info0,Info) -->
    {
	Lneq is Leq+1,
	Lundef is Leq+2,
	L0 is Leq+3,
/*	assoc(AL,Pos,Reg1), */
	Susp1 is Susp0+1
    },
    load_elem(Pos,AL0,AL,R0,R,Reg1),
    [eq(Reg,Reg1,Leq,Lneq,Lundef)],
    [label(Leq)],
    gen_code(E,Lneq,W0,W1,L0,L1,R,AL,Susp0,Susp2,Info0,Info1),
    [label(Lundef), push_reason(Reg1,Susp0)],
    [label(Lneq)],
    { max(Susp1, Susp2, Susp3) },
    gen_eq(T,Lint,W1,W,L1,L,Reg,R,AL,Susp3,Susp,Info1,Info).

gen_atoms((Cases,[]),Lint,W0,W,L0,L,Reg,R,AL,Susp0,Susp,Info0,Info) --> !,
   gen_atoms2(Cases,Lint,W0,W,L0,L,Reg,R,AL,Susp0,Susp,Info0,Info).
gen_atoms((Cases,Blts),Lint,W0,W,L0,L,Reg,R,AL,Susp0,Susp,Info0,Info) -->
   { L1 is L0 + 1 },
   gen_atoms2(Cases,L0,W0,W1,L1,L2,Reg,R,AL,Susp0,Susp1,Info0,Info1),
   [label(L0)],
   gen_gb(Blts,Lint,W1,W,L2,L,R,AL,Susp0,Susp2,Info1,Info),
   {max(Susp1,Susp2,Susp)}.

gen_list((Cases,[]),Lint,W0,W,L0,L,_,R,AL,Susp0,Susp,Info0,Info) --> !,
   gen_code(Cases,Lint,W0,W,L0,L,R,AL,Susp0,Susp,Info0,Info).
gen_list((Cases,Blts),Lint,W0,W,L0,L,_,R,AL,Susp0,Susp,Info0,Info) -->
   { L1 is L0 + 1 },
   gen_code(Cases,L0,W0,W1,L1,L2,R,AL,Susp0,Susp1,Info0,Info1),
   [label(L0)],
   gen_gb(Blts,Lint,W1,W,L2,L,R,AL,Susp0,Susp2,Info1,Info),
   {max(Susp1,Susp2,Susp)}.

gen_functs((Cases,[]),Lint,W0,W,L0,L,Reg,R,AL,Susp0,Susp,Info0,Info) --> !,
   gen_functs2(Cases,Lint,W0,W,L0,L,Reg,R,AL,Susp0,Susp,Info0,Info).
gen_functs((Cases,Blts),Lint,W0,W,L0,L,Reg,R,AL,Susp0,Susp,Info0,Info) -->
   { L1 is L0 + 1 },
   gen_functs2(Cases,L0,W0,W1,L1,L2,Reg,R,AL,Susp0,Susp1,Info0,Info1),
   [label(L0)],
   gen_gb(Blts,Lint,W1,W,L2,L,R,AL,Susp0,Susp2,Info1,Info),
   {max(Susp1,Susp2,Susp)}.

gen_atoms2([],Lint,W,W,L,L,_,_,_,Susp,Susp,Info,Info) --> !, [goto(Lint)].
gen_atoms2(Cases,Lint,W0,W,L0,L,Reg,R,AL,Susp0,Susp,Info0,Info) -->
    [sw_atom(Reg)],
    gen_cases(Cases,Lint,W0,W,L0,L,Reg,R,AL,case_atom,Susp0,Susp,Info0,Info).

gen_functs2([],Lint,W,W,L,L,_,_,_,Susp,Susp,Info,Info) --> !, [goto(Lint)].
gen_functs2(Cases,Lint,W0,W,L0,L,Reg,R,AL,Susp0,Susp,Info0,Info) -->
    {Cases = [('VECT'/_)-_|_]},!,
    [sw_vect(Reg)],
    gen_cases(Cases,Lint,W0,W,L0,L,Reg,R,AL,case_vect,Susp0,Susp,Info0,Info).
gen_functs2(Cases,Lint,W0,W,L0,L,Reg,R,AL,Susp0,Susp,Info0,Info) -->
    [sw_funct(Reg)],
    gen_cases(Cases,Lint,W0,W,L0,L,Reg,R,AL,case_funct,Susp0,Susp,Info0,Info).

gen_cases([],Lint,W,W,L,L,_,_,_,_,Susp,Susp,Info,Info) --> !,
    [case_default,goto(Lint),end_sw].
gen_cases([V-Node|T],Lint,W0,W,L0,L,Reg,R,AL,Which,Susp0,Susp,Info0,Info) -->
    { Case =.. [Which,V] },
    [Case],
    gen_code(Node,Lint,W0,W1,L0,L1,R,AL,Susp0,Susp1,Info0,Info1),
    gen_cases(T,Lint,W1,W,L1,L,Reg,R,AL,Which,Susp0,Susp2,Info1,Info),
    { max(Susp1,Susp2,Susp) }.

gen_gb([Node],Lint,W0,W,L0,L,R,AL,Susp0,Susp,Info0,Info) --> !,
    gen_code(Node,Lint,W0,W,L0,L,R,AL,Susp0,Susp,Info0,Info).
gen_gb([Node|T],Lint,W0,W,L0,L,R,AL,Susp0,Susp,Info0,Info) -->
    {L1 is L0 + 1},
/*    gen_code(Node,L2,W0,W1,L1,L2,R,AL,Susp0,Susp1,Info0,Info1),*/
    gen_code(Node,L0,W0,W1,L1,L2,R,AL,Susp0,Susp1,Info0,Info1),
    [label(L0)],
    gen_gb(T,Lint,W1,W,L2,L,R,AL,Susp0,Susp2,Info1,Info),
    { max(Susp1,Susp2,Susp) }.

gen_bodies([],_,W,W,_,L,L,_,Info,Info) --> [].
gen_bodies([Vars=body(_,Body,NV)|Rest],R0,W0,W,AL,L0,L,Susp,Info0,Info) -->
    make_var_assoc(Vars,AL,R0,R1,VarAL),
    { linearize_body(Body,Linear,Queue,Queue,[],NV,_) },
    action_code(Linear,VarAL,_,R1,R2,L0,L2,Susp,Info0,Info1),
    {(R2 > W0 -> W1 = R2 ; W1 = W0)},
    gen_bodies(Rest,R0,W1,W,AL,L2,L,Susp,Info1,Info).

make_var_assoc([],_,R,R,[]) --> [].
make_var_assoc([(Var=Pos,_)|VT],AL0,R0,R,[Var=Where|RT]) -->
    prepare_elem(Pos,AL0,AL,R0,R1,Where),
    make_var_assoc(VT,AL,R1,R,RT).

prepare_elem(Pos,AL,AL,R,R,r(Reg)) --> { assoc(AL,Pos,Reg) }, !.
prepare_elem([Pos],AL0,[[Pos] = x(R0)|AL0],R0,R,x(R0)) --> {Pos < 0},!,
    { R is R0 + 1 }.
prepare_elem([H|T],AL0,[[H|T] = r(x(R0))|AL1],R0,R,e(Reg,H)) -->
    load_elem(T,AL0,AL1,R0,R,Reg).

load_elem(Pos,AL,AL,R,R,Reg) --> { assoc(AL,Pos,Reg) }, !.
load_elem([Pos],AL0,[[Pos] = x(R0)|AL0],R0,R,x(R0)) --> {Pos < 0},!,
    { R is R0 + 1 }.
load_elem([H|T],AL0,[[H|T] = x(R0)|AL1],R0,R,x(R0)) -->
    { R1 is R0+1 },
    load_elem(T,AL0,AL1,R1,R,Reg0),
    [get_elem(x(R0),Reg0,H)].

assoc2([(H=T,_)|_], H, T) :- !.
assoc2([_|Rest], H, T) :-
    assoc2(Rest, H, T).

linearize_body(true,L,L,Q,Q,NV,NV) :- !.
linearize_body((X,Y),L0,L,Q0,Q,NV0,NV) :- !,
    linearize_body(X,L0,L1,Q1,Q,NV0,NV1),
    linearize_body(Y,L1,L,Q0,Q1,NV1,NV).
linearize_body(('{}'=Y),L0,L,Q0,Q,NV0,NV) :- !,
    X =.. ['VECT',0],
    linearize_body((X=Y),L0,L,Q0,Q,NV0,NV).
linearize_body((X='{}'),L0,L,Q0,Q,NV0,NV) :- !,
    Y =.. ['VECT',0],
    linearize_body((X=Y),L0,L,Q0,Q,NV0,NV).
linearize_body((X=Y),[(XX=YY)|L],L,Q,Q,NV0,NV) :- !,
    linearize_body_vector(X,XX,NV0,NV1),
    linearize_body_vector(Y,YY,NV1,NV).
/* temporary */
linearize_body(X,L0,L,Q0,Q,NV0,NV) :-
    functor(X,':=',2),!,
    X =.. [_,Y,Z],
    linearize_exp(Y,Z,L0,L,Q0,Q,NV0,NV).
linearize_body(X,L,L,[X|Q],Q,NV,NV) :- atom(X),!.
linearize_body(X,L0,L,Q0,Q,NV0,NV) :-
    X =.. [F|X1],
    linearize_body_args(X1,X2,L0,L,Q0,[XX|Q],NV0,NV),
    XX =.. [F|X2].

linearize_body_vector('$VAR'(K),'$VAR'(K),NV,NV):- !.
linearize_body_vector(X,XX,NV0,NV) :- 
    functor(X,'{}',1), !,
    X =.. [_,V],
    linearize_body_vectors(V,VV,1,M,NV0,NV),
    XX =.. ['VECT',M|VV].
linearize_body_vector([X|Y],[XX|YY],NV0,NV):- !,
    linearize_body_vector(X,XX,NV0,NV1),
    linearize_body_vector(Y,YY,NV1,NV).
linearize_body_vector(X,X,NV,NV).

linearize_body_vectors((X,Y),[XX|VV],M0,M,NV0,NV) :- !,
    M1 is M0 + 1,
    linearize_body_vector(X,XX,NV0,NV1),
    linearize_body_vectors(Y,VV,M1,M,NV1,NV).
linearize_body_vectors(X,[XX],M,M,NV0,NV):-
    linearize_body_vector(X,XX,NV0,NV).

linearize_body_args([],[],L,L,Q,Q,NV,NV) :- !.
linearize_body_args([X|Y],[X1|YY],L0,L,Q0,Q,NV0,NV) :- !,
    linearize_body_arg(X,X1,L0,L1,Q0,Q1,NV0,NV1),
    linearize_body_args(Y,YY,L1,L,Q1,Q,NV1,NV).
linearize_body_args((X,Y),[X1|YY],L0,L,Q0,Q,M0,M,NV0,NV) :- !,
    M1 is M0 + 1,
    linearize_body_arg(X,X1,L0,L1,Q0,Q1,NV0,NV1),
    linearize_body_args(Y,YY,L1,L,Q1,Q,M1,M,NV1,NV).
linearize_body_args(X,[X1],L0,L,Q0,Q,M0,M,NV0,NV) :- !,
    M is M0 + 1,
    linearize_body_arg(X,X1,L0,L,Q0,Q,NV0,NV).

linearize_body_arg('$VAR'(K),'$VAR'(K),L,L,Q,Q,NV,NV):- !.
linearize_body_arg(X,X1,L0,L,Q0,Q,NV0,NV) :-
    functor(X,'~',1),!,
    X =.. [_,Y],
    linearize_exp(X1,Y,L0,L,Q0,Q,NV0,NV).
linearize_body_arg(X,X1,L,L,Q,Q,NV0,NV) :-
/* temporary */
    functor(X,'{}',1),!,
    X =.. [_,V],
    linearize_body_vectors(V,VV,1,M,NV0,NV),
    X1 =.. ['VECT',M|VV].
linearize_body_arg([X|Y],[XX|YY],L0,L,Q0,Q,NV0,NV):- !,
    linearize_body_arg(X,XX,L0,L1,Q0,Q1,NV0,NV1),
    linearize_body_arg(Y,YY,L1,L,Q1,Q,NV1,NV).
linearize_body_arg(X,X,L,L,Q,Q,NV,NV).

action_code([],AL,AL,R,R,L,L,_,Info,Info) --> [proceed].
action_code([H],AL0,AL,R0,R,L0,L,Susp,Info0,Info) --> !,
    body_call(H,AL0,AL,R0,R,L0,L,last(Susp),Info0,Info).
action_code([H|T],AL0,AL,R0,R,L0,L,Susp,Info0,Info) -->
    body_call(H,AL0,AL1,R0,R1,L0,L1,non_last,Info0,Info1),
    action_code(T,AL1,AL,R1,R,L1,L,Susp,Info1,Info).

body_call(X=Y,AL0,AL,R0,R,L,L,Last,Info,Info) --> !,
    prepare_arg(X,XX,AL0,AL1,R0,R1),
    prepare_arg(Y,YY,AL1,AL,R1,R2),
    body_unify(XX,YY,R2,R),
    proceed_if_last(Last).
body_call(:(Mod,Call),AL0,AL,R0,R,L,L,Last,
	  [ext(Mod/Prd/Arty)|Info],Info) --> !,
    [set_ext_pred(Mod,Prd,Arty)],
    {
	functor(Call,Prd,Arty),
	Call =.. [_|Args]
    },
    push_goal(Args,0,AL0,AL,R0,R,qp),
    proceed_if_last(Last).
body_call(Builtin,AL0,AL,R0,R,L,L,Last,
	  [ext(builtins/Prd/Arty)|Info],Info) -->
    {
	functor(Builtin,Prd,Arty),
	is_body_builtin(Prd,Arty),
	Builtin =.. [_|Args]
    },
    !,
/*    body_builtin(Builtin,AL0,AL,R0,R,L0,L), */
    [set_ext_pred(builtins,Prd,Arty)],
    push_goal(Args,0,AL0,AL,R0,R,qp),
    proceed_if_last(Last).
body_call(Call,AL0,AL,R0,R,L,L,non_last,Info,Info) -->
    [set_pred(Prd,Arty)],
    {
	functor(Call,Prd,Arty),
	Call =.. [_|Args]
    },
    push_goal(Args,0,AL0,AL,R0,R,qp).
body_call(Call,AL0,AL,R0,R,L,L,last(Susp),[exec(Prd/Arty)|Info],Info) -->
    {
	functor(Call,Prd,Arty),
	Call =.. [_|Args]
    },
    prepare_args(Args,Args1,AL0,AL1,R0,R1),
    move_args(Args1,0,AL1,AL,R1,R),
    [execute(Prd,Arty,Susp)].

proceed_if_last(last(_)) --> [proceed].
proceed_if_last(non_last) --> [].

body_unify(e(Reg,K),Y,R0,R) --> !,
    { R1 is R0+1 },
    [get_elem(x(R0),Reg,K)],
    body_unify(r(x(R0)),Y,R1,R).
body_unify(X,e(Reg,K),R0,R) --> !,
    { R1 is R0+1 },
    [get_elem(x(R0),Reg,K)],
    body_unify(X,r(x(R0)),R1,R).
body_unify(r(R0),r(R1),R,R) --> !, [unify(R0,R1)].
body_unify(r(R0),a(A1),R,R) --> !, [unify_atom(R0,A1)].
body_unify(r(R0),v(R1),R,R) --> !, [equiv(R0,R1)].
/* bug? */
body_unify(v(R0),r(R1),R,R) --> !, [equiv(R1,R0)].
body_unify(v(R0),a(A1),R,R) --> !, [load_atom(R0,A1)].
/* */
body_unify(a(A0),a(A0),R,R) --> !, [].
body_unify(a(A0),a(A1),R,R) --> !,
    {
	format(user_error,
	"Constant unification failure: ~w = ~w~n",
	[A0, A1])
    }.
body_unify(a(A0),r(R1),R,R) --> !, [unify_atom(R1,A0)].
body_unify(a(A0),v(R1),R,R) --> !, [load_atom(R1,A0)].
body_unify(v(R0),v(R1),R,R) --> !,
    [alloc_var(R0,0),equiv(R0,R1),make_space(1)].
body_unify(X,Y) :- body_unify(Y,X).

push_goal([],_,AL,AL,R,R,_) --> [push_goal].
push_goal([A0|Args],Pos0,AL0,AL,R0,R,GR0) -->
    (
	{ max_arg_pos(Pos0), Args \== [] } ->
	  [extend_goalrec(GR0,GR)],
	   { Pos1 is -2, GR = xp }
    ;
	{ Pos1 = Pos0, GR = GR0 }
    ),
    prepare_arg(A0,A,AL0,AL1,R0,R1),
    set_arg(A,Pos1,AL1,AL2,R1,R2,GR),
    { Pos is Pos1+1 },
    push_goal(Args,Pos,AL2,AL,R2,R,GR).

set_arg(a(A),P,AL,AL,R,R,GR) --> !, [set_atomic(P,A,GR)].
set_arg(r(Reg),P,AL,AL,R,R,GR) --> !, [set_value(P,Reg,GR)].
set_arg(v(Reg),P,AL,AL,R,R,GR) --> !, [set_newvar(P,Reg,GR)].
set_arg(e(Reg,K),P,AL,AL,R,R,GR) --> !, [set_elem(P,Reg,K,GR)].

move_args([],_,AL,AL,R,R) --> !.
move_args([A|As],P0,AL0,AL,R0,R) --> { used_in(As,a(P0)) }, !,
    { R01 is R0+1 },
    move_arg(A,x(R0),AL0,AL1,R01,R1),
    { P is P0+1 },
    move_args(As,P,AL1,AL,R1,R),
    [move(a(P0),x(R0))].
move_args([A|As],P0,AL0,AL,R0,R) --> !,
    move_arg(A,a(P0),AL0,AL1,R0,R1),
    { P is P0+1 },
    move_args(As,P,AL1,AL,R1,R).

move_arg(a(A),Areg,AL,AL,R,R) --> !, [load_atom(Areg,A)].
move_arg(r(Reg),Areg,AL,AL,R,R) --> !, [move(Areg,Reg)].
move_arg(v(Reg),Areg,AL,AL,R,R) --> !, [load_newvar(Areg,Reg)].
move_arg(e(Reg,K),Areg,AL,AL,R,R) --> !, [get_elem(Areg,Reg,K)].

used_in([r(Reg)|_], Reg) :- !.
used_in([e(Reg,_)|_], Reg) :- !.
used_in([_|T], Reg) :- used_in(T, Reg).

prepare_args([],[],AL,AL,R,R) --> [].
prepare_args([H0|T0],[H|T],AL0,AL,R0,R) -->
    prepare_arg(H0,H,AL0,AL1,R0,R1),
    prepare_args(T0,T,AL1,AL,R1,R).

prepare_arg(A,a(A),AL,AL,R,R) --> { atomic(A) }, !.
prepare_arg('$VAR'(K),Where,AL,AL,R,R) -->
    { assoc(AL,'$VAR'(K),Where) }, !.
prepare_arg('$VAR'(K),v(x(R0)),AL,['$VAR'(K)=r(x(R0))|AL],R0,R) --> !,
    { R is R0+1 }.
prepare_arg([H0|T0],r(x(R00)),AL0,AL,R00,R) --> !,
    { R0 is R00+1 },
    prepare_args([T0,H0],As,AL0,AL,R0,R),
    alloc_args(As,0),
    [make_cons(x(R00)), make_space(2)].
prepare_arg(S,r(x(R00)),AL0,AL,R00,R) --> !,
    {
	functor(S,F,A),
	S =.. [_|As0],
	R0 is R00+1
    },
    prepare_args(As0,As,AL0,AL,R0,R),
    { A1 is A+1 },
    alloc_args(As,1),
    [make_functor(x(R00),F,A), make_space(A1)].

alloc_args([],_) --> [].
alloc_args([H|T],K) -->
    alloc_arg(H,K),
    { K1 is K+1 },
    alloc_args(T,K1).

alloc_arg(a(A),K) --> !, [alloc_atomic(A,K)].
alloc_arg(r(Reg),K) --> !, [alloc_value(Reg,K)].
alloc_arg(v(Reg),K) --> !, [alloc_var(Reg,K)].
alloc_arg(e(Reg,Pos),K) --> !, [alloc_elem(Reg,Pos,K)].

linearize_exp(X,Exp,L0,L,Q0,Q,NV0,NV):-
    functor(Exp,Op,1),
    linearize_op(Op,1,Funct),!,
    numbervars(X,NV0,NV1),
    Exp =.. [_,ExpA],
    linearize_exp2(ExpA,NewExpA,L0,L,Q0,[NewExp|Q],NV1,NV),
    NewExp =.. [Funct,NewExpA,X].
linearize_exp(X,Exp,L0,L,Q0,Q,NV0,NV):-
    functor(Exp,Op,2),
    linearize_op(Op,2,Funct),!,
    numbervars(X,NV0,NV1),
    Exp =.. [_,ExpA,ExpB],
    linearize_exp2(ExpA,NewExpA,L0,L1,Q0,Q1,NV1,NV2),
    linearize_exp2(ExpB,NewExpB,L1,L,Q1,[NewExp|Q],NV2,NV),
    NewExp =.. [Funct,NewExpA,NewExpB,X].
linearize_exp(X,Exp,[(X=Exp)|L],L,Q,Q,NV0,NV):- numbervars(X,NV0,NV).

linearize_exp2(Exp,NewExp,L0,L,Q0,Q,NV0,NV) :-
    functor(Exp,Op,1),
    linearize_op(Op,1,Funct),!,
    numbervars(NewExp,NV0,NV1),
    Exp =.. [_,ExpA],
    linearize_exp2(ExpA,NewExpA,L0,L,Q0,[NewExp2|Q],NV1,NV),
    NewExp2 =.. [Funct,NewExpA,NewExp].
linearize_exp2(Exp,NewExp,L0,L,Q0,Q,NV0,NV) :-
    functor(Exp,Op,2),
    linearize_op(Op,2,Funct),!,
    numbervars(NewExp,NV0,NV1),
    Exp =.. [_,ExpA,ExpB],
    linearize_exp2(ExpA,NewExpA,L0,L1,Q0,Q1,NV1,NV2),
    linearize_exp2(ExpB,NewExpB,L1,L,Q1,[NewExp2|Q],NV2,NV),
    NewExp2 =.. [Funct,NewExpA,NewExpB,NewExp].
linearize_exp2(Exp,Exp,L,L,Q,Q,NV,NV).

linearize_op(+,1,plus).
linearize_op(-,1,minus).
linearize_op(+,2,add).
linearize_op(-,2,subtract).
linearize_op(*,2,multiply).
linearize_op(/,2,divide).
linearize_op(mod,2,modulo).
linearize_op(/\,2,and).
linearize_op(\/,2,or).
linearize_op(>>,2,right_shift).
linearize_op(<<,2,left_shift).

macro_expand('#'([A]),A):- !.
macro_expand('#'(_,_),0):- !.

