/* ---------------------------------------------------------- 
%   (C)1993, 1994 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <klic/config.h>
#include "klic.h"
#ifdef USELOCKF
#include <fcntl.h>
#endif

extern void *malloc();

enum filetype { kl1_file, c_file, asm_file, obj_file, other_file };

struct namerec {
  struct namerec *next;
  enum filetype type;
  char *name;
};

static void reverse_names(names)
     struct namerec **names;
{
  struct namerec *one, *next;
  next = *names;
  one = 0;
  while (next != 0) {
    struct namerec *last = one;
    one = next;
    next = one->next;
    one->next = last;
  }
  *names = one;
}

static void error_exit(format, arg)
     char *format, *arg;
{
  (void)fprintf(stderr, format, arg);
  putc('\n', stderr);
  exit(-1);
}

static void usage_error(cmd)
     char *cmd;
{
  error_exit("Usage: %s -options files ...\nSpecify -h option for help", cmd);
}

static void usage_help(cmd)
     char *cmd;
{
  fprintf(stderr, "Usage: %s -options files ...\n", cmd);
  fprintf(stderr, "Options available are:\n");
  fprintf(stderr, "  -c: no link; only compile to .o\n");
  fprintf(stderr, "  -C: no C compilation; only compile to .c\n");
  fprintf(stderr, "  -d: dry run; no actual compilation; implies -v\n");
  fprintf(stderr, "  -D dbmaker: specifies KLIC database manager program\n");
  fprintf(stderr, "  -g: add debug option for C compiler\n");
  fprintf(stderr, "  -K compiler: specifies KL1 to C compiler program\n");
  fprintf(stderr, "  -I incdir: add .h include directory for cc");
  fprintf(stderr, "  -L libdir: add library directory for linkage");
  fprintf(stderr, "  -l libary: add library archive for linkage");
  fprintf(stderr, "  -o file: specifies executable file name");
  fprintf(stderr, "  -O[n]: specifies cc optimization");
  fprintf(stderr, "  -R: recompile everything ignoring file dates");
  fprintf(stderr, "  -S: generate assembly language code");
  fprintf(stderr, "  -v: verbose compilation");
  fprintf(stderr, "  -n: link with non-debugging runtime system");
  fprintf(stderr, "  -x dbdir: specifies KLIC database directory");
  fprintf(stderr, "  -X initdbdir: specifies database initiation directory");
}

static void access_error(file)
     char *file;
{
  error_exit("Can't access file %s", file);
}

static void get_stat(file, st)
     char *file;
     struct stat *st;
{
  if (stat(file, st) != 0) {
    access_error(file);
  }
}

/* Options */

char *ofile = 0;
int optlevel = 0;
char *optflags;
char debug = 1;
char cdebug = 0;
char verbose = 0;
char no_link = 0;
char no_cc = 0;
char make_asm = 0;
char dryrun = 0;
char do_recompile = 0;
char *dbdir = 0;
char *initdbdir = KLICLIB;

char *klic_compiler = KLIC_COMPILER;
char *klic_dbmaker = KLIC_DBMAKER;
char *klic_libdir = KLICLIB;

struct namerec *files = 0;
struct namerec *incdirs = 0;
struct namerec *libraries = 0;
struct namerec *libdirs = 0;

char buf[BUFSIZE];
time_t inittime;

static void exec_command(msg, info, garbage)
     char *msg;
     char *info;
     void (*garbage)();
{
  if (verbose) {
    (void)fputs(buf, stderr);
    (void)putc('\n', stderr);
  }
  if (!dryrun) {
    if (system(buf) != 0) {
      (void)fprintf(stderr, msg, info);
      (void)putc('\n', stderr);
      garbage(info);
      exit(-1);
    }
  }
}

static char *make_path(name)
     char *name;
{
  char *path;
  if (dbdir != 0) {
    path = (char *)malloc(strlen(dbdir)+strlen(name)+2);
    (void)sprintf(path, "%s/%s", dbdir, name);
  } else {
    path = name;
  }
  return path;
}

/* Cleaning after commmand failure */
static void delete_file(file, prefix, suffix)
     char *file, *prefix, *suffix;
{
  buf[0] = 0;
  (void)sprintf(buf, "%s%s%s%s%s",
		(prefix ? prefix : ""), (prefix ? "/" : ""),
		file,
		(suffix ? "." : ""), (suffix ? suffix : ""));
  if (unlink(buf) == 0) (void)fprintf(stderr, "%s removed\n", buf);
}

static void failed_kl1c(file)
     char *file;
{
  delete_file(file, 0, "c");
  delete_file(file, 0, "ext");
}

static void failed_makedb(dbdir)
     char *dbdir;
{
  delete_file("klic", dbdir, "db");
  delete_file("atom", 0, "h");
  delete_file("funct", 0, "h");
  delete_file("atom", dbdir, "c");
  delete_file("funct", dbdir, "c");
}

static void noop(dummy)
     char *dummy;
{}

static void kl1_to_c()
{
  struct namerec *f = files;
  while (f != 0) {
    if (f->type == kl1_file) {
      int recompile = 0;
      struct stat kl1_stat, c_stat, ext_stat;
      (void)sprintf(buf, "%s.kl1", f->name);
      get_stat(buf, &kl1_stat);
      (void)sprintf(buf, "%s.c", f->name);
      if (do_recompile ||
	  stat(buf, &c_stat) != 0 ||
	  kl1_stat.st_mtime > c_stat.st_mtime) {
	recompile = 1;
      } else {
	(void)sprintf(buf, "%s.kl1", f->name);
	if (do_recompile ||
	    stat(buf, &ext_stat) != 0 ||
	    kl1_stat.st_mtime > ext_stat.st_mtime) {
	  recompile = 1;
	}
      }
      if (recompile) {
	(void)sprintf(buf, "%s %s.kl1 </dev/null", klic_compiler, f->name);
	exec_command("KL1 to C translation failed for %s",
		     f->name, failed_kl1c);
      }
      f->type = c_file;
    }
    f = f->next;
  }
}

make_database()
{
  struct namerec *f;
  struct stat initstat;
  int makedb = 0;
  (void)sprintf(buf, "%s%sklicdb.init",
		(initdbdir ? initdbdir : ""),
		(initdbdir ? "/" : ""));
  get_stat(buf, &initstat);
  inittime = initstat.st_mtime;
  (void)sprintf(buf, "%s %s%s%s%s%s",
		klic_dbmaker,
		(no_cc ? " -n" : ""),
		(initdbdir ? " -X " : ""), (initdbdir ? initdbdir : ""),
		(dbdir ? " -x " : ""), (dbdir ? dbdir : ""));
  for (f = files; f != 0; f = f->next) {
    char xbuf[BUFSIZE];
    (void)sprintf(xbuf, "%s.ext", f->name);
    if (access(xbuf, F_OK) == 0) {
      makedb = 1;
      (void)strcat(buf, " ");
      (void)strcat(buf, xbuf);
    }
  }
  if (makedb || !no_cc) {
    exec_command("Atom/functor database merging failed",
		 dbdir, failed_makedb);
  }
}

void compile_c_file(name, type, masm)
     char *name;
     enum filetype type;
     int masm;
{
  int recompile;
  char suffix = (masm ? 's' : 'o');
  char *bufp;
  if (type == c_file) {
    struct stat c_stat, o_stat;
    (void)sprintf(buf, "%s.c", name);
    get_stat(buf, &c_stat);
    (void)sprintf(buf, "%s.%c", name, suffix);
    recompile =
      (do_recompile ||
       stat(buf, &o_stat) != 0 ||
       c_stat.st_mtime > o_stat.st_mtime ||
       inittime > o_stat.st_mtime);
    if (recompile) {
      struct namerec *inc;
      (void)
	sprintf(buf, "%s -%c %s%s -o %s.%c",
		CC, (masm ? 'S' : 'c'), optflags,
		(cdebug ? " -g" : ""), name, suffix);
      bufp = buf+strlen(buf);
      for (inc = incdirs; inc != 0; inc = inc->next) {
	(void)sprintf(bufp, " -I%s", inc->name);
	bufp += strlen(bufp);
      }
      (void)sprintf(bufp, " -I%s -I. %s.c", KLICINCLUDE, name);
      exec_command("C compilation failed for file %s.c",
		   name, noop);
    }
  } else if (type == asm_file && !masm) {
    struct stat s_stat, o_stat;
    (void)sprintf(buf, "%s.s", name);
    get_stat(buf, &s_stat);
    (void)sprintf(buf, "%s.o", name);
    if (do_recompile ||
	stat(buf, &o_stat) != 0 ||
	s_stat.st_mtime > o_stat.st_mtime) {
      (void)sprintf(buf, "%s -c %s%s -I%s -o %s.o %s.s",
		    CC, optflags,
		    (cdebug ? " -g" : ""),
		    KLICINCLUDE, name, name);
      exec_command("Assembly failed for file %s.s",
		   name, noop);
    }
  }
}

static void c_to_o()
{
  struct namerec *f;
  for (f = files; f != 0; f = f->next) {
    compile_c_file(f->name, f->type, make_asm);
    f->type = (make_asm ? asm_file : obj_file);
  }
}

make_atom_o()
{
  compile_c_file(make_path("atom"), c_file, 0);
}

make_funct_o()
{
  compile_c_file(make_path("funct"), c_file, 0);
}

make_pred_o()
{
  compile_c_file(make_path("predicates"), c_file, 0);
}

linkage()
{
  struct namerec *f;
  char *bufp;

  /* We will link the program anyway, as we can't tell whether the */
  /* existing executable was linked with the tracing library or not */

  (void)sprintf(buf, "%s%s%s %s %s %s",
		LD,
		(ofile ? " -o " : ""), (ofile ? ofile : ""),
		make_path("atom.o"),
		make_path("funct.o"),
		( debug ? make_path("predicates.o") : ""));
  bufp = buf+strlen(buf);
  for (f = files; f != 0; f = f->next) {
    (void)sprintf(bufp, " %s.o", f->name);
    bufp += strlen(bufp);
  }
  for (f = libdirs; f != 0; f = f->next) {
    (void)sprintf(bufp, " -L%s", f->name);
    bufp += strlen(bufp);
  }
  (void)sprintf(bufp, " -L%s", klic_libdir);
  bufp += strlen(bufp);
  (void)sprintf(bufp, " %s", (debug ? LIBRARIES_T : LIBRARIES ));
  bufp += strlen(bufp);
  for (f = libraries; f != 0; f = f->next) {
    (void)sprintf(bufp, " -l%s", f->name);
    bufp += strlen(bufp);
  }
  exec_command("Linkage failed", 0, noop);
}

#define Optarg() \
( argv[optind][charind+1] != 0 ? \
  argv[optind]+charind+1 : \
  (optind++, argv[optind] ))

main(argc,argv)
     int argc;
     char **argv;
{
  int optind;
  int optc;
  int c;

#ifdef SETLINEBUF
  setlinebuf(stderr);
#endif
  for (optind=1;
       optind<argc && argv[optind][0]=='-';
       optind++) {
    int charind;
    for (charind = 1;
	 argv[optind][charind] != 0;
	 charind++) {
      switch (c=argv[optind][charind]) {
      case 'c': no_link = 1; break;
      case 'C': no_cc = 1; break;
      case 'd': dryrun = 1; verbose = 1; break;
      case 'D': klic_dbmaker = Optarg(); goto nextarg;
      case 'g': cdebug = 1; break;
      case 'K': klic_compiler = Optarg(); goto nextarg;
      case 'h': usage_help(argv[0]); exit(0);
      case 'I':
      case 'L':
      case 'l': {
	struct namerec *newname =
	  (struct namerec *)malloc(sizeof(struct namerec));
	newname->name = Optarg();
	newname->type = other_file;
	switch (c) {
	case 'I':
	  newname->next = incdirs;
	  incdirs = newname;
	  break;
	case 'L':
	  newname->next = libdirs;
	  libdirs = newname;
	  break;
	case 'l':
	  newname->next = libraries;
	  libraries = newname;
	  break;
	}
	goto nextarg;
      }
      case 'o': ofile = Optarg(); goto nextarg;
      case 'O':
	if (isdigit(argv[optind][charind+1])) {
	  optlevel = atoi(argv[optind]+charind+1);
	  goto nextarg;
	} else {
	  optlevel = -1;
	  break;
	}
      case 'R': do_recompile = 1; break;
      case 'S': make_asm = 1; break;
      case 'v': verbose = 1; break;
      case 'n': debug = 0; break;
      case 't':
	fprintf(stderr, "This version links debugging library by default\n");
	fprintf(stderr, "To avoid linking debugging library, specify -n\n");
	break;
      case 'X': initdbdir = Optarg(); goto nextarg;
      case 'x': dbdir = Optarg(); goto nextarg;
      default: usage_error(argv[0]);
      }
    }
  nextarg:;
  }

  if (optlevel==0) {
    optflags = "";
  } else {
    if (optlevel<1) {
      (void)sprintf(buf, " -O %s", UOPTFLAGS);
    } else {
      (void)sprintf(buf, " -O%d %s", optlevel, UOPTFLAGS);
    }
    optflags = strcpy((char *)malloc(strlen(buf)+1), buf);
  }
  for (optc = optind; optc < argc; optc++) {
    char *suffix;
    struct namerec *newname =
      (struct namerec *)malloc(sizeof(struct namerec));
    suffix = strrchr(argv[optc], '.');
    if (suffix == 0) {
      error_exit("Can't process file: %s", argv[optc]);
    } else {
      int len = suffix-argv[optc];
      newname->name =
	strncpy((char *)malloc(len+1), argv[optc], len);
      newname->name[len] = 0;
      if (strcmp(suffix, ".kl1") == 0) {
	newname->type = kl1_file;
      } else if (strcmp(suffix, ".c") == 0) {
	newname->type = c_file;
      } else if (strcmp(suffix, ".a") == 0) {
	newname->type = asm_file;
      } else if (strcmp(suffix, ".o") == 0) {
	newname->type = obj_file;
      } else {
	error_exit("Can't process file: %s", argv[optc]);
      }
    }
    newname->next = files;
    files = newname;
  }

  reverse_names(&files);
  reverse_names(&libraries);
  reverse_names(&libdirs);

  kl1_to_c();
  make_database();
  if (!no_cc) {
    c_to_o();
    if (!make_asm) {
#ifdef USELOCKF
      int fd;
      char fdbuf[BUFSIZE];
      strcpy(fdbuf,make_path(DBFILENAME));
      strcat(fdbuf,".lock");
      lockf(fd=open(fdbuf,O_RDONLY),F_LOCK,1);
#endif
      if (!no_link) {
	make_atom_o();
	make_funct_o();
	if (debug) make_pred_o();
	linkage();
      }
#ifdef USELOCKF
      close(fd);
      lockf(fd,F_ULOCK,1);
#endif
    }
  }
  return 0;
}
