/*- -*- Mode: C++ -*-							 -*/
/*- Copyright (C) 1992 Institute for New Generation Computer Technology. -*/
/*- $BG[IU$=$NB>$O(B COPYRIGHT $B%U%!%$%k$r;2>H$7$F$/$@$5$$!%(B                  -*/
/*- (Read COPYRIGHT for detailed information.)                           -*/
/*-                                                                      -*/
/*-		    Author: Shinji Yanagida (yanagida@nsis.cl.nec.co.jp) -*/
/*-		    Author: Toshio Tange (t-tange@nsis.cl.nec.co.jp)	 -*/

#include "config.h"
#include "aum/error.h"
#include "table/protocol.h"
#include "table/atomhash.h"
#include "aum/debugger.h"

#ifdef PAS_DEBUGGER
#include <stdio.h>
#include <stream.h>
#endif

_ProtocolTable* ProtocolTable;


// /////////////////////////////////////////////////////////////////
// class ProtocolEntry
// /////////////////////////////////////////////////////////////////

inline u_long
MakeFunctor(const Word atom, const u_char arity)
{
    return (arity << PROE_ARITY_SHIFT_WIDTH) + AtomNumber(atom);
}

Name const
ProtocolEntry::Printname() const
{
    u_long anum = PROE_ANUM_MASK & functor;
    AtomEntry* ae = AtomTable->AtomEntry(anum);
    return ae->Printname();
}

const u_char
ProtocolEntry::Arity()
    const
{
    return u_char(functor >> PROE_ARITY_SHIFT_WIDTH);
}

const u_long
ProtocolEntry::Mode_mask()
    const
{
    return modes;
}

// /////////////////////////////////////////////////////////////////
// class _ProtocolTable
// /////////////////////////////////////////////////////////////////

void
_ProtocolTable::Initialize(int size)
{
    if (size < MAX_PROTOCOL_ID)
	size = MAX_PROTOCOL_ID;
    max_of_pids = MAX_PROTOCOL_ID;
    no_of_pids = START_USER_PID;
    Array = (ProtocolEntry*)SHARED_ALLOC(size * sizeof(ProtocolEntry));
}
#ifdef PAS_DEBUGGER
void
_ProtocolTable::End__ProtocolTable()
{
    SHARED_FREE(Array,max_of_pids * sizeof(ProtocolEntry));
}
#endif
ProtocolEntry*
_ProtocolTable::NextEntry(u_long f, u_long m)
{
    if (no_of_pids >= max_of_pids)
	fatal("InternProtocolID",
	      "Protocol-ID Table Overflow maxsize=(%d)",
	      max_of_pids);

    ProtocolEntry* pe = &Array[no_of_pids++];
    pe->Initialize(f, m);
    return pe;
}

ProtocolID
_ProtocolTable::get_ProtocolID(const ProtocolEntry* address) const
{
    return ProtocolID(address->Arity(), address-&Array[0], PT_Compound);
}

ProtocolEntry*
_ProtocolTable::get_ProtocolEntry(const int index)
{
    if (index >= max_of_pids)
	fatal("get_ProtocolEntry", "argument is not a protocol ID");
    return &Array[index];
}


///////////////////////////////////////////////////////////////////////////////

void*
Node::operator new(size_t s)
{
    return SHARED_ALLOC(s);
}

void
Node::operator delete(void* v)
{
    SHARED_FREE(v, sizeof(Node));
}

///////////////////////////////////////////////////////////////////////////////

static Node* HashTable[PROTOCOL_HASH_SIZE];

ProtocolID
InternProtocolID(Word atom, u_char arity, u_long modes)
{
    u_long  functor = MakeFunctor(atom, arity);
    u_long  index   = (AtomNumber(atom)+arity) % PROTOCOL_HASH_SIZE;

    for (Node* head = HashTable[index]; head; head = head->Next()) {
	if (head->Entry()->Equal(functor, modes) == TRUE)
	    return ProtocolTable->get_ProtocolID(head->Entry());
    }

    ProtocolEntry *entry = ProtocolTable->NextEntry(functor, modes);
    HashTable[index] = new Node(entry, HashTable[index]);

    return ProtocolTable->get_ProtocolID(entry);
}

ProtocolID
SearchProtocolID(Word atom, u_char arity, u_long modes)
{
    u_long  functor = MakeFunctor(atom, arity);
    u_long  index   = (AtomNumber(atom)+arity) % PROTOCOL_HASH_SIZE;

    for (Node* head = HashTable[index]; head; head = head->Next()) {
	ProtocolEntry* pe = head->Entry();
	if (pe->Equal(functor,modes))
	    return ProtocolTable->get_ProtocolID(pe);
    }
    return SearchDefaultPID;
}

#ifdef PAS_DEBUGGER
void
End_ProtocolIDHash()
{
    for (int index = 0;index < PROTOCOL_HASH_SIZE;index++) {
	for (Node* head = HashTable[index]; head;) {
	    Node* work = head;
	    head = head->Next();
	    delete work;
	}
	HashTable[index] = (Node*)0;
    }
}
extern Word InternAtom(Name name);
int Symbol2Pid_Entry_Index(unsigned char* str)
{
    unsigned char buff[BUFSIZ];
    int np = 0;
    u_char arity = 0x0;
    u_long mode = 0x0;
    while((*str != '\0')&&(*str != '/')){
	*(buff+np) = *str;
	np++;
	str++;
    }
    *(buff+np) = '\0';
    ProtocolID pid;
    Word atomname;
    if (*str == '/'){
	str++;
	while(*str != '\0'){
	    if (*str == '-'){
		arity++;
		str++;
	    }else if (*str == '+'){
		mode |= (0x1<<arity);
		arity++;
		str++;
	    }else
		return -1;
	}
	atomname = InternAtom((char*)buff);
    }else{
	atomname = InternAtom((char*)buff);
    }
    if ((pid = SearchProtocolID(atomname,arity,mode)) == SearchDefaultPID)
	return -1;
    for (u_long index = 0;index<(u_long)(ProtocolTable->No_of_pids());index++){
	ProtocolEntry* pe = ProtocolTable->get_ProtocolEntry(index);
	if (ProtocolTable->get_ProtocolID(pe)== pid)
	    return (int)index;
    }
    return -1;
}
ProtocolEntry*
SearchProtocolID(ProtocolID& so)
{

    for (u_int index = 0;index<PROTOCOL_HASH_SIZE;index++)
	for (Node* head = HashTable[index]; head; head = head->Next()) {
	    if (ProtocolTable->get_ProtocolID(head->Entry())==so)
		return head->Entry();
	}
    return (ProtocolEntry*)0;
}
#endif

///////////////////////////////////////////////////////////////////////////////

ProtocolID intern_pid(Name name, u_char arity, u_long modes)
{
    Word atom = InternAtom(name);
    return InternProtocolID(atom, arity, modes);
}

static void
ReserveAtom(Name name, Word reserve)
{
    Word atom = InternAtom(name);
    if (atom != reserve)
	fatal(form("ReserveAtom %s", name)
	      ,"no match a reserved atom with interned atom");

}

static void
ReservePID(Name name, u_char arity, u_long modes, ProtocolID reserve)
{
    ProtocolID pid = intern_pid(name, arity, modes);
    if (pid != reserve)
	fatal(form("ReservePID %s/%d", name, arity)
	      ,"no match a reserved pid with interned pid");

}

void
Initialize_ProtocolID(int size)
{
    if (!ProtocolTable)
	ProtocolTable = (_ProtocolTable*)SHARED_ALLOC(sizeof(_ProtocolTable));

    // set PID to START_USER_PID
    ProtocolTable->Initialize(size);
    // generating in automatically
#include "useratom.gc"
#include "userpid.gc"
}

#ifdef PAS_DEBUGGER
void End_ProtocolID()
{
    End_ProtocolIDHash();
    ProtocolTable->End__ProtocolTable();
    SHARED_FREE((void*)ProtocolTable,sizeof(_ProtocolTable));
    ProtocolTable = (_ProtocolTable*)0;
}
extern message_to_Listener(debugger_symbol);
extern message_to_Listener(debugger_symbol,unsigned char*);
#define LIST_PAGE 20
extern char* sprint_protocol_mode(char* buf, u_char arity, u_long modes);

void _ProtocolTable::Dump(int indent)
{
    char teb[BUFSIZ];
    char argstr[37];
    int table_count,line;
    for (table_count = START_USER_PID,line =  0;table_count < no_of_pids;table_count++,line++) {
	ProtocolEntry* p_ent = Array+table_count;
	int leng = p_ent->Arity();
	u_int ms = p_ent->Mode_mask();
	const char tr = (p_ent->IsTracePoint())?'t':'-';
	const char br = (p_ent->IsBreakPoint())?'b':'-';
	char* w = argstr;
	if (leng != 0){
	    *w = '/';
	    w++;
	}
	for (int ii = 0;ii<leng;ii++,ms>>=1,w++)
	    if (ms%2 == 1)
		*w = '+';
	    else
		*w = '-';
	*w = '\0';
	sprintf(teb," %d %c%c :%s%s \n",table_count,br,tr,
		p_ent->Printname(),argstr);
	message_to_Listener(TRECEIVE,(unsigned char*)teb);
	if ((PAS_npe == 1)&&(line > 25)) {
	    cout << "--- key type return ---";
	    cout.flush();
	    getchar();
	    line = 0;
	}
    }
}
void DumpProtocolIDTable()
{
    ProtocolTable->Dump(0);
    message_to_Listener(NRECEIVE);
}
#endif
/*-----------------
 * Local Variables:
 * c-indent-level:4
 * c-continued-statement-offset:4
 * c-brace-offset:0
 * c-imaginary-offset:0
 * c-argdecl-indent:4
 * c-label-offset:-4
 * c++-electric-colon:t
 * c++-empty-arglist-indent:nil
 * c++-friend-offset:-4
 * c++-member-init-indent-offset:0
 * c++-continued-member-init-offset:nil
 * End:
 */
