/*- -*- Mode: C++ -*-							 -*/
/*- Copyright (C) 1992 Institute for New Generation Computer Technology. -*/
/*- $BG[IU$=$NB>$O(B COPYRIGHT $B%U%!%$%k$r;2>H$7$F$/$@$5$$!%(B                  -*/
/*- (Read COPYRIGHT for detailed information.)                           -*/
/*-                                                                      -*/
/*-		    Author: Shinji Yanagida (yanagida@nsis.cl.nec.co.jp) -*/
/*-		    Author: Toshio Tange (t-tange@nsis.cl.nec.co.jp)	 -*/
#include "config.h"
#include "aum/error.h"
#include "table/atomhash.h"

_AtomTable*	AtomTable;

void
Initialize_AtomTable(int size)
{
    if (!AtomTable)
	AtomTable = (_AtomTable*)SHARED_ALLOC(sizeof(_AtomTable));
    AtomTable->Initialize(size);
}

#ifdef PAS_DEBUGGER
void End_AtomHash();

void End_AtomTable()
{
    if (AtomTable){
	End_AtomHash(); // Atom Chain shared free
	AtomTable->End__AtomTable();
	SHARED_FREE(AtomTable,sizeof(_AtomTable));
	AtomTable = (_AtomTable*)0;
	extern void RefreshPureString();
	RefreshPureString();
    }
}
#endif

void*
AtomChain::operator new(size_t s)
{
    return SHARED_ALLOC(s);
}

void
AtomChain::operator delete(void* v)
{
    SHARED_FREE(v, sizeof(AtomChain));
}

// /////////////////////////////////////////////////////////////////
// class _AtomTable
// /////////////////////////////////////////////////////////////////

void
_AtomTable::Initialize(int size)
{
    if (size < MAX_ATOM_NUMBER)
	size = MAX_ATOM_NUMBER;
    max_of_atoms = size;
    no_of_atoms = 0;
    Array = (class AtomEntry*)SHARED_ALLOC(size * sizeof(class AtomEntry));
}

#ifdef PAS_DEBUGGER
void
_AtomTable::End__AtomTable()
{
    SHARED_FREE(Array,max_of_atoms * sizeof(class AtomEntry));
}
#endif

AtomEntry*
_AtomTable::NextEntry(Name name)
{

    if (no_of_atoms >= max_of_atoms)
	fatal("InternAtom", "Too many intern atoms. Atom table overflowed");

    class AtomEntry* aep = &Array[no_of_atoms++];
    aep->Initialize(name);
    return aep;
}

Word
_AtomTable::Atom(AtomEntry* address) const
{
    return AtomWord(address-&Array[0]);
}

AtomEntry*
_AtomTable::AtomEntry(int index)
{
    if (index >= max_of_atoms)
	fatal("AtomEntry",
	      "%d is too large, max size is %d", index, max_of_atoms);

    return &Array[index];
}

// /////////////////////////////////////////////////////////////////

static AtomChain* AtomHashTable[ATOMHASHSIZE];

u_long Hash(Name x)
{
    u_long h = 0;
    u_long g;
    while (*x != 0) {
	h = (h << 4) + *x++;
	if ((g = h & 0xF0000000) != 0)
	    h = (h ^ (g >> 24)) ^ g;
    }
    return h;
}

Word
InternAtom(Name name)
{
    u_long index = Hash(name) % ATOMHASHSIZE;

    for (AtomChain* head = AtomHashTable[index]; head; head = head->Next()) {
	if (head->Entry()->Equal(name) == TRUE)
	    return AtomTable->Atom(head->Entry());
    }

    AtomEntry* entry = AtomTable->NextEntry(name);
    AtomHashTable[index] = new AtomChain(entry, AtomHashTable[index]);

    return AtomTable->Atom(entry);
}

#ifdef PAS_DEBUGGER
void End_AtomHash()
{
    for (int index = 0;index < ATOMHASHSIZE;index++) {
	for (AtomChain* head = AtomHashTable[index];head;) {
	    AtomChain* work = head;
	    head = head->Next();
	    delete work;
	}
	AtomHashTable[index] = (AtomChain*)0;
    }
}
#endif

Name
AtomPrintname(Word atom)
{
    if (!IsAtom(atom)) {
	if (IsGarbageCollected(atom)) return "*GARBAGE*";
	fatal ("AtomPrintname", INVALID_DATA_TYPE,form("%x",atom));
	return "*ERROR*";
    }
    u_long aid = AtomNumber(atom);
    AtomEntry* item = AtomTable->AtomEntry(aid);
    return item->Printname();
}

/*-----------------
 * Local Variables:
 * c-indent-level:4
 * c-continued-statement-offset:4
 * c-brace-offset:0
 * c-imaginary-offset:0
 * c-argdecl-indent:4
 * c-label-offset:-4
 * c++-electric-colon:t
 * c++-empty-arglist-indent:nil
 * c++-friend-offset:-4
 * c++-member-init-indent-offset:0
 * c++-continued-member-init-offset:nil
 * End:
 */
