/*- -*- Mode: C -*-						       -*/
/*- Copyright (C) 1992 Institute for New Generation Computer Technology. -*/
/*- $BG[IU$=$NB>$O(B COPYRIGHT $B%U%!%$%k$r;2>H$7$F$/$@$5$$!%(B                  -*/
/*- (Read COPYRIGHT for detailed information.)                           -*/
/*-                                                                      -*/
/*-		    Author: Shinji Yanagida (yanagida@nsis.cl.nec.co.jp) -*/
/*-		    Author: Toshio Tange (t-tange@nsis.cl.nec.co.jp)	 -*/

#ifdef	sequent
#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <parallel/microtask.h>
#include <parallel/parallel.h>

/*
 * VERBOSE=1 : PE ξ֤Ѳrunning/sleeping ɽ롥
 * VERBOSE=2 : PE ֤Υå̿ɽ롥
 * VERBOSE=3 : VERBOSE=1  VERBOSE=2 ξɽ롥
 */
#ifndef NDEBUG
#define VERBOSE 1
#endif

struct message {
    u_char		h1;
    u_char		h2;
    u_short		h3;
    struct message	*next;
};

/* maximum number of Processing Element */
#define MAX_PE		16

#define SUSPENDED	0
#define ACTIVE		1

struct msgbox {
    slock_t		mutex;
    u_char		status;
    u_short		pid;
    struct message	*first;
    struct message	*last;
    int			resource;
};

#define MESSAGE_RESOURCE	1000

struct resource {
    slock_t		mutex;
    u_char		running_pes;
    short		dummy;
    int			total;
};

shared struct msgbox	msgbox[MAX_PE];
shared struct resource	Resource;
int*			MessageQueue;

#define STATISTICS
#ifdef STATISTICS
static int		PE_nmsgsnd;
int			PE_nmsgrcv;
#endif

extern void abort();
extern long mctime();

#if VERBOSE > 0
extern int		PElog;
extern void		pelog();
#endif

#define PAS_SIG_CONT	SIGUSR1
#define PAS_SIG_KILL	SIGUSR2

static void pas_sig_cont() {}
static void pas_sig_kill() {}
static int  myid;

void
Initialize_Communication()
{
    S_INIT_LOCK(&Resource.mutex);
    Resource.total = 0;
    Resource.running_pes = 0;

    signal(PAS_SIG_CONT, pas_sig_cont);
    signal(PAS_SIG_KILL, pas_sig_kill);
}

void
Initialize_PE()
{
    myid = m_get_myid();
    S_LOCK(&Resource.mutex);
    Resource.total += MESSAGE_RESOURCE;
    Resource.running_pes += 1;
    S_UNLOCK(&Resource.mutex);
    {
	register struct msgbox *mp = &msgbox[myid];
	S_LOCK(&mp->mutex);
	mp->first = mp->last = NULL;
	mp->status = ACTIVE;
	mp->resource = MESSAGE_RESOURCE;
	mp->pid = getpid();
	MessageQueue = (int*)&mp->first;
	S_UNLOCK(&mp->mutex);
    }
    m_sync();
}

static void
reDistribution()
{
    S_LOCK(&Resource.mutex);
    Resource.total += MESSAGE_RESOURCE;
    S_UNLOCK(&Resource.mutex);

    S_LOCK(&msgbox[myid].mutex);
    msgbox[myid].resource += MESSAGE_RESOURCE;
    S_UNLOCK(&msgbox[myid].mutex);
}

void
Write_message_to_pe(msg, dest)
    struct message* msg;
    int dest;
{
#ifndef NDEBUG
    if (dest < 0 || dest >= MAX_PE)
	abort();
    if (msg->h1 == 0xFF || msg->h1 == 0 || msg->h3 == 0xFFFF || msg->h3 == 0)
	abort();
#endif
    {
	msg->next = NULL;
#ifdef STATISTICS
	PE_nmsgsnd++;
#endif
	{
	    S_LOCK(&msgbox[dest].mutex);
	    if (msgbox[dest].first == NULL) {
		msgbox[dest].first = msg;
		msgbox[dest].last = msg;
	    }
	    else {
		msgbox[dest].last->next = msg;
		msgbox[dest].last = msg;
	    }
	    msgbox[dest].resource++;
	    S_UNLOCK(&msgbox[dest].mutex);

	}
	{
	    S_LOCK(&msgbox[myid].mutex);
	    msgbox[myid].resource--;
	    if (msgbox[myid].resource == 0) {
		S_UNLOCK(&msgbox[myid].mutex);
		reDistribution();
	    }
	    else{
		S_UNLOCK(&msgbox[myid].mutex);
	    }
	}
	if (msgbox[dest].status == SUSPENDED) {
	    S_LOCK(&msgbox[dest].mutex);
	    msgbox[dest].status = ACTIVE;
	    S_UNLOCK(&msgbox[dest].mutex);
	    if (kill(msgbox[dest].pid, PAS_SIG_CONT) < 0)
		perror("System Error : kill");
#if VERBOSE & 1
	    if (PElog)
		pelog("activate PE#%d", dest);
#endif
	}
    }
#if VERBOSE & 2
    if (PElog)
	pelog("PE#%d  sent a message %x to PE#%d", m_get_myid(), msg, dest);
#endif
}

int*
Check_SystemMessage(waitflag)
    int waitflag;
{
#ifndef NDEBUG
    if (myid < 0 || myid >= MAX_PE)
	abort();
#endif
    {
	struct msgbox *mbp = &msgbox[myid];
	struct message *first;
	top:
	if ((first = mbp->first) == NULL) {
	    if (!waitflag)
		return NULL;
	    S_LOCK(&mbp->mutex);
	    if ((first = mbp->first) != NULL) {
		S_UNLOCK(&mbp->mutex);
	    }
	    else {
		S_LOCK(&Resource.mutex);
		Resource.total -= mbp->resource;
		S_UNLOCK(&Resource.mutex);
		mbp->status = SUSPENDED;
		mbp->resource = 0;
		S_UNLOCK(&mbp->mutex);
		sigblock(sigmask(PAS_SIG_CONT));
		if (Resource.total == 0) {
		    register int i;
		    for (i = 0; i < Resource.running_pes; i++) {
			if (i != myid)
			    kill(msgbox[i].pid, PAS_SIG_KILL);
		    }
		    return NULL;
		}
		else {
#if VERBOSE & 1
		    if (PElog)
			pelog("suspend (resource is %d)", Resource.total);
#endif
		    sigpause(sigblock(0) &~ sigmask(PAS_SIG_CONT));
#if VERBOSE & 1
		    if (PElog)
			pelog("resume (resource is %d)", Resource.total);
#endif
		    if (Resource.total == 0) {
			return NULL;
		    }
		    reDistribution();
		    goto top;
		}
	    }
	}
	S_LOCK(&mbp->mutex);
	mbp->first = NULL;
	mbp->last = NULL;
	S_UNLOCK(&mbp->mutex);
#if VERBOSE & 2
	if (PElog)
	    pelog("PE#%d  got a message %x", myid, first);
#endif
#ifndef NDEBUG
	if (first->h1 == 0xFF || first->h1 == 0
	    || first->h3 == 0xFFFF || first->h3 == 0)
	    abort();
#endif
	return (int*)first;
    }
}

int
Wait_terminate ()
{
    return Resource.total == 0 ? -1 : 1;
}

struct monitoring {
    int		pe_nmsgsnd;
    int		pe_nmsgrcv;
    int		pid;
    int		ppid;
    float	utime;
    float	stime;
    long	elapsed;
    int		maxrss;
    int		minflt;
    int		majflt;
    int		nswap;
    int		inblock;
    int		oublock;
    int		nsignals;
    int		nvcsw;
    int		nivcsw;
};

shared struct monitoring Monitoring[MAX_PE];

void
Begin_monitoring()
{
#ifdef STATISTICS
    PE_nmsgsnd = 0;
    PE_nmsgrcv = 0;
#endif
    Reset_rusage();
    Monitoring[myid].pid = getpid();
    Monitoring[myid].ppid = getppid();
    Monitoring[myid].elapsed = mctime();
}

void
End_monitoring()
{
    struct rusage ru;

#ifdef STATISTICS
    Monitoring[myid].pe_nmsgsnd = PE_nmsgsnd;
    Monitoring[myid].pe_nmsgrcv = PE_nmsgrcv;
#endif
    Report_rusage(&ru);

    Monitoring[myid].utime =
	((float)(ru.ru_utime.tv_sec * 1000000 + ru.ru_utime.tv_usec))/1000000;
    Monitoring[myid].stime =
	((float)(ru.ru_stime.tv_sec * 1000000 + ru.ru_stime.tv_usec))/1000000;
    Monitoring[myid].elapsed  = mctime() - Monitoring[myid].elapsed;
    Monitoring[myid].maxrss   = ru.ru_maxrss;
    Monitoring[myid].minflt   = ru.ru_minflt;
    Monitoring[myid].majflt   = ru.ru_majflt;
    Monitoring[myid].nswap    = ru.ru_nswap;
    Monitoring[myid].inblock  = ru.ru_inblock;
    Monitoring[myid].oublock  = ru.ru_oublock;
    Monitoring[myid].nsignals = ru.ru_nsignals;
    Monitoring[myid].nvcsw    = ru.ru_nvcsw;
    Monitoring[myid].nivcsw   = ru.ru_nivcsw;
}

void
Report_monitoring()
{
    int i;
    fprintf(stderr,"PE#	  utime	  stime elapsed	 peSnd	peRcv\n");
    for (i = 0; i < Resource.running_pes; i++) {
	fprintf(stderr," %02d %7.2f %7.2f %7.2f %6d %6d\n",
	       i,
	       Monitoring[i].utime,
	       Monitoring[i].stime,
	       ((float)Monitoring[i].elapsed)/1000,
	       Monitoring[i].pe_nmsgsnd,
	       Monitoring[i].pe_nmsgrcv
	       );
    }
    fprintf(stderr,"\n\
PE# MaxRSS  MajorPF  MinorPF  Swaps   blkI   blkO Nsig	 Vcsw	Icsw\n"
	   );
    for (i = 0; i < Resource.running_pes; i++) {
	fprintf(stderr," %02d %6d %8d %8d %6d %6d %6d %4d %6d %6d\n",
	       i,
	       Monitoring[i].maxrss,
	       Monitoring[i].majflt,
	       Monitoring[i].minflt,
	       Monitoring[i].nswap,
	       Monitoring[i].inblock,
	       Monitoring[i].oublock,
	       Monitoring[i].nsignals,
	       Monitoring[i].nvcsw,
	       Monitoring[i].nivcsw);
    }
}

shared int *Sndrcv_data[MAX_PE];

void
Sndrcvstats_data_to_shared_memory (pe, block, nbytes)
    unsigned char pe;
    char *block;
{
    int n = nbytes;
    char *f = block;
    char *t;
    t = shmalloc(n);
    Sndrcv_data[pe] = (int*)t;
    while (--n >= 0)
	*t++ = *f++;
}

void
Sndrcv_reporting (npe, tag, nitem)
    char *tag;
    int nitem;
{
    int i, j;
    int pe;
    int total_rows[MAX_PE];
    int rows[MAX_PE];
    int sum = 0;

    for (i = 0; i < nitem; i++) total_rows[i] = 0;

    for (pe = 0; pe < npe; pe++) {
	int *points = Sndrcv_data[pe];

	for (i = 0; i < nitem; i++) rows[i] = 0;

	fprintf (stderr, "PE%02d send%s	  %%sum\n", pe, tag);
	for (i = 0; i < npe; i++) {
	    fprintf (stderr, "	 to %02d:", i);
	    sum = 0;
	    for (j = 0; j < nitem; j++) {
		int x = points[j*MAX_PE];
		fprintf (stderr, "%7d", x);
		sum += x;
		rows[j] += x;
	    }
	    points++;
	    fprintf (stderr, "%7d\n", sum);
	}
	fprintf (stderr, "     sum:");
	sum = 0;
	for (j = 0; j < nitem; j++) {
	    fprintf (stderr, "%7d", rows[j]);
	    sum += rows[j];
	    total_rows[j] += rows[j];
	}
	fprintf (stderr, "%7d\n", sum);
    }
    fprintf (stderr, "Sum      %s   %%sum\n", tag);
    sum = 0;
    fprintf (stderr, "	       ");
    for (j = 0; j < nitem; j++) {
	fprintf (stderr, "%7d", total_rows[j]);
	sum += total_rows[j];
    }
    fprintf (stderr, "%7d\n", sum);
}
#ifdef DEBUGGER
int shared_resource()
{
    int res = 0;
    int c;
    for (c = 0;c<(Resource.running_pes-1);c++){
	res+=msgbox[c].resource;
    }
    return res;
}
#endif
/*-----------------
 * Local Variables:
 * c-indent-level:4
 * c-continued-statement-offset:4
 * c-brace-offset:0
 * c-imaginary-offset:0
 * c-argdecl-indent:4
 * c-label-offset:-4
 * c++-electric-colon:t
 * c++-empty-arglist-indent:nil
 * c++-friend-offset:-4
 * c++-member-init-indent-offset:0
 * c++-continued-member-init-offset:nil
 * End:
 */
#endif	/* !sequent */
