/*- -*- Mode: C++ -*-							 -*/
/*- Copyright (C) 1992 Institute for New Generation Computer Technology. -*/
/*- $BG[IU$=$NB>$O(B COPYRIGHT $B%U%!%$%k$r;2>H$7$F$/$@$5$$!%(B                  -*/
/*- (Read COPYRIGHT for detailed information.)                           -*/
/*-                                                                      -*/
/*-		    Author: Shinji Yanagida (yanagida@nsis.cl.nec.co.jp) -*/
/*-		    Author: Toshio Tange (t-tange@nsis.cl.nec.co.jp)	 -*/

#include <assert.h>
#include <signal.h>
#include "aum/stdlib.h"
#include "aum/fetch.h"
#include "aum/error.h"
#include "aum/string.h"
#include "aum/builtin.h"
#include "class/template.h"
#include "table/nlist.h"

static const char TAG[] = "foreign_call";

static union {
    double dv;
    int iv[2];
} foreign_di_conv;

static int
AUmdouble_to_Cdouble (double d, int i, int argv[], const char* ptrs[])
{
    foreign_di_conv.dv = d;
    ptrs[i] = NULL, argv[i] = foreign_di_conv.iv[0];
    i++;
    ptrs[i] = NULL, argv[i] = foreign_di_conv.iv[1];
    return i;
}

static int
AUmarg_to_Carg (Word ax, int i, int argv[], FPType ptype, const char *ptrs[])
{
    Word x = Dereference (ax);

    switch (Type_of (x)) {
    case AUm_Fixnum:
	if (ptype != FP_INTEGER)
	    goto error_occurred;
	ptrs[i] = NULL;
	argv[i] = Fix2Int (x);
	break;
    case AUm_Object:
	switch (Pointer (x)->Type ()) {
	case ASCII_STR:
	    if (ptype != FP_STRING)
		goto error_occurred;
	    ptrs[i] = ASCII_StrObject_ptr (x)->new_C_string ();
	    argv[i] = (int) ptrs[i];
	    break;
	case EUC_STR:
	    if (ptype != FP_STRING)
		goto error_occurred;
	    ptrs[i] = EUC_StrObject_ptr (x)->new_C_string ();
	    argv[i] = (int) ptrs[i];
	    break;
	case DFLOAT:
	    if (ptype != FP_DOUBLE)
		goto error_occurred;
	    {
		double d = Double2double (x);
		i = AUmdouble_to_Cdouble (d, i, argv, ptrs);
	    }
	    break;
	case FOREIGN_OBJ:
	    if (ptype != FP_FOREIGN)
		goto error_occurred;
	    ptrs[i] = NULL;
	    argv[i] = (Word) ForeignObject_ptr (x)->Address ();
	    break;
	default:
	    goto error_occurred;
	}
	break;
    default:
	if (IsSFloat (x)) {
	    if (ptype != FP_DOUBLE)
		goto error_occurred;
	    double s = (double) Single2float (x);
	    i = AUmdouble_to_Cdouble (s, i, argv, ptrs);
	}
	else
	    goto error_occurred;
	break;
    }
    return i + 1;

 error_occurred:
    error (TAG, "unmatched parameter type of %s", print (ax));
    return 0;
}

static void
foreign_string_memory_free (int argc, const char* argv[])
{
    for (int i = 0; i < argc; i++) {
	const char *p = argv[i];
	if (p)
	    SHARED_FREE ((char *) p, strlen (p) + 1);
    }
}

static int
AUmargs_to_Cargs (int nargs, int argv[],
		  const FPType ptypes[], int &retg, const char* ptrs[])
{
    register RegNumber* v = FetchSkip((nargs + 1 + 3) / 4);

    retg = *v++;
    int i = 0;
    for (int n = 0; n < nargs; n++) {
	i = AUmarg_to_Carg (Reg[*v++], i, argv, ptypes[n], ptrs);
    }
    return i;
}

static int
FuncCall4 (const ForeignNameList* np, int argc, int argv[])
{
    FLI_i fp = (FLI_i) np->FunctionAddress ();
    switch (argc) {
    case 0:
	return (*fp) ();
    case 1:
	return (*fp) (argv[0]);
    case 2:
	return (*fp) (argv[0], argv[1]);
    case 3:
	return (*fp) (argv[0], argv[1], argv[2]);
    case 4:
	return (*fp) (argv[0], argv[1], argv[2], argv[3]);
    case 5:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4]);
    case 6:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5]);
    case 7:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6]);
    case 8:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7]);
    case 9:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8]);
    case 10:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9]);
    case 11:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9],
		      argv[10]);
    case 12:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9],
		      argv[10], argv[11]);
    case 13:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9],
		      argv[10], argv[11], argv[12]);
    case 14:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9],
		      argv[10], argv[11], argv[12], argv[13]);
    case 15:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9],
		      argv[10], argv[11], argv[12], argv[13], argv[14]);
    case 16:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9],
		      argv[10], argv[11], argv[12], argv[13], argv[14],
		      argv[15]);
    default:
	abort ();
    }
    return 0;
}

static double
FuncCall8 (const ForeignNameList* np, int argc, int argv[])
{
    FLI_d fp = (FLI_d) np->FunctionAddress ();
    switch (argc) {
    case 0:
	return (*fp) ();
    case 1:
	return (*fp) (argv[0]);
    case 2:
	return (*fp) (argv[0], argv[1]);
    case 3:
	return (*fp) (argv[0], argv[1], argv[2]);
    case 4:
	return (*fp) (argv[0], argv[1], argv[2], argv[3]);
    case 5:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4]);
    case 6:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5]);
    case 7:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6]);
    case 8:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7]);
    case 9:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8]);
    case 10:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9]);
    case 11:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9],
		      argv[10]);
    case 12:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9],
		      argv[10], argv[11]);
    case 13:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9],
		      argv[10], argv[11], argv[12]);
    case 14:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9],
		      argv[10], argv[11], argv[12], argv[13]);
    case 15:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9],
		      argv[10], argv[11], argv[12], argv[13], argv[14]);
    case 16:
	return (*fp) (argv[0], argv[1], argv[2], argv[3], argv[4],
		      argv[5], argv[6], argv[7], argv[8], argv[9],
		      argv[10], argv[11], argv[12], argv[13], argv[14],
		      argv[15]);
    default:
	abort ();
    }
    return 0;
}

static const char* current_fcn;

void
fatal_in_foreign_call (int sig, int code,
		       const struct sigcontext* scp, const char* addr)
{
    error ("foreign_call",
	   "Catch %d signal in during to call %s", sig, current_fcn);
    exit (1);
}

static Word
foreign_call (const ForeignNameList* np, int argc, int argv[])
{
    Word retw;
    current_fcn = np->Name ();
    SignalHandler sigquit = signal (SIGQUIT, fatal_in_foreign_call);
    SignalHandler sigill = signal (SIGILL, fatal_in_foreign_call);
    SignalHandler sigiot = signal (SIGIOT, fatal_in_foreign_call);
    SignalHandler sigfpe = signal (SIGFPE, fatal_in_foreign_call);
    SignalHandler sigbus = signal (SIGBUS, fatal_in_foreign_call);
    SignalHandler sigsegv = signal (SIGSEGV, fatal_in_foreign_call);

    switch (np->ReturnType ()) {
    case FP_VOID:
	(void) FuncCall4 (np, argc, argv);
	retw = INT0;
	break;
    case FP_INTEGER:
	retw = Int2Fix (FuncCall4 (np, argc, argv));
	break;
    case FP_DOUBLE:
	retw = CreateDFloat (FuncCall8 (np, argc, argv));
	break;
    case FP_FOREIGN:
	retw = CreateForeignObject ((int*)FuncCall4 (np, argc, argv), "foreign");
	break;
    case FP_STRING:
	retw = CreateASCIIString ((const char*) FuncCall4 (np, argc, argv));
	break;
    default:
	error (TAG, "unsupported return value type");
	retw = INT0;
	break;
    }
    signal (SIGQUIT, sigquit);
    signal (SIGILL, sigill);
    signal (SIGIOT, sigiot);
    signal (SIGFPE, sigfpe);
    signal (SIGBUS, sigbus);
    signal (SIGSEGV, sigsegv);
    current_fcn = NULL;
    return retw;
}

static void
doForeignCall (int index, int nargs)
{
    const ForeignNameList *np = CurrentClass->ForeignInfo (index);
    assert (np->Arity () == nargs);
    /* dump_foreign_namelist (np); */
    int	  argv[FOREIGN_MAXARGS];
    const char* ptrs[FOREIGN_MAXARGS];
    int retg;
    int argc = AUmargs_to_Cargs (nargs, argv, np->ParaTypes (), retg, ptrs);
    Reg[retg] = foreign_call (np, argc, argv);
    foreign_string_memory_free (argc, ptrs);
}

METHOD (foreign_call, RnAf_OP)
    // {}
    // foreign_call Ix, R0, R1, ..., Rn
    // {}
    // [	   address ]
    // [       Ix|	 n ]
    // [  R0|  R1| ...	   ]
    // {}
{
    Fetch22 ();
    InstructionPointer += sizeof (*ip) / sizeof (Instruction);
    // ip->s1 - 1 ==> minus Rreturn arg from nargs
    doForeignCall (ip->s0, ip->s1 - 1);
    DO_TRACE_INSTRUCTION;
    JUMPNEXT;
}

/*-----------------
 * Local Variables:
 * c-indent-level:4
 * c-continued-statement-offset:4
 * c-brace-offset:0
 * c-imaginary-offset:0
 * c-argdecl-indent:4
 * c-label-offset:-4
 * c++-electric-colon:t
 * c++-empty-arglist-indent:nil
 * c++-friend-offset:-4
 * c++-member-init-indent-offset:0
 * c++-continued-member-init-offset:nil
 * End:
 */
