/*
SGPC: Simple Genetic Programming in C
(c) 1993 by Walter Alden Tackett and Aviram Carmi
 
 This code and documentation is copyrighted and is not in the public domain.
 All rights reserved. 
 
 - This notice may not be removed or altered.
 
 - You may not try to make money by distributing the package or by using the
   process that the code creates.
 
 - You may not distribute modified versions without clearly documenting your
   changes and notifying the principal author.
 
 - The origin of this software must not be misrepresented, either by
   explicit claim or by omission.  Since few users ever read sources,
   credits must appear in the documentation.
 
 - Altered versions must be plainly marked as such, and must not be
   misrepresented as being the original software.  Since few users ever read
   sources, credits must appear in the documentation.
 
 - The authors are not responsible for the consequences of use of this 
   software, no matter how awful, even if they arise from flaws in it.
 
If you make changes to the code, or have suggestions for changes,
let us know!  (gpc@ipld01.hac.com)
*/

#ifndef lint
static char gpc_c_rcsid[]="$Id: gpc.c,v 2.13 1993/04/22 07:39:12 gpc-avc Exp gpc-avc $";
#endif

/*
 *
 * $Log: gpc.c,v $
 * Revision 2.13  1993/04/22  07:39:12  gpc-avc
 * Removed old log messages
 *
 * Revision 2.12  1993/04/15  10:35:55  gpc-avc
 * Added optional call to times() (#ifdef USE_TIMES) for non-Sun machines
 *
 *
 */

#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>

#define _MAIN_
#include "gpc.h"
#include "random.h"
#undef _MAIN_

#ifdef ANSI_FUNC

int main(
  int	argc,
  char	**argv,
  char	**envp
)
#else

int main(argc, argv, envp)
  int	argc;
  char	**argv;
  char  **envp;
#endif
{

#ifdef MALLOPTS
  mallopt(M_MXFAST,64);   /* mem blocks <64 will be alloced in chunks */
  mallopt(M_NLBLKS,1024); /* number of blocks in a chunk */
#endif

  startup(argc,argv,&NUMPOPS,&NUMGENS,&DEMES,&DEMEROWS,&DEMECOLS,&START_GEN,
	  &CKPT_FILE,&POP,&GRID); 
  run_gp_system(NUMPOPS,NUMGENS,START_GEN,POP,DEMES,GRID,DEMEROWS,DEMECOLS);

  exit(0);
}

#ifdef ANSI_FUNC

VOID startup(
  int 		argc,
  char 		**argv,
  int		*numpops,
  int 		*numgens,
  int		*demes,
  int		*demerows,
  int		*demecols,	       
  int		*start_gen,
  FILE		**ckpt_file,
  pop_struct 	**pop,
  pop_struct    ****grid
  )
#else

VOID startup(argc,argv,numpops,numgens,demes,demerows,demecols,start_gen,
	     ckpt_file,pop,grid)
  int 		argc;
  char 		**argv;
  int 		*numpops;
  int 		*numgens;
  int		*demes;
  int		*demerows;
  int		*demecols;	       
  int		*start_gen;
  FILE		**ckpt_file;
  pop_struct 	**pop;
  pop_struct    ****grid;
#endif
{
  int p;
  int popnum;
  int i; 
  char	buf[132], val[132];

  setbuffer(stdout,'\0',0);
  setbuffer(stderr,'\0',0);
  gaussian_noise_toggle = 1;
  getparams(argc,argv,numpops,numgens,demes,demerows,demecols,start_gen,
	    ckpt_file,pop,grid);
  make_function_table(*numpops, *pop);
  make_terminal_table(*numpops, *pop);
  allocate_populations(*numpops,*pop); 
  if (*demes) {
    setup_deme_grid(*numpops,*demerows,*demecols,*pop,*grid);
  }		  
  define_fitness_cases(*numpops,*numgens,*pop);
  if(*start_gen) {

    (*start_gen)++; /* restart from next gen after crash */
    for (p = 0; p < *numpops; p++) {
      popnum = -1;
      fscanf(*ckpt_file,"%s = %s", buf, val);
      if (strcasecmp("_START_OF_POPULATION_",buf)) {
	fprintf(stderr,"Error, Should have seen _START_OF_POPULATION_ = %d\n",
		p);
	exit(1);
      }
      sscanf(val,"%d",&popnum);
      if (popnum != p) {
	fprintf(stderr,"Error reading ckpt file: Population mismatch %d %d\n",
		p,popnum);
	exit(1);
      }
      printf("Reading population %d from checkpoint file\n", popnum);
      for (i = 0; i < (*pop)[p].population_size; i++) {
	(*pop)[p].population[i] = read_tree(*pop, p, *ckpt_file);
      }
    }

    for (p = 0; p < *numpops; p++) {

      for (i = 0; i < (*pop)[p].population_size; i++) {
	(*pop)[p].fitness_sort_index[i] = i;
      }

      zero_fitness_of_populations(*numpops,*pop,p);
      evaluate_fitness_of_populations(*numpops,*numgens,*pop,p);
      if ((*pop)[p].parsimony_factor > 0.0) add_parsimony_to_fitness(*pop,p);

      normalize_fitness_of_population(*pop,p);

      sort_population_by_fitness(*pop,p);

      fscanf(*ckpt_file,"%s = %s", buf, val);
      if (!strcmp("best_of_run_fitness",buf)) {
	sscanf(val,"%f",&((*pop)[p].best_of_run_fitness));
      }
      else {
	fprintf(stderr,"Error reading checkpoint file.\n");
      }
      fscanf(*ckpt_file,"%s = %s", buf, val);
      if (!strcmp("best_of_run_gen",buf)) {
	sscanf(val,"%d",&((*pop)[p].best_of_run_gen));
      }
      else {
	fprintf(stderr,"Error reading checkpoint file.\n");
      }
      (*pop)[p].best_of_run = read_tree(*pop, p, *ckpt_file);

    }

    fclose(*ckpt_file);
  }
}

#if defined(sun) && !defined(USE_TIMES)
#include <sys/time.h>
#include <sys/resource.h>
#elif defined(USE_TIMES)
#include <sys/types.h>
#include <sys/times.h>
#endif

#ifdef ANSI_FUNC

VOID run_gp_system(
  int		numpops,
  int  		numgens,
  int		start_gen,
  pop_struct 	*pop,
  int		demes,
  pop_struct	***grid,
  int		demerows,
  int		demecols
  )
#else

VOID run_gp_system(numpops,numgens,start_gen,pop,demes,grid,demerows,demecols)
  int		numpops;
  int		numgens;
  int		start_gen;
  pop_struct 	*pop;
  int		demes;
  pop_struct	***grid;
  int		demerows;
  int		demecols;
#endif
{

#if defined(sun) && !defined(USE_TIMES)
  int getrusage P_((int who, struct rusage *rusage));
  struct rusage usage;
  long sec, usec;
#elif defined(USE_TIMES)
  struct tms ticks;
#define HZ 60.0
#else
  long clocktime, clock P_((void));

  clock(); 
#endif

  generations(numpops,numgens,start_gen,pop,demes,grid,demerows,demecols);

#if defined(sun) && !defined(USE_TIMES)
  /*** NOTE *** getrusage is very consistent with the shell time command ***/
  getrusage(RUSAGE_SELF, &usage);
  sec = usage.ru_utime.tv_sec+usage.ru_stime.tv_sec;
  if ((usec = usage.ru_utime.tv_usec+usage.ru_stime.tv_usec) >= 1000000) {
    sec++;
    usec-=1000000;
  }
  printf("\ntime= %ld.%06ld seconds\n",sec,usec);

#elif defined(USE_TIMES)
  /*** NOTE *** times() has granularity of only 1/HZ sec where HZ is 60 ***/
  if(times(&ticks)) fprintf(stderr,"times error\n"); 
  printf("time= %f seconds\n",(float)(ticks.tms_utime+ticks.tms_stime)/HZ);

#else
  /*** NOTE *** clock() rolls over after about 36 minutes ***/
  clocktime = clock();
  printf("\ntime= %f seconds\n", ((double)clocktime)/1.0e6);
#endif

  report_on_run(numpops,pop);
}
