/*
SGPC: Simple Genetic Programming in C
(c) 1993 by Walter Alden Tackett and Aviram Carmi
 
 This code and documentation is copyrighted and is not in the public domain.
 All rights reserved. 
 
 - This notice may not be removed or altered.
 
 - You may not try to make money by distributing the package or by using the
   process that the code creates.
 
 - You may not distribute modified versions without clearly documenting your
   changes and notifying the principal author.
 
 - The origin of this software must not be misrepresented, either by
   explicit claim or by omission.  Since few users ever read sources,
   credits must appear in the documentation.
 
 - Altered versions must be plainly marked as such, and must not be
   misrepresented as being the original software.  Since few users ever read
   sources, credits must appear in the documentation.
 
 - The authors are not responsible for the consequences of use of this 
   software, no matter how awful, even if they arise from flaws in it.
 
If you make changes to the code, or have suggestions for changes,
let us know!  (gpc@ipld01.hac.com)
*/

#ifndef lint
static char getparams_c_rcsid[]="$Id: getparams.c,v 2.18 1993/04/23 01:56:25 gpc-avc Exp gpc-avc $";
#endif

/*
 *
 * $Log: getparams.c,v $
 * Revision 2.18  1993/04/23  01:56:25  gpc-avc
 * Added COMPRESS macro so that the compression command for checkpoint
 * files is easily changed in Makefile
 *
 * Revision 2.17  1993/04/22  07:39:12  gpc-avc
 * Removed old log messages
 *
 * Revision 2.16  1993/04/17  03:03:01  gpc-avc
 * Added a test to check if the load_from_file file exists
 *
 *
 */

#include <stdio.h>
#include <malloc.h>
#include <errno.h>
#include "gpc.h"
#include "malloc4.h"

#ifdef ANSI_FUNC

VOID getparams(
  int		argc,
  char		**argv,
  int 		*numpops,
  int		*numgens,
  int		*demes,
  int		*demerows,
  int		*demecols,	       
  int 		*start_gen,
  FILE		**ckpt_file,
  pop_struct 	**pop,
  pop_struct    ****grid
  )
#else

VOID getparams(argc,argv,numpops,numgens,demes,demerows,demecols,start_gen,
	       ckpt_file,pop,grid)
  int		argc;
  char		**argv;
  int 		*numpops;
  int 		*numgens;
  int		*demes;
  int		*demerows;
  int		*demecols;	       
  int 		*start_gen;
  FILE		**ckpt_file;
  pop_struct 	**pop;
  pop_struct    ****grid;
#endif
{
  int	i,j;
  int   arg_offset = 0;
  int   argp = 1;
  FILE  *paramfile;
  char ckptfile[255];
  static char usage[] = 
  "%s: Usage: %s [-d rows cols] npops ngens paramfile | 'none'... [seed]\n%s: Usage: %s -r ckpt_filename\n";


  if (argc < 2) {
    fprintf(stderr, usage, argv[0],argv[0],argv[0],argv[0]);
    exit(1);
  }

  if (!strcmp(argv[1],"-r")) {
    if (argc < 3) {
      fprintf(stderr, usage, argv[0],argv[0], argv[0],argv[0]);
      fprintf(stderr, "%s: Missing checkpoint file name.\n",argv[0]);
      exit(1);
    }
    if ((*ckpt_file = fopen(argv[2],"r")) != (FILE *)NULL) {
      recover_params (*ckpt_file,numpops,numgens,demes,demerows,demecols,pop,
		      grid,start_gen);
    }
    else {
      fprintf(stderr, "%s: Checkpoint file not found: '%s'.\n",
	      argv[0],argv[2]);
      exit(1);
    }
  }
  else {
    if (!strcmp(argv[1],"-d")) {
      if (argc < 4) {
	fprintf(stderr, usage, argv[0],argv[0], argv[0],argv[0]);
	fprintf(stderr, "%s: Missing deme rows or cols.\n",argv[0]);
	exit(1);
      }
      if (argc < 7) {
	fprintf(stderr, usage,  argv[0],argv[0], argv[0],argv[0]);
	if (argc < 6) {
	  fprintf(stderr, "%s: Missing number of populations or generations.\n",
		  argv[0]);
	}
	else {
	  fprintf(stderr, "%s: Missing parameter file name.\n",argv[0]);
	  fprintf(stderr,
		  "%s: Use 'none' as the file name in order to use defaults.\n",
		  argv[0]);
	}
	exit(1);
      }
      *demes = TRUE;
      arg_offset += 3;
      argp++;
    }
    else {
      if (argc < 4) {
	fprintf(stderr, usage, argv[0],argv[0], argv[0],argv[0]);
	if (argc < 3) {
	  fprintf(stderr, "%s: Missing number of populations or generations.\n",
		  argv[0]);
	}
	else {
	  fprintf(stderr,	"%s: Missing parameter file name.\n",argv[0]);
	  fprintf(stderr,
		  "%s: Use 'none' as the file name in order to use defaults.\n",
		  argv[0]);
	}
	exit(1);
      }
      *demes = FALSE;
      *demerows = 0;
      *demecols = 0;
    }
    
    if (*demes) {
      *demerows = -1;
      sscanf(argv[argp++],"%d",demerows);
      if (*demerows < 1) {
	fprintf(stderr, "%s: Deme rows has to be greater than 1: '%s'.\n",
		argv[0], argv[argp-1]);
	exit(1);
      }
      
      *demecols = -1;
      sscanf(argv[argp++],"%d",demecols);
      if (*demecols < 1) {
	fprintf(stderr, "%s: Deme cols has to be greater than 1: '%s'.\n",
		argv[0], argv[argp-1]);
	exit(1);
      }
    }
    
    *numpops = -1;
    sscanf(argv[argp++],"%d",numpops);
    if (*numpops < 1) {
      fprintf(stderr, 
	      "%s: Number of populations has to be greater than 1: '%s'.\n",
	      argv[0], argv[argp-1]);
      exit(1);
    }
    
    *numgens = -1;
    sscanf(argv[argp++],"%d",numgens);
    if (*numgens < 1) {
      fprintf(stderr, 
	      "%s: Number of generations has to be greater than 1: '%s'.\n",
	      argv[0], argv[argp-1]);
      exit(1);
    }
    
    *pop = (pop_struct *) malloc(*numpops*sizeof(pop_struct));
    if (*demes) {
      *grid = (pop_struct ***) malloc2(*demerows,*demecols*sizeof(pop_struct *));
      for (i=0;i<*demerows;i++) {
	for (j=0;j<*demecols;j++) {
	  (*grid)[i][j] = (pop_struct *) malloc(*numpops*sizeof(pop_struct));
	}
      }
    }
    
    if (argc < *numpops+3+arg_offset) {
      fprintf(stderr, "%s: Missing parameter file name.\n", argv[0]);
      fprintf(stderr,
	      "%s: Use 'none' as the file name in order to use defaults.\n",
	      argv[0]);
      exit(1);
    }
    
    for(i=0;i<*numpops;i++,argp++) {
      defaultparams(*pop,i);
      if (strcasecmp(argv[argp],"none")) {
	if ((paramfile = fopen(argv[argp],"r")) != (FILE *)NULL) {
	  readparams(*pop,i,paramfile);
	  fclose(paramfile);
	}
	else {
	  fprintf(stderr, "%s: Parameter file not found: '%s'.\n",
		  argv[0],argv[argp]);
	  exit(1);
	}
      }
    }
    
    
    if (argc == (j=*numpops+4+arg_offset)) {
      long seed = -1;
      sscanf(argv[j-1],"%d",&seed);
      if (seed < 0) {
	fprintf(stderr, "%s: Bad seed value: '%s'.\n", argv[0], argv[j-1]);
	exit(1);
      }
      set_seed((unsigned long)seed);
    }
    if (argc > (j=*numpops+4+arg_offset)) {
      fprintf(stderr, usage, argv[0],argv[0], argv[0],argv[0]);
      fprintf(stderr, "%s: Too many arguments on command line.\n",argv[0]);
      exit(1);
    }
  }

  if (*demes) {
    for(i=0;i<*numpops;i++) {
      if ((*pop)[i].population_size%((*demerows) * (*demecols))) {
	fprintf(stderr,
		"population %d size (%d) must be evenly divisible by rows*cols (%d)\n",
		i,(*pop)[i].population_size, ((*demerows) * (*demecols)));
	exit(1);
      }
    }
  }
  
  for (i=0;i<argc;i++) {
    printf("%s ",argv[i]);
  }
  printf("\n\n");
  printf("Random seed:            %ld\n",(unsigned long)get_seed());
  printf("Number of populations:  %d\n",*numpops);
  printf("Number of generations:  %d\n",*numgens);
  if (*demes) {
    printf("Number of deme rows:    %d\n",*demerows);
    printf("Number of deme columns: %d\n",*demecols);
  }
  if (CHECKPOINT_FREQUENCY) {
    sprintf(ckptfile,"gpc_%ld_%d.ckpt",gethostid(),getpid());
    printf("Checkpointing to:       %s\n",ckptfile);
  }
  if (*start_gen) {
    printf("Restarting from gen:    %d\n", *start_gen+1);
  }
  printf("\n");

  argp = arg_offset+3;
  for(i=0;i<*numpops;i++,argp++) {
    if (*start_gen) {
      printf("Reading parameters for population %d from: %s\n\n",
	     i,argv[2]);
    }
    else if (strcasecmp(argv[argp],"none")) {
      printf("Reading parameters for population %d from: %s\n\n",
	     i,argv[argp]);
    }
    else {
      printf("Using default parameters for population %d\n\n",i);
    }
    writeparams(*pop,i,stdout);
    printf("\n");
  }

}


#ifdef ANSI_FUNC

VOID defaultparams(
  pop_struct 	*pop,
  int		i
  )
#else

VOID defaultparams(pop,i)
  pop_struct 	*pop;
  int		i;
#endif
{
  FILE	*paramfile;

  set_seed((unsigned long)11287);
  CHECKPOINT_FREQUENCY			= 0;
  START_GEN				= 0;

  pop[i].population_size 		= 100;
  pop[i].steady_state			= FALSE;
  pop[i].load_from_file 		= (char *) malloc(132*sizeof(char));
  pop[i].load_from_file[0] 		= '\0';
  pop[i].max_depth_for_new_trees 	= 6;
  pop[i].max_depth_after_crossover 	= 17; 
  pop[i].max_mutant_depth 		= 4;
  pop[i].grow_method 			= GROW;
  pop[i].selection_method 		= FITNESSPROP;
  pop[i].tournament_K 			= 7;
  pop[i].deme_search_radius_sigma	= 1.0;
  pop[i].crossover_func_pt_fraction 	= 0.695;
  pop[i].crossover_any_pt_fraction 	= 0.195;
  pop[i].fitness_prop_repro_fraction	= 0.095;
  pop[i].parsimony_factor 		= 0.0;

  if ((paramfile = fopen("default.in","r")) != (FILE *)NULL) {
    readparams(pop,i,paramfile);
    fclose(paramfile);
  }
}

#ifdef ANSI_FUNC

VOID readparams(
  pop_struct 	*pop,
  int		i,
  FILE 		*f
  )
#else

VOID readparams(pop,i,f)
  pop_struct 	*pop;
  int		i;
  FILE		*f;
#endif
{
  char	buf[132], name[132], val[132];
  unsigned long seed;
  FILE *tmp;

  while (fgets(buf,132,f) != (char *)NULL) {
    if (buf[0] == '#') continue;
    sscanf(buf,"%s = %s", name, val);
    if (!strcasecmp("seed",name)) {
      sscanf(val,"%ld",&seed);
      set_seed(seed);
    } else if (!strcasecmp("steady_state",name)) {
      sscanf(val,"%d",&(pop[i].steady_state));
    } else if (!strcasecmp("population_size",name)) {
      sscanf(val,"%d",&(pop[i].population_size));
    } else if (!strcasecmp("load_from_file",name)) {
      sscanf(val,"%s",  pop[i].load_from_file);
      if ((tmp=fopen(pop[i].load_from_file,"r")) == (FILE *)NULL) {
	fprintf(stderr,"Error: load_from_file not found: %s\n",
		pop[i].load_from_file);
	exit(1);
      }
      else {
	fclose(tmp);
      }
    } else if (!strcasecmp("checkpoint_frequency",name)) {
      sscanf(val,"%d",&(CHECKPOINT_FREQUENCY));
    } else if (!strcasecmp("max_depth_for_new_trees",name)) {
      sscanf(val,"%d",&(pop[i].max_depth_for_new_trees));
    } else if (!strcasecmp("max_depth_after_crossover",name)) {
      sscanf(val,"%d",&(pop[i].max_depth_after_crossover));
    } else if (!strcasecmp("max_mutant_depth",name)) {
      sscanf(val,"%d",&(pop[i].max_mutant_depth));
    } else if (!strcasecmp("grow_method",name)) {
      if (!strcasecmp(val,"FULL")) {
	pop[i].grow_method = FULL;
      } else if (!strcasecmp(val,"GROW")) {
	pop[i].grow_method = GROW;
      } else if (!strcasecmp(val,"RAMPED")) {
	pop[i].grow_method = RAMPED;
      } else {
	fprintf(stderr,"readparams(): bad grow_method val name %s\n",val);
      }
    } else if (!strcasecmp("selection_method",name)) {
      if (!strcasecmp(val,"TOURNAMENT")) {
	pop[i].selection_method = TOURNAMENT;
      } else if (!strcasecmp(val,"OVERSELECT")) {
	pop[i].selection_method = OVERSELECT;
      } else if (!strcasecmp(val,"FITNESSPROP")) {
	pop[i].selection_method = FITNESSPROP;
      } else {
	fprintf(stderr,"readparams(): bad selection_method val name %s\n",val);
      }
    } else if (!strcasecmp("tournament_K",name)) {
      sscanf(val,"%d",&(pop[i].tournament_K));
    } else if (!strcasecmp("deme_search_radius_sigma",name)) {
      sscanf(val,"%f",&(pop[i].deme_search_radius_sigma));
    } else if (!strcasecmp("crossover_func_pt_fraction",name)) {
      sscanf(val,"%f",&(pop[i].crossover_func_pt_fraction));
    } else if (!strcasecmp("crossover_any_pt_fraction",name)) {
      sscanf(val,"%f",&(pop[i].crossover_any_pt_fraction));
    } else if (!strcasecmp("fitness_prop_repro_fraction",name)) {
      sscanf(val,"%f",&(pop[i].fitness_prop_repro_fraction));
    } else if (!strcasecmp("parsimony_factor",name)) {
      sscanf(val,"%f",&(pop[i].parsimony_factor));
    } else if (!strcmp("_END_OF_PARAMETERS_POPULATION",name)) {
      int popnum;
      sscanf(val,"%d",&popnum);
      if (popnum != i) {
	fprintf(stderr,"Warning: readparams() possible population mismatch\n");
      }
      break;
    } else {
      fprintf(stderr,"readparams(): bad param name %s\n",name);
    }
  }
}

#ifdef ANSI_FUNC

VOID writeparams(
  pop_struct 	*pop,
  int		p,
  FILE 		*f
  )
#else

VOID writeparams(pop,p,f)
  pop_struct 	*pop;
  int		p;
  FILE		*f;
#endif
{
  fprintf(f, "seed = %ld\n",                    get_seed());
  fprintf(f,"checkpoint_frequency = %d\n",	CHECKPOINT_FREQUENCY);
  fprintf(f,"population_size = %d\n",		pop[p].population_size);
  fprintf(f,"steady_state = %d\n",		pop[p].steady_state);
  if (pop[p].load_from_file[0] != '\0') {
    fprintf(f,"load_from_file = %s\n", 		pop[p].load_from_file);
  }
  fprintf(f,"max_depth_for_new_trees = %d\n",	pop[p].max_depth_for_new_trees);
  fprintf(f,"max_depth_after_crossover = %d\n",	pop[p].max_depth_after_crossover);
  fprintf(f,"max_mutant_depth = %d\n",		pop[p].max_mutant_depth);
  fprintf(f,"grow_method = ");
  switch(pop[p].grow_method) {
  case FULL:
    fprintf(f,"FULL\n");
    break;
  case GROW:
    fprintf(f,"GROW\n");
    break;
  case RAMPED:
    fprintf(f,"RAMPED\n");
    break;
  }
  fprintf(f,"selection_method = ");
  switch(pop[p].selection_method) {
  case TOURNAMENT:
    fprintf(f,"TOURNAMENT\n");
    break;
  case OVERSELECT:
    fprintf(f,"OVERSELECT\n");
    break;
  case FITNESSPROP:
    fprintf(f,"FITNESSPROP\n");
  }
  fprintf(f,"tournament_K = %d\n",
	  pop[p].tournament_K);
  fprintf(f,"deme_search_radius_sigma = %f\n",
	  pop[p].deme_search_radius_sigma);
  fprintf(f,"crossover_func_pt_fraction = %f\n",
	  pop[p].crossover_func_pt_fraction);
  fprintf(f,"crossover_any_pt_fraction = %f\n",
	  pop[p].crossover_any_pt_fraction);
  fprintf(f,"fitness_prop_repro_fraction = %f\n",
	  pop[p].fitness_prop_repro_fraction);
  fprintf(f,"parsimony_factor = %f\n",
	  pop[p].parsimony_factor);
}


#ifdef ANSI_FUNC
void checkpoint (
  int           numpops,
  int           numgens,
  int           demes,
  int           demerows,
  int           demecols,
  pop_struct 	*pop,
  int		g
  )
#else
VOID checkpoint(numpops,numgens,demes,demerows,demecols,pop,g)
  int           numpops;
  int           numgens;
  int           demes;
  int           demerows;
  int           demecols;
  pop_struct	*pop;
  int		g;
#endif
{
  int	i, p;
  char  ckptfile[256], cmd_buf[256];
  FILE	*cf;

  sprintf(ckptfile,"gpc_%ld_%d.ckpt",gethostid(),getpid());
  
  cf = fopen(ckptfile,"w");

  fprintf(cf, "checkpoint_for_generation =  %d\n", g);
  fprintf(cf, "random_seed = %ld\n",get_seed());
  fprintf(cf, "number_of_populations = %d\n",numpops);
  fprintf(cf, "number_of_generations = %d\n",numgens);
  fprintf(cf, "demes = %d\n",demes);
  fprintf(cf, "number_of_deme_rows = %d\n",demerows);
  fprintf(cf, "number_of_deme_columns = %d\n",demecols);
  fprintf(cf, "gaussian_noise_uniform1 = %22.20f\n",gaussian_noise_uniform1);
  fprintf(cf, "gaussian_noise_uniform2 = %22.20f\n",gaussian_noise_uniform2);
  fprintf(cf, "gaussian_noise_temp = %22.20f\n",gaussian_noise_temp);
  fprintf(cf, "gaussian_noise_toggle = %d\n",gaussian_noise_toggle);

  fprintf(cf,"\n");

  for (p=0; p<numpops; p++) {
    writeparams(pop, p, cf);
    fprintf(cf,"_END_OF_PARAMETERS_POPULATION = %d\n",p);
  }

  for (p=0; p<numpops; p++) {
    fprintf(cf,"_START_OF_POPULATION_ = %d\n",p);
    for (i=0; i<pop[p].population_size; i++) {
      write_tree(pop, pop[p].population[i], pop[p].ckpt_format, cf);
    }
  }

  for (p=0; p<numpops; p++) {
    fprintf(cf, "best_of_run_fitness = %22.20f\n",pop[p].best_of_run_fitness);
    fprintf(cf, "best_of_run_gen = %d\n",pop[p].best_of_run_gen);
    write_tree(pop, pop[p].best_of_run, pop[p].ckpt_format, cf);
  }

  fclose(cf);
  sprintf(cmd_buf,"%s %s &",COMPRESS,ckptfile);
  system(cmd_buf);
}


#ifdef ANSI_FUNC
void recover_params (
  FILE		*cf,
  int           *numpops,
  int           *numgens,
  int           *demes,
  int           *demerows,
  int           *demecols,
  pop_struct	**pop,
  pop_struct    ****grid,
  int		*g
  )
#else
VOID recover_params (cf,numpops,numgens,demes,demerows,demecols,pop,grid,g)
  FILE		*cf;
  int           *numpops;
  int           *numgens;
  int           *demes;
  int           *demerows;
  int           *demecols;
  pop_struct	**pop;
  pop_struct    ****grid;
  int		*g;
#endif
{
  int 	p;
  int	i, j;
  unsigned long 	seed;
  char	buf[132], name[132], val[132];

  fscanf(cf,"%s = %s", name, val);
  if (!strcmp("checkpoint_for_generation",name)) { 
    sscanf(val, "%d", g);
  }
  else {
    fprintf(stderr,"Error reading checkpoint file.\n");
  }
  fscanf(cf,"%s = %s", name, val);
  if (!strcmp("random_seed",name)) { 
    sscanf(val, "%ld",&seed);
  }
  else {
    fprintf(stderr,"Error reading checkpoint file.\n");
  }
  fscanf(cf,"%s = %s", name, val);
  if (!strcmp("number_of_populations",name)) { 
    sscanf(val, "%d",numpops);
  }
  else {
    fprintf(stderr,"Error reading checkpoint file.\n");
  }
  fscanf(cf,"%s = %s", name, val);
  if (!strcmp("number_of_generations",name)) { 
    sscanf(val, "%d",numgens);
  }
  else {
    fprintf(stderr,"Error reading checkpoint file.\n");
  }
  fscanf(cf,"%s = %s", name, val);
  if (!strcmp("demes",name)) { 
    sscanf(val, "%d",demes);
  }
  else {
    fprintf(stderr,"Error reading checkpoint file.\n");
  }
  fscanf(cf,"%s = %s", name, val);
  if (!strcmp("number_of_deme_rows",name)) { 
    sscanf(val, "%d",demerows);
  }
  else {
    fprintf(stderr,"Error reading checkpoint file.\n");
  }
  fscanf(cf,"%s = %s", name, val);
  if (!strcmp("number_of_deme_columns",name)) { 
    sscanf(val, "%d",demecols);
  }
  else {
    fprintf(stderr,"Error reading checkpoint file.\n");
  }
  fscanf(cf,"%s = %s", name, val);
  if (!strcmp("gaussian_noise_uniform1",name)) { 
    sscanf(val, "%f",&gaussian_noise_uniform1);
  }
  else {
    fprintf(stderr,"Error reading checkpoint file.\n");
  }
  fscanf(cf,"%s = %s", name, val);
  if (!strcmp("gaussian_noise_uniform2",name)) { 
    sscanf(val, "%f",&gaussian_noise_uniform2);
  }
  else {
    fprintf(stderr,"Error reading checkpoint file.\n");
  }
  fscanf(cf,"%s = %s", name, val);
  if (!strcmp("gaussian_noise_temp",name)) { 
    sscanf(val, "%f",&gaussian_noise_temp);
  }
  else {
    fprintf(stderr,"Error reading checkpoint file.\n");
  }
  fscanf(cf,"%s = %s", name, val);
  if (!strcmp("gaussian_noise_toggle",name)) { 
    sscanf(val, "%d",&gaussian_noise_toggle);
  }
  else {
    fprintf(stderr,"Error reading checkpoint file.\n");
  }

  fscanf(cf,"\n");
  
  *pop = (pop_struct *) malloc(*numpops*sizeof(pop_struct));
  if (*demes) {
    *grid = (pop_struct ***) malloc2(*demerows,*demecols*sizeof(pop_struct *));
    for (i=0;i<*demerows;i++) {
      for (j=0;j<*demecols;j++) {
	(*grid)[i][j] = (pop_struct *) malloc(*numpops*sizeof(pop_struct));
      }
    }
  }

  for (p = 0; p < *numpops; p++) {
    (*pop)[p].load_from_file = (char *) malloc(132*sizeof(char));
    readparams(*pop, p, cf);
    (*pop)[p].load_from_file[0] = '\0'; /* don't load from file upon recovery */
  }

  set_seed(seed);

}

