//
//  randGraph.c
//  
//
//  Created by Henry Leung on 5/6/12.
//  Copyright 2012 Carnegie Mellon University. All rights reserved.
//
// Generates an file with two graphs with equal number of vertices.
// Specify whether they are isomorphic by command line arguments.

#define MAX_DIST 1
#define EDGE_PROBABILITY 0.9

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/time.h>
#include <omp.h>
#include <string.h>

//General linked list use
typedef struct Node
{
    int value;
    struct Node* next;
    struct Node* prev;
} Node;

Node* getNode(Node* head, int index);
Node* deleteNode(Node* head, Node* toDelete);
void permute(int* adjB, int* adjA,int* P);
void transpose(int* result, int* adjA);
void randomSwap(int* adjA);

int numVertices;
int isomorphic = 0;
int i,j,k;
int* adjA;
int* adjB;
int* P;    
int* transP;
int* newAdjB;
int n;
int tid;
int blockSize;
int procs;

Node* head;
Node* cur;
int randIndex,vertexIndex,numRemaining;

int main(int argc, char *argv []){
    
    //------------------Command line argument parsing---------------
    int opt;
    while ((opt = getopt(argc, argv, "n:p:o:c:")) != -1)
    {
        switch (opt)
        {
            case 'n':
            {
                numVertices = atoi(optarg);
                if(numVertices <= 0){
                    printf("Number of vertices must be positive...\n");
                    exit(1);
                }
                break;
            }
            case 'o':
            {
                if (dup2(open(optarg, O_CREAT | O_WRONLY, 0644), STDOUT_FILENO) < 0)
                {
                    perror("");
                    exit(EXIT_FAILURE);
                }
                break;
            }
            case 'p':
            {
                if(atoi(optarg))
                    isomorphic = 1;
                break;
            }
            case 'c':
            {
                procs = atoi(optarg);
                break;
            }
            default:
                exit(1);
        }
    }
    
    printf("%d\n",numVertices);
    
    srand(time(0));
    
    //Generate random graph A
    adjA = (int *)malloc(sizeof(int) * numVertices * numVertices);
#pragma	omp	parallel default(shared) private(i,tid,j)
    {
        tid = omp_get_thread_num();
        blockSize = numVertices/procs + 1;
        for(i = 0; i < blockSize; i++){
            if(i*procs+tid >= numVertices)
                break;
            for(j = 0; j<(i*procs+tid==j)+1;j++){
                        if(i*procs+tid==j)
                            adjA[(i*procs+tid==j)*numVertices+j] = 0;
                        else{
                            if((double)rand() < (double)EDGE_PROBABILITY*(double)RAND_MAX)
                                adjA[(i*procs+tid==j)*numVertices+j] = rand()%MAX_DIST + 1;
                            else
                                adjA[(i*procs+tid==j)*numVertices+j] = -1;
                            adjA[j*numVertices+(i*procs+tid==j)] = adjA[(i*procs+tid==j)*numVertices+j];
                        }
                    }
                }
            }
        
    
    //printf("Matrix A:\n");
    for (i = 0; i < numVertices; i++)
    {
        for (j = 0; j < numVertices; j++)
        {
            printf("%d ",adjA[i*numVertices+j]);
        }
        printf("\n");
    }
    
    adjB = (int *)malloc(sizeof(int) * numVertices * numVertices);
    
    P = (int *)malloc(sizeof(int) * numVertices * numVertices);
    
    //Build a doubly linked list of vertex indices
    head = (Node*)malloc(sizeof(Node));
    head->value = 0;
    head->prev = 0;
    cur = head;
    for(i = 1; i < numVertices; i++){
        Node* newNode = (Node*)malloc(sizeof(Node));
        newNode->prev = cur;
        cur->next = newNode;
        newNode->value = i;
        cur = newNode;
    }
    cur->next = 0;
    
    numRemaining = numVertices;
    i = 0;
    while(head){
        randIndex = rand()% numRemaining;
        vertexIndex = getNode(head,randIndex)->value;
        head = deleteNode(head,getNode(head,randIndex));
        for (j = 0; j < numVertices; j++)
            P[i*numVertices+j] = (vertexIndex == j) ? 1:0;
        i++;
        numRemaining--;
    }
    
    transP = (int *)malloc(sizeof(int) * numVertices * numVertices);
    transpose(transP,P);
    newAdjB = (int *)malloc(sizeof(int) * numVertices * numVertices);
    permute(adjB,adjA,P);
    permute(newAdjB,transP,adjB);
    adjB = newAdjB;
    
    if(isomorphic);
    else{
        randomSwap(adjB);
    }
    
    //printf("Matrix B:\n");
    for (i = 0; i < numVertices; i++)
    {
        for (j = 0; j < numVertices; j++)
        {
            printf("%d ",adjB[i*numVertices+j]);
        }
        printf("\n");
    }
    
    
    
    printf("P:\n");
    for (i = 0; i < numVertices; i++)
    {
        for (j = 0; j < numVertices; j++)
        {
            printf("%d ",P[i*numVertices+j]);
        }
        printf("\n");
    }
    return 0;
}

Node* getNode(Node* head, int index){
    int i;
    Node* cur = head;
    for(i=0;i<index;i++)
        cur= cur->next;
    return cur;
}

Node* deleteNode(Node* head, Node* toDelete){
    if(toDelete == head)
        head = head->next;
    if(toDelete->prev)
        toDelete->prev->next = toDelete->next;
    if(toDelete->next)
        toDelete->next->prev = toDelete->prev;
    free(toDelete);
    return head;
}

void permute(int* adjB,int* adjA,int* P){
    for (i = 0; i < numVertices; i++)
        for (j = 0; j < numVertices; j++)
            adjB[i*numVertices+j] = 0;
    for (i = 0; i < numVertices; i++)
        for (j = 0; j < numVertices; j++)
            for(k = 0; k < numVertices; k++)
                adjB[i*numVertices+j] += P[i*numVertices+k]*adjA[k*numVertices+j];
}

void transpose(int* result, int* adjA){
    for (i = 0; i < numVertices; i++)
        for (j = 0; j < numVertices; j++)
            result[i*numVertices+j] = adjA[j*numVertices+i];
}

void randomSwap(int* adjA){//broken
    int i1 = rand()%numVertices;
    int j1 = rand()%numVertices;
    int i2 = rand()%numVertices;
    int j2 = rand()%numVertices;
    int temp;
    
    while(i1 == j1){
        i1 = rand()%numVertices;
        j1 = rand()%numVertices;
    }
    
    while((i2 == i1)&&(j2 == j1)||(j2==i2)){
        i2 = rand()%numVertices;
        j2 = rand()%numVertices;
    }
    
    temp = adjA[i1*numVertices+j1];
    adjA[i1*numVertices+j1] = adjA[i2*numVertices+j2];
    adjA[i2*numVertices+j2] = temp;
    
}
