#include "benchmark.hpp"
#include "benchmark_type.hpp"
#include "fib.hpp"
#include "sparray.hpp"
#include "tabulate.hpp"
#include "filter.hpp"
#include <string>

benchmark_type fib_bench() {
  long n = pasl::util::cmdline::parse_or_default_long("n", 42);
  long* result = new long;
  
  auto init = [=] { };
  
  auto bench = [=] { 
#ifdef SEQUENTIAL_BASELINE
    *result = demo::fib_seq(n);
#else
    *result = demo::fib_par(n);
#endif
  };
  
  auto output = [=] {
    std::cout << "result " << *result << std::endl;
  };

  auto destroy = [=] {
    delete result;
  };

  return make_benchmark(init, bench, output, destroy);
}

void tab(long* data, long n) {
#ifdef SEQUENTIAL_BASELINE
  for (long i = 0; i < n; i++) data[i] = i;
#else
  demo::tabulate(data, n);
#endif
}

benchmark_type tabulate_bench() {
  long n = pasl::util::cmdline::parse_or_default_long("n", 1l<<20);
  
  long* data = new long[n];
  auto init = [=] { };
  auto bench = [=] {
    tab(data, n);
  };
  auto output = [&] { };
  auto destroy = [=] { };
  return make_benchmark(init, bench, output, destroy);
}

benchmark_type filter_bench() {
  long n = pasl::util::cmdline::parse_or_default_long("n", 1l<<20);
  std::string impl = pasl::util::cmdline::parse_or_default_string("impl", "demo");
  sparray* inp = new sparray(0);
  sparray* outp = new sparray(0);
  auto init = [=] {
    *inp = gen_random_sparray(n);
  };
  auto bench = [=] {
#ifdef SEQUENTIAL_BASELINE
    *outp = demo::filter_seq(is_even_fct, *inp);
#else
    if (impl == "demo") *outp = demo::filter(is_even_fct, *inp);
    else                *outp = filter(is_even_fct, *inp);
#endif
  };
  auto output = [=] {
    std::cout << "result " << (*outp)[outp->size()-1] << std::endl;
  };
  auto destroy = [=] {
    //delete inp;
    //delete outp;
  };
  return make_benchmark(init, bench, output, destroy);
}

/*---------------------------------------------------------------------*/
/* PASL Driver */

int main(int argc, char** argv) {

  benchmark_type bench;

  auto init = [&] {
    pasl::util::cmdline::argmap<std::function<benchmark_type()>> m;
    m.add("fib",                  [&] { return fib_bench(); });
    m.add("tabulate",             [&] { return tabulate_bench(); });
    m.add("filter",               [&] { return filter_bench(); });
    bench = m.find_by_arg("bench")();
    bench_init(bench);
  };
  auto run = [&] (bool) {
    bench_run(bench);
  };
  auto output = [&] {
    bench_output(bench);
  };
  auto destroy = [&] {
    bench_destroy(bench);
  };
  pasl::sched::launch(argc, argv, init, run, output, destroy);
}

/***********************************************************************/
