//// Visualization rendering using GL.
///
/// This file contains methods for Datmo, ObjectBase and ObjectTrack classes.
/// It is split off to localize the dependency on the modutils/GL/fltk and to
/// clarify the main-line code.

#include "datmo.h"
#include <FL/gl.h>
#include "params.h"


//// Color stuff:

int color_tab[16] = {
  0x000000, //Black
  0xFFFFFF, //White
  0x0000FF, //Blue
  0xFF0000, //Red
  0x808080, //Gray
  0xC0C0C0, //Silver
  0x008000, //Green
  0xFFFF00, //Yellow
  0x00FF00, //Lime
  0x808000, //Olive
  0x800000, //Maroon
  0x000080, //Navy
  0x800080, //Purple
  0x008080, //Teal
  0xFF00FF, //Fuchsia
  0x00FFFF //Aqua
};

static const int BLACK_COLOR = 0;
static const int WHITE_COLOR = 1;
static const int BLUE_COLOR = 2;
static const int RED_COLOR = 3;
static const int GRAY_COLOR = 4;
static const int SILVER_COLOR = 5;
static const int GREEN_COLOR = 6;
static const int YELLOW_COLOR = 7;
static const int LIME_COLOR = 8;
static const int OLIVE_COLOR = 9;
static const int MAROON_COLOR = 10;
static const int NAVY_COLOR = 11;
static const int PURPLE_COLOR = 12;
static const int TEAL_COLOR = 13;
static const int FUCHSIA_COLOR = 14;
static const int AQUA_COLOR = 15;

static const int FIRST_TRACK_COLOR = GREEN_COLOR;
static const int NUM_TRACK_COLORS = 16 - FIRST_TRACK_COLOR;

static void glColor16 (int code) {
  ASSERT(code >= 0 && code < 16, "Bad color code.");
  int color24 = color_tab[code];
  glColor3ub((color24 >> 16) & 0xFF,
	     (color24 >> 8) & 0xFF,
	     color24 & 0xFF);
}

// Map track ID to the non-reserved subset of the colors.
int ObjectTrack::color16 () {
  if (!_render_verbose) {
    return ever_moving ? RED_COLOR : BLACK_COLOR;
  } else if (id && (valid || ever_moving)) {
    // if split from, use ID of oldest ancestor track.
    unsigned int eff_id = false && split_from_vec.size() ? split_from_vec.front() : id;
    return (eff_id % NUM_TRACK_COLORS) + FIRST_TRACK_COLOR;
  } else {
    return _black_background ? GRAY_COLOR : SILVER_COLOR;
  }
}


// Robot coordinate convention is X north, Y east, Z down, so to create a map
// view, we swap X and Y.
inline static void vertex_vec2d (const Vec2d &v) {
  glVertex2d(v[1], v[0]);
}
//
inline static void rasterpos_vec2d (const Vec2d &v) {
  glRasterPos2d(v[1], v[0]);
}


// Draw filled rectangle representing approximate measurement uncertainty
// for each scan point.
static void render_points (int color, bool show_occl,
			   vector<PointData> &points, ScannerInfo *scanner)
{
  double sin_res = sin(scanner->resolution());
  for (unsigned int j = 0; j < points.size(); j++) {
    if (show_occl && points[j].occluded && _render_verbose) 
      glColor16(RED_COLOR);
    else
      glColor16(color);
    Vec2d dir = points[j].pos - scanner->location;
    double len = dir.length();
    if (len > 0.1) {
      dir *= 1/len;
      Vec2d norm = normal(dir);
      Vec2d rinc = dir * (_point_depth/2);
      Vec2d ainc = norm * (sin_res * points[j].range/2);
      Vec2d ctr = points[j].pos;
      glBegin(GL_POLYGON);
      vertex_vec2d(ctr - rinc - ainc);
      vertex_vec2d(ctr + rinc - ainc);
      vertex_vec2d(ctr + rinc + ainc);
      vertex_vec2d(ctr - rinc + ainc);
      glEnd();
    }
  }
}


/// Datmo:: methods

// Draw tracks and unassociated segments.
void Datmo::render(double e_o, double n_o) {

  if (!_inited) {
    return;
  }

  gl_font(FL_COURIER_BOLD, _font_size);
  glLineWidth(_line_width);

  if (_render_points_only) {
    for (unsigned int s_ix = 0; s_ix < _scanners.size(); s_ix++) {
      ScannerInfo *scanner = _scanners[s_ix];
      vector<PointData> points;
      for (unsigned int i = 0; i < scanner->num_points; i++)
	points.push_back(scanner->points[i]);
      render_points((_render_verbose ? ((s_ix % NUM_TRACK_COLORS) + FIRST_TRACK_COLOR)
		     : BLACK_COLOR),
		    false, points, scanner);
    }
  } else {
    if (_render_tracks) {
      for (list<ObjectTrack *>::iterator it = _tracks.begin();
	   it != _tracks.end();
	   it++)
	(*it)->render();
    }

    if (_render_verbose) {
      for (list<Segment *>::iterator it = _segment_log.begin();
	   it != _segment_log.end();
	   it++) {
	Segment *seg = *it;
	ObjectTrack *bm = (ObjectTrack *)seg->best_match;
	if (!_render_tracks
	    || !bm
	    || bm->last_associated_time < bm->scanner->cur_time)
	  // assoc failed
	  seg->render();
      }
    }
  }
  if (_ignored_points && _render_verbose)
    render_points(BLUE_COLOR, false, *_ignored_points, _scanner);
}


void Datmo::render_callback(double e_o, double n_o, double low_x, double low_y, double high_x, double high_y,
			    utils::SymbolTable *st, void *data) {
  ((Datmo *)data)->render(e_o, n_o);
}


/// ObjectBase/ObjectTrack methods:

// Expand bounding box by this so that the line is not on top of the points.
static const float box_nudge = 0.3;

// Draw scan points and features.
void ObjectBase::render (unsigned int color16) {
  render_points(color16, true, points, scanner);
  glColor16(color16);
  // Plot the features.
  if (shape_class == LINE || shape_class == CORNER) {
    // plot line or corner.
    glBegin(GL_LINE_STRIP);
    vertex_vec2d(features[FIRST].pos());
      
    if (shape_class == CORNER) {
      vertex_vec2d(features[CORNER_F].pos());
    }

    vertex_vec2d(features[LAST].pos());
    glEnd();

    // tag non-vague line ends.
    if (_render_verbose) {
      for (int i = FIRST; i <= LAST; i++) {
	FeaturePoint &feature = features[i];
	if (!feature.vague) {
	  rasterpos_vec2d(feature.state);
	  gl_draw("X");
	}
      }
    }
  } else if (compact || !_render_tracks) {
    // Plot bounding box.
    Vec2d minp = features[MIN_BOUND].pos();
    Vec2d maxp = features[MAX_BOUND].pos();

    glBegin(GL_LINE_LOOP);
    vertex_vec2d(Vec2d(minp[0] - box_nudge, minp[1] - box_nudge));
    vertex_vec2d(Vec2d(maxp[0] + box_nudge, minp[1] - box_nudge));
    vertex_vec2d(Vec2d(maxp[0] + box_nudge, maxp[1] + box_nudge));
    vertex_vec2d(Vec2d(minp[0] - box_nudge, maxp[1] + box_nudge));
    glEnd();
  }
}


// render 1 track.
void ObjectTrack::render () {
  if (!_show_early_tracks && associated_count < _min_associate_count)
    return;

  unsigned int color = color16();
  ObjectBase::render(color);

  // If has an ID, plot the label.  If moving, show predicted path and
  // acceleration.
  Vec2d pos = (shape_class == COMPLEX && (compact || !_render_tracks)
	       ? features[MAX_BOUND].pos() + Vec2d(box_nudge*2, box_nudge)
	       : position() + norm_dir*box_nudge);
  rasterpos_vec2d(pos);
  
  char buf[80];
  char sbuf[80];
  char vbuf[80];
  if (moving) {
    Vec2d vel = velocity();
    Vec2d acc = acceleration();
    if (_render_verbose) {
      snprintf((char *)&vbuf, sizeof(vbuf), " V%.1lf A%.1lf T%2.lf",
	       vel.length(), acc.length() * signum(vel.dot(acc)),
	       kf_rotation.state[V_THETA] * 180/M_PI);
    } else {
      snprintf((char *)&vbuf, sizeof(vbuf), " V%.1lf", vel.length());
    }
  } else {
    vbuf[0] = '\0';
  }
  
  if (split_from())
    snprintf((char *)&sbuf, sizeof(sbuf), ":%d", split_from());
  else
    sbuf[0] = '\0';

  const char *assoc
    = (((datmo->_scan_time - last_associated_time)
	> _min_no_associate_time)
       ? "?" : " ");

  if (_render_verbose) {
    snprintf((char *)&buf, sizeof(buf), "%s%d%s%s",
	     assoc, id, sbuf, vbuf);
  } else if (ever_moving) {
    snprintf((char *)&buf, sizeof(buf), "%s%d%s",
	     assoc, id % 1000, vbuf);
  } else {
    buf[0] = '\0';
  }

  glColor16(color);
  gl_draw((char *)buf);

  // Show path if it exists, we were ever_moving.
  if (path && ever_moving) {
    static const int space_stride = 1;
    const int dash_stride = _render_verbose ? 5 : 1000000;
    unsigned int ix = 0;
    while (ix < path->size()) {
      Vec2d translate;
      if (_path_incremental_motion)
	translate = position() - incremental_motion();
      glBegin(GL_LINE_STRIP);
      for (int i = 0; i < dash_stride; i++, ix++) {
	if (ix == path->size()) {
	  vertex_vec2d(position());
	  break;
	}
	vertex_vec2d(path->at(ix) + translate);
      }
      glEnd();
      
      ix += space_stride;
    }
  }
  
  if (moving && valid) {
    // Predicted path and acceleration vector.
    glColor16(_render_verbose ? RED_COLOR : BLUE_COLOR);
    glBegin(GL_LINE_STRIP);
    predict_positions();
    for (int i = 0; i < NUM_PREDICT; i++)
      vertex_vec2d(predictions[i]);
    glEnd();

    if (!_render_verbose) {
      glColor16(YELLOW_COLOR);
      glBegin(GL_LINE_STRIP);
      vertex_vec2d(position());
      vertex_vec2d(position() + acceleration());
      glEnd();
    }
  }
}

void Segment::render () {
  int color;
  ObjectTrack *bm = (ObjectTrack *)best_match;
  if (bm && !_render_tracks)
    color = bm->color16();
  else
    color = _black_background ? GRAY_COLOR : SILVER_COLOR;

  ObjectBase::render(color);
  if (id && bm && !_render_tracks) {
    rasterpos_vec2d(features[CENTER].pos() + norm_dir*box_nudge);
    char buf[80];
    char sbuf[80];
    if (bm->split_from())
      snprintf((char *)&sbuf, sizeof(sbuf), ":%d", bm->split_from());
    else
      sbuf[0] = '\0';
  
    snprintf((char *)&buf, sizeof(buf), "%d%s", id, sbuf);
    glColor16(color);
    gl_draw((char *)buf);
  }
}
