#include <math.h>
#include <new>
#include <stdio.h>

#include <utils/Linear.h>
#include <utils/ConfigFile.h>
#include <MapBase/MapBase.h>
#include "datmomod.h"
#include <unistd.h>
#include <time.h>

datmomod::datmomod(const char *spec)
  : Module("datmomod", spec)
{
  _display = NULL;
  _gui = NULL;
  _last_add_time = 0;
}

datmomod::~datmomod()
{
  delete _datmo;
  delete _gui;
  for (unsigned int i = 0; i < _scanners.size(); i++)
    delete _scanners[i];
  if (_scan_log) delete _scan_log;

  // ram: this seg-faults
  //  delete _display;
}

bool datmomod::initialize(ConfigSource *config, 
                          utils::SymbolTable *table)
{
  const char *spec;
  if (config->getBool("display", true)) {
    spec = config->getString("display_spec");
    if (!spec[0]) {
      fprintf(stderr, "No display_spec defined\n");
      return false;
    }

    _display = new Display(spec, getSymbolTable());
    if (_display == NULL) {
      fprintf(stderr, "Spec '%s' failed to create a valid display\n", spec);
      return false;
    }
  }

  spec = config->getString("vehstate_spec");
  if (!spec[0]) {
    fprintf(stderr, "No vehstate_spec defined\n");
    return false;
  }
  _vs = create<VehState>(spec);

  if (!_vs) {
    fprintf(stderr, "Spec '%s' failed to create a valid VS\n", spec);
    return false;
  }

  const int max_scanners = 10;
  const char *scanner_names[max_scanners];
  int num_scanners = config->getStrings("scanners", scanner_names, max_scanners);
  ASSERT(num_scanners, "No scanners configured.");

  for (int i = 0; i < num_scanners; i++)
    _scanners.push_back(new ScannerInfo(this, scanner_names[i], config));

  int num_boxes = config->numValues("ignored_spaces");
  num_boxes /= 4;
  num_boxes *= 4;
  double* space_values = new double[num_boxes];
  memset(space_values, 0, num_boxes*sizeof(double));
  config->getDoubles("ignored_spaces", space_values, num_boxes);
  std::vector<datmo_bounding_box_t> ignored_spaces;
  datmo_bounding_box_t b;
  for (int i=0;i<num_boxes;i+=4) {
    b.x0 = space_values[i];
    b.y0 = space_values[i+1];
    b.x1 = space_values[i+2];
    b.y1 = space_values[i+3];
    ignored_spaces.push_back(b);
    printf("Ignoring %f, %f -> %f, %f\n", b.x0, b.y0, b.x1, b.y1);
  }
  delete [] space_values;


  const char *scan_log_name = config->getString("scan_log");
  if (scan_log_name && scan_log_name[0]) {
    _scan_log = new ofstream(scan_log_name);
    ASSERT(_scan_log, "Couldn't open " <<scan_log_name <<" for output.");
    *_scan_log <<"# scanners: ";
    for (unsigned int i = 0; i < _scanners.size(); i++) {
      *_scan_log <<_scanners[i]->name <<" ";
    }
    *_scan_log <<endl;
  } else {
    _scan_log = NULL;
  }

  _datmo = new Datmo(_vs, config,
                     ignored_spaces,  // vector of ignored areas
		     _scanners,
                     _display,
		     _scan_log);

  _min_x = config->getDouble("min_x_", -10.0);
  _max_x = config->getDouble("max_x", 10.0);
  _min_y = config->getDouble("min_y", 0.0);
  _max_y = config->getDouble("max_y", 5.0);
  spec = config->getString("scan_track_out_spec");
  _track_out = create<ScanTrackOut>(spec);
  if (_track_out) {
    printf("Outputting tracks with %s\n", spec);
  }

  _tracker_objects_limit = config->getInt("tracker_objects_limit");
  ASSERT(_tracker_objects_limit, "No tracker_objects_limit?");
  _objects_limit_accum = 0;

  _exit_on_no_data = config->getBool("exit_on_no_data", true);
  _sleep_for = config->getDouble("sleep_for", 0.0);
  _read_strategy = config->getInt("read_strategy", 0);

  return true;
}

bool datmomod::display()
{
  double northings, eastings, elevation, heading;
#if 0
  _cur_state.getAbsolutePose(northings, eastings, elevation, heading);
#else
  northings = _cur_state.x;
  eastings = _cur_state.y;
  elevation = _cur_state.z;
  heading = _cur_state.yaw;
#endif

  _display->set_veh_pos(eastings, northings, heading,
                        _cur_state.x, _cur_state.y, _cur_state.yaw);

  _display->redraw();
  Fl::check();
  return true;
}

static void rotate_covariance(const utils::Mat2d& src_cov,
                              utils::Mat2d& dest_cov, double yaw)
{
  utils::Mat2d rot_mat(cos(yaw), -sin(yaw), sin(yaw), cos(yaw));
  
  dest_cov = rot_mat.transpose()*src_cov*rot_mat;
}

void datmomod::make_local(const utils::Vec2d& src_vec,
                          float& out_x, float& out_y)
{
  utils::Vec3d v(src_vec[0], src_vec[1], 0);
  v = _inv_cur_state*v;
  out_x = v[0]; out_y=v[1];
}

bool datmomod::initializeDisplay(ConfigSource *config)
{
  MapBase     *_mb;

  _display->size_range(500, 300);
  _display->track_veh_pos(true);
  _display->set_northing_window(20.0);
  _gui = new Gui(_display, 750, 750, "Visualize");

  /*
   * Load a mapbase file if one is available
   */
  const char* spec = config->getString("mapbase_spec");
  if (!spec[0]) {
    printf("No mapbase file defined, so we won't use one\n");
    _mb = 0;
  } else {
    _mb = create<MapBase>(spec);
    if (!_mb) {
      fprintf(stderr, "Spec '%s' failed to create a mapbase, so ignoring it.\n", spec);
    }
  }

  _gui->show();
  return true;
}

void datmomod::export_TrackVal (TrackVal &val, const PointVec &v2,
				const Mat2d &vel_cov_arg)
{
  Mat2d vel_cov;
  rotate_covariance(vel_cov_arg, vel_cov, -_cur_state.yaw);
  val.xx = vel_cov.data[0][0];
  val.yy = vel_cov.data[1][1];
  val.xy = vel_cov.data[0][1];
  
  utils::Vec3d v_rot(v2[0], v2[1], 0);
  _inv_cur_state.multMatrixDir(v_rot, v_rot);
  val.x = v_rot[0];
  val.y = v_rot[1];
}


// Read data from all scanners, waiting if none is available.  Send the data
// to datmo.
//
bool datmomod::add_scans () {
  bool read_failed = false;
  bool read_any = false;

  switch (_read_strategy) {
  case 0:
    // nothing clever, just read from all scanners until at least one either
    // returns some data or fails.  Normally the business of getLine not
    // failing but also not returning new data happens only in canned replay.
    // The sleep prevents us from spin looping in that case.  If the read time
    // is older than the _last_add_time, then we do an extra bonus read to
    // help that scanner get caught up.
    while (true) {
      for (unsigned int i = 0; i < _scanners.size(); i++) {
	ScannerInfo *scanner = _scanners[i];
	if (!scanner->getLine())
	  read_failed = true;
	else if (scanner->cur_time != scanner->last_time) {
	  if (scanner->cur_time < _last_add_time) {
	    if (_scan_log)
	      *_scan_log <<"catch_up: " <<scanner->name <<endl;
	    scanner->getLine();
	  }
	  read_any = true;
	}
      }
      if (read_failed || read_any) break;
      usleep(5000);
    }
    break;

  case 1:
    // First poll all scanners and see if there is any new data.
    for (unsigned int i = 0; i < _scanners.size(); i++) {
      ScannerInfo *scanner = _scanners[i];
      if (scanner->ls->poll()) {
	if (scanner->getLine()) {
	  if (scanner->cur_time != scanner->last_time)
	    read_any = true;
	} else {
	  read_failed = true;
	}
      }
    }

    // If no new data, wait on the scanner that most recently had data.  This
    // way we will wait the longest in the case where the scanners have the
    // same update rate.  If one is faster than the others, we will also tend
    // to poll it faster, though in that case probably this code should be
    // cleverer.
    if (!read_any) {
      ScannerInfo *max_scan = _scanners[0];
      for (unsigned int i = 0; i < _scanners.size(); i++) {
	ScannerInfo *scanner = _scanners[i];
	if (scanner->cur_time > max_scan->cur_time)
	  max_scan = scanner;
      }

      // Keep trying to read until we either actually get data or fail.  With
      // GUI timesource this may take many tries.
      while (true) {
	if (!max_scan->getLine()) {
	  read_failed = true;
	  break;
	}
	if (max_scan->cur_time != max_scan->last_time) break;
	usleep(5000);
      }
    
      // Now re-poll all the other scanners and see if they now have new data.
      for (unsigned int i = 0; i < _scanners.size(); i++) {
	ScannerInfo *scanner = _scanners[i];
	if (scanner != max_scan && scanner->ls->poll()) {
	  if (!scanner->getLine()) read_failed = true;
	}
      }
    }
    break;

  default:
    ERROR("Bad read strategy: " <<_read_strategy);
  }


  // Tracker load control.  Drop a set of scans if we have been tracking too
  // much stuff.

  // Don't let us build up more than 2 cycles worth of credit for good
  // performance.
  int num_tracks = (int)(_datmo->get_tracks().size());
  double start_time = Time::getRealTimeOfDay().getValue();
  _objects_limit_accum += _tracker_objects_limit;
  _objects_limit_accum = min(_objects_limit_accum, _tracker_objects_limit*2);
  if (_objects_limit_accum < 0) {
    if (_scan_log) {
      do_scan_log();
      *_scan_log <<"(dropped for tracker_objects_limit)\n";
    }
  } else {
    _objects_limit_accum -= num_tracks;
    // Don't drop more than 5 scans no matter what.
    _objects_limit_accum = max(_objects_limit_accum, -_tracker_objects_limit*5);

    bool begun_yet = false;
    // Now process available scans in time order.  This amounts to a sort.
    while (true) {
      ScannerInfo *min_scan = NULL;
      for (unsigned int i = 0; i < _scanners.size(); i++) {
	ScannerInfo *scanner = _scanners[i];
	if (scanner->cur_time != scanner->last_time
	    && (!min_scan || scanner->cur_time < min_scan->cur_time))
	  min_scan = scanner;
      }
      if (!min_scan) break;
      if (!begun_yet) {
	_datmo->begin_scans();
	do_scan_log();
	begun_yet = true;
      }
      _last_add_time = max(min_scan->cur_time, _last_add_time);
      if (!_datmo->add_scan(min_scan)) {
	WARN("add_scan failed.");
      }
      min_scan->last_time = min_scan->cur_time;
    }

    if (begun_yet) {
      _datmo->end_scans();
    }
  }

  if (_scan_log) {
    *_scan_log <<"stats: " <<num_tracks <<" "
	       <<Time::getRealTimeOfDay().getValue() - start_time
	       <<endl;
  }

  if (read_failed) {
    WARN("Error getting line");
    sleep(1);
    return !_exit_on_no_data;
  } else {
    return true;
  }
}

void datmomod::do_scan_log () {
  if (_scan_log) {
    double t0 = _scanners[0]->cur_time;
    *_scan_log <<"reads: ";
    _scan_log->precision(13);
   *_scan_log <<t0 <<" ";
    _scan_log->precision(4);
    for (unsigned int s_ix = 0; s_ix < _scanners.size(); s_ix++) {
      *_scan_log <<_scanners[s_ix]->cur_time - t0 <<" ";
    }

    *_scan_log <<" ";
    for (unsigned int s_ix = 0; s_ix < _scanners.size(); s_ix++) {
      *_scan_log <<_scanners[s_ix]->cur_time - _scanners[s_ix]->last_time <<" ";
    }

    *_scan_log <<endl;
    _scan_log->precision(6);
  }
}

// Convert single track to TrackDescription and output it.
void datmomod::do_output_1 (ObjectTrack &track) {
  TrackDescription desc;
  memset(&desc, 0, sizeof(TrackDescription));

  desc.num_points = _out_points.numElems();
  desc.points = _out_points.getData();

  make_local(track.position(), desc.center.x, desc.center.y);

  utils::Mat2d pos_cov = track.position_covariance();
  rotate_covariance(pos_cov, pos_cov, -_cur_state.yaw);
  desc.center.xx = pos_cov.data[0][0];
  desc.center.yy = pos_cov.data[1][1];
  desc.center.xy = pos_cov.data[0][1];

  export_TrackVal(desc.vel, track.velocity(), track.velocity_covariance());
  export_TrackVal(desc.accel, track.acceleration(),
		  track.acceleration_covariance());

  double theta, c_theta;
  track.get_theta(theta, c_theta);
  desc.rot.theta = theta;
  desc.rot.theta_theta = c_theta;
  desc.rot.omega = track.kf_rotation.state[ObjectTrack::V_THETA];
  desc.rot.omega_omega
    = track.kf_rotation.covariance().at(ObjectTrack::V_THETA, ObjectTrack::V_THETA);

  export_TrackVal(desc.delta, track.incremental_motion(),
		  track.incremental_motion_covariance());

  desc.track_start_secs = (int)track.create_time;
  desc.track_start_usecs
    = (int)((track.create_time - desc.track_start_secs)*1e6);
  desc.track_id = track.id;
  desc.merged_with = track.merged_with;
  desc.split_from = track.split_from();
  desc.shape_class = track.shape_class;
  desc.type_class = track.kind;

  if (track.moving)
    desc.flags |= TRACK_MOVING;
  if (track.valid)
    desc.flags |= TRACK_VALID;
  if (track.compact)
    desc.flags |= TRACK_COMPACT;
  if (track.disoriented)
    desc.flags |= TRACK_DISORIENTED;
  if (track.ever_moving)
    desc.flags |= TRACK_EVER_MOVING;
  if (track.kind == ObjectTrack::DEAD_TRACK
      && !track.living_offspring()
      && track.occlusion != ObjectTrack::TOTAL)
    desc.flags |= TRACK_DIED_WITHOUT_OFFSPRING;

  int ix = 0;
  if (track.shape_class == ObjectTrack::COMPLEX) {
    make_local(track.features[ObjectTrack::MIN_BOUND].pos(),
	       desc.features[ix].x, desc.features[ix].y);
    ix++;
    make_local(track.features[ObjectTrack::MAX_BOUND].pos(),
	       desc.features[ix].x, desc.features[ix].y);
    ix++;
  } else {
    make_local(track.features[ObjectTrack::FIRST].pos(),
	       desc.features[ix].x, desc.features[ix].y);
    ix++;
    if (track.shape_class == ObjectTrack::CORNER) {
      make_local(track.features[ObjectTrack::CORNER_F].pos(),
		 desc.features[ix].x, desc.features[ix].y);
      ix++;
    }
    make_local(track.features[ObjectTrack::LAST].pos(),
	       desc.features[ix].x, desc.features[ix].y);
    ix++;
  }

  for (int i=ix;i<3;i++)
    desc.features[i].x = desc.features[i].x = TRACK_UNKNOWN_POS;

  // points set w/ num_points.
  _track_out->output(desc);
}


// See which tracks are in our interest region, and output them if so.
bool datmomod::do_output () {
  if (!_vs->getState(_cur_state)) {
    WARN("Error getting vehicle state");
    sleep(1);
    return !_exit_on_no_data;
  }

  std::vector<ObjectTrack *> tracks = _datmo->get_tracks();

  utils::Transform t;
  _cur_state.setTransform2D(t);
  _inv_cur_state = t.inverse();

  utils::Time tag(_cur_state.secs, _cur_state.usecs);
  int num_out = 0;
  if (_track_out && _track_out->start(tag)) {
    TrackPoint pt;
    for (unsigned int i=0; i<tracks.size(); i++) {
      ObjectTrack& track = *tracks[i];

      bool in_region = false;
      _out_points.clear();
      for (unsigned int j=0; j<track.points.size(); j++) {
        utils::Vec3d data_point(track.points[j].pos[0], track.points[j].pos[1],
                                0);
        data_point = _inv_cur_state*data_point;
        pt.x = data_point[0];
        pt.y = data_point[1];
        _out_points.append(pt);
        if (in_region || (pt.x >= _min_x && pt.x <= _max_x &&
                          pt.y >= _min_y && pt.y <= _max_y)) {
          in_region = true;
        }
      }
      if (in_region) {
	do_output_1(track);
	num_out++;
      }
    }

    char buffer[300];
    snprintf(buffer, sizeof(buffer), "%d objects", num_out);
    getConfigSource()->setStatusMessage(buffer);
    _track_out->send();
  }

#if 0
  double elapsed = (Time::getRealTimeOfDay().getValue() - start_time);
  cout <<elapsed <<" " <<_datmo->get_tracks().size() <<endl;
#endif
  return true;
}


bool datmomod::run() {
  bool res = add_scans() && do_output();
  if (_sleep_for)
    Time::sleep(_sleep_for);
  return res;
}

MODULE_MAIN(datmomod, datmomod);
