// Detection And Tracking of Moving Objects (in LIDAR data)
//
// Author: Robert A MacLachlan
//
// Earlier DATMO implementations were written by Justin Carlson and 
// Chieh-Chih Wang.  

#include "datmo.h"
#include <new>
#include <math.h>
#include <stdio.h>
#include <algorithm>
#include "TrackLogger.h"
#include "params.h"
#include <float.h>
#include "float_trap.h"


//// Datmo class

static Datmo *gDatmo = NULL;

Datmo::Datmo (VehState *vs, ConfigSource *config,
	      const vector<datmo_bounding_box_t> &ignored_spaces,
	      vector<ScannerInfo *> &scanners,
	      Display *display, ostream *scan_log) :
  _vs(vs),
  _display(display),
  _inited(false),
  _iteration(0),
  _first_scan_time(-1),
  _prev_scan_time(-1),
  _scan_time(-1),
  _next_track_id(1),
  _used(NULL),
  _ignored_points(NULL),
  _scan_log(scan_log),
  _last_validated(0)
{
  _ignored_spaces = ignored_spaces;
  _scanners = scanners;
  init_params(config);
  ObjectTrack::static_init();
  Segment::static_init();
  _times.setNumElems(_max_scan_size);
  _states.setNumElems(_max_scan_size);
  _used = new (unsigned char)[_max_scan_size];
  if (_show_ignored_points && _display)
    _ignored_points = new vector<PointData>;
}

void Datmo::init ()
{
  ASSERT(!_inited, "already initialized?");
  gDatmo = this;
  _inited = true;
  if (_display != NULL) {
    _display->add_callback(render_callback, 4, DISPLAY_SMOOTH, this);
  }
  ft_init();
}
      
Datmo::~Datmo () {
  delete [] _used;
  _used = NULL;
  for (list<Segment *>::iterator seg_it = _segment_log.begin();
       seg_it != _segment_log.end();
       seg_it++)
    delete *seg_it;
  _segment_log.clear();
  _segments.clear();

  for (list<ObjectTrack *>::iterator track_it = _tracks.begin();
       track_it != _tracks.end();
       track_it++)
    delete *track_it;
  _tracks.clear();

  delete _ignored_points;
  _ignored_points = NULL;
}


// Entry point to get datmo results. Return all active tracks and all dead
// tracks.
vector<ObjectTrack *> &Datmo::get_tracks () {
  _output_tracks.clear();
  for (list<ObjectTrack *>::iterator track_it = _tracks.begin();
       track_it != _tracks.end();
       track_it++) {
    ObjectTrack *track = *track_it;
    _output_tracks.push_back(track);
  }

  for (list<ObjectTrack *>::iterator track_it = _dead_tracks.begin();
       track_it != _dead_tracks.end();
       track_it++) {
    ObjectTrack *track = *track_it;
    _output_tracks.push_back(track);
  }

  return _output_tracks;
}


//// Datmo::import_data

// Check if a scan return is or in one of our ignore boxes, returning
// true if it is.
bool Datmo::in_ignore_region (const LineScanElem &p) {
  double vx = _scanner->x_offset + p.range * cos(_scanner->yaw_offset + p.bearing);
  double vy = _scanner->y_offset + p.range * sin(_scanner->yaw_offset + p.bearing);
  for (unsigned int j = 0; j < _ignored_spaces.size(); j++) {
    if ((_ignored_spaces[j].x0 < vx)
	&& (_ignored_spaces[j].x1 > vx) 
	&& (_ignored_spaces[j].y0 < vy) 
	&& (_ignored_spaces[j].y1 > vy)) 
      return true;
  }
  return false;
}


// Convert coordinates and stuff our points vector.  For ignored points, we
// set _used true, which prevents them from being placed in any segment.  We
// compute the XY position even of points that are ignored because the
// vagueness test later on looks at the adjacent points to a segment,
// whereever they lie.
// 
// ### we could avoid the trig functions in the inner loop by using transform
// matrices.  The transform for each point could be computed by repeatedly
// multiplying rotation matrices for the bus rotation increment and the
// scanner rotation increment.
//
// ### there's also an issue here due to our flat-earth assumption.  If
// there's significant pitch we're going to get mismatch between the pose XY
// change and the in-plane change in target distance.  This can cause apparent
// motion in objects that are actually stationary.  This will cause objects to
// appear to move towards us when we are approaching them.
//
bool Datmo::import_data (ScannerInfo *scanner) {
  ft_poll_before("import_data");
  ASSERT(scanner->lse.numElems() <= _max_scan_size,
	 "Actual num elements " <<scanner->lse.numElems()
	 <<" exceeds max_scan_size.");

  LineScanElem *lse_ptr  = scanner->lse.getData();
  utils::Time *time_ptr = _times.getData();
  VehStateStruct *vss_ptr = _states.getData();
  _scanner = scanner;
  _scanner->num_points = scanner->lse.numElems();

  // Get time for each scan point.
  for (unsigned int i = 0; i < _scanner->num_points; i++) {
    time_ptr[i].setValue(lse_ptr[i].secs, lse_ptr[i].usecs);
  }

  _prev_scan_time = _scan_time;
  _scan_time = scanner->cur_time;
  if (_first_scan_time < 0) {
    _prev_scan_time = _first_scan_time = _scan_time;
  }

  double apparent_delta_t = _scan_time - _prev_scan_time;
  if (_scan_log)
    *_scan_log <<"add: " <<_scan_time - _first_scan_time <<" "
	       <<scanner->name <<" " <<apparent_delta_t <<" ";

  // Time can jump backwards during canned replay.  It can also jump backward
  // by small amounts when there are multiple scanners and one scanner's data
  // is delayed for some reason.  With very small amounts, we pretend the time
  // glitch didn't happen.  For larger glitches, we discard the data.  If the
  // backwards motion accumulates large enough, then we reset some state and
  // actually back up.
  //
  if (apparent_delta_t < 0) {
    if (-apparent_delta_t < _back_time_ignore) {
      _scan_time = _prev_scan_time;
      if (_scan_log) *_scan_log <<"(ignored) ";
    } else if (-apparent_delta_t < _back_time_discard) {
      if (_scan_log) *_scan_log <<"(discarded) ";
      _scan_time = _prev_scan_time;
      return true;
    } else {
      // Clear the history, as it will no longer be in the past, and confuse
      // things badly.  Also, the last associated time is set to the scan_time
      // so that it won't be in the future and confuse the track timeout.  The
      // _prev_scan_time is used to drive the update process, which interprets
      // negative delta_t as backward motion.
      if (_scan_log) *_scan_log <<"(resetting) ";
      DEBUG(_scanner->name <<": Time went backwards by "
	    <<-apparent_delta_t <<", resetting.");
      for (list<ObjectTrack *>::iterator track_it = _tracks.begin();
	   track_it != _tracks.end();
	   track_it++) {
	(*track_it)->history->clear();
	(*track_it)->last_associated_time = _scan_time;
      }
    }
  }

  // getStates is slow (maybe only in canned replay), and we only use the x,
  // y, yaw, so just linearly interpolate those.
  VehStateStruct first;
  if (!_vs->getState(_times[0], first)) {
    WARN("Couldn't get first state.");
    return false;
  }
  VehStateStruct last;
  if (!_vs->getState(_times[_scanner->num_points-1], last)) {
    last = first;
    WARN("Failed to get last state.\n"
	 "    This should only happen at the end of canned replay.");
  }
  double dx = (last.x - first.x)/(_scanner->num_points-1);
  double dy = (last.y - first.y)/(_scanner->num_points-1);
  double delta_yaw = last.yaw - first.yaw;
  // handle wrap.
  if (fabs(delta_yaw) > M_PI)
    delta_yaw -= signum(delta_yaw) * 2*M_PI;
  double dyaw = delta_yaw/(_scanner->num_points-1);
  ASSERT(fabs(dyaw) < 0.003, "Ridiculous apparent yaw rate: " <<dyaw);
  double x = first.x;
  double y = first.y;
  double yaw = first.yaw;
  for (unsigned int i = 0; i < _scanner->num_points; i++) {
    _states[i] = first;
    _states[i].x = x;
    _states[i].y = y;
    _states[i].yaw = yaw;
    x += dx;
    y += dy;
    yaw += dyaw;
  }

  for (unsigned int i = 0; i < _scanner->num_points; i++) {
    double cosvy = cos(vss_ptr[i].yaw);
    double sinvy = sin(vss_ptr[i].yaw);
    double scanner_x = (vss_ptr[i].x + (_scanner->x_offset * cosvy)
			- (_scanner->y_offset * sinvy));
    double scanner_y = (vss_ptr[i].y + (_scanner->x_offset * sinvy)
			+ (_scanner->y_offset * cosvy));
    double return_yaw = vss_ptr[i].yaw + _scanner->yaw_offset + lse_ptr[i].bearing;
    _scanner->points[i].pos[0] = scanner_x + lse_ptr[i].range * cos(return_yaw);
    _scanner->points[i].pos[1] = scanner_y + lse_ptr[i].range * sin(return_yaw);
    _scanner->points[i].range = lse_ptr[i].range;
    _scanner->points[i].scan_pos = i;
    _scanner->points[i].occluded = false;
    _scanner->location = Vec2d(scanner_x, scanner_y);
    bool ignore = in_ignore_region(lse_ptr[i]);
    if (ignore || lse_ptr[i].range > _range_limit) {
      _used[i] = 1;
      if (ignore) {
	_scanner->ignored[i] = 1;
	if (_ignored_points)
	  _ignored_points->push_back(_scanner->points[i]);
      }
    } else {
      _scanner->ignored[i] = 0;
      _used[i] = 0;
    }
  }

  if (ft_poll_after("import_data"))
    return false;
  else
    return true;
}


//// Datmo::segment

// Place points within _max_segment distance in the same segment.  These
// points are not nececessarily strictly contiguous in scan order because the
// object may have a hole revealing a back object, or some front outlier due
// to a false return or obscuring object.  So if we find a point too far we
// must continue scanning for some additional bearing bins.  The delta-bearing
// depends on the range.  We find the max angle that could possibly be between
// this point and a next point within the segment threshold distance.  The
// pi/2 compensates for the fact that when we multiply radius * angle, we are
// measuring an arc, not a straight line.  This gives a conservative bound.
//
// During this process, we set the occluded flag in points that are adjacent
// to a point not in the segment, where the adjacent point lies in front
// (shadowing.)
//
void Datmo::segment () {
  // Clear out old segments.  These are deleted later (out of _segment_log)
  _segments.clear();

  unsigned int next_unsegmented = 0; // Start of current segment
  vector<PointData> points;
  Vec2d prev;
  
  do {
    // skip over points used by previous segment or that are ignored.
    while (next_unsegmented < _scanner->num_points && _used[next_unsegmented]) {
      next_unsegmented++;
    }
    if (next_unsegmented == _scanner->num_points) break;
    // now at first point in new segment.

    unsigned int ix = next_unsegmented; // scan pointer
    // How far we need to scan looking for points in this seg. =0 for -Wall
    unsigned int end = 0;
    bool prev_valid = false; // false only on first point in segment.
    bool skipping = true; // scanned one or points outside current segment.
    unsigned int num_non_occluded = 0;
    do {
      PointData &newp = _scanner->points[ix];
      if (!prev_valid
	  || (!_used[ix]
	      && (newp.pos - prev).length() < _max_segment_distance)) {

	// If this is the first good point after one or more bad, check for
	// occlusion on this point by preceding.
	if (skipping) {
	  if (ix == 0) {
	    newp.occluded = true; // first point always occluded.
	  } else {
	    float prev_rng = _scanner->points[ix-1].range;
	    if (prev_rng < newp.range)
	      newp.occluded = true;
	  }
	  skipping = false;
	}

	prev = newp.pos;
	prev_valid = true;
	points.push_back(newp);
	if (!newp.occluded) num_non_occluded++;
	_used[ix] = 1;
	float max_angle = _max_segment_distance
          // so we don't divide by zero
	  / max(_scanner->lse[ix].range, (float)0.01)
	  * M_PI/2;
	end = min(ix + (unsigned int)(ceil(max_angle/_scanner->resolution())),
		  _scanner->num_points);
      } else if (!skipping) {
	// If first skipped point, check for occlusion on prev point by this
	// point.
	assert(ix);
	if (newp.range < _scanner->points[ix-1].range) {
	  _scanner->points[ix-1].occluded = true;
	  points.back().occluded = true;
	}
	skipping = true;
      }
	
      ix++;
    } while (ix < end);
    _scanner->points[_scanner->num_points-1].occluded = true; // last point always occluded.

    if (num_non_occluded >= _min_segment_points) {
      Segment *seg = new Segment(this, _scanner, points);
      seg->shape_classify();
      _segments.push_back(seg);
    } else {
      ignore_points(points);
    }
    points.clear();
  } while (next_unsegmented < _scanner->num_points);
}


// ignore_point, ignore_points -- internal
//
// Mark a point or points as ignored.  Save it ignored_points for
// visualization and also set the ignored flag for that point in the scanner
// so that occlusion detection can tell if the point is significant.
//
void Datmo::ignore_point (const PointData &point) {
  _scanner->ignored[point.scan_pos] = 1;
  if (_ignored_points)
    _ignored_points->push_back(point);
}

void Datmo::ignore_points (const vector<PointData> &points) {
  for (unsigned int i = 0; i < points.size(); i++)
    ignore_point(points[i]);
}


//// Datmo::model_segment
//
//    Use existing nearby tracks as a model for fitting the measured points
// using a K-means algorithm.


// Find the SegmentSets describing the overlap relationships.  When there is a
// 1-1 relationship or no overlap, leave it in the segments list and let
// associate handle it.
void Datmo::find_segment_sets () {
  // clear out old info.
  for (list<ObjectTrack *>::iterator track_it = _tracks.begin();
       track_it != _tracks.end();
       track_it++) {
    ObjectTrack *track = *track_it;
    track->segment_set = NULL;
  }
  for (unsigned int i = 0; i < _segment_sets.size(); i++)
    delete _segment_sets[i];
  _segment_sets.clear();

  // Iterate over all raw segments finding the mutually overlapping tracks.
  for (list<Segment *>::iterator seg_it = _segments.begin();
       seg_it != _segments.end();
       seg_it++) {
    Segment *seg = *seg_it;

    int n_overlap = seg->overlap.size();
    // If seg has no overlap track (will become new track) or match is 1-1
    // (association trivial), then ignore.
    if (!((n_overlap == 0)
	  || (n_overlap == 1 && seg->overlap.front()->overlap.size() == 1))) {
      // SegmentSet we are currently building.  Null the first time around the
      // loop.  If non-null, it already has this segment's points added.
      SegmentSet *sset = NULL;
      for (list<ObjectBase *>::iterator track_it = seg->overlap.begin();
	   track_it != seg->overlap.end();
	   track_it++) {
	ObjectTrack *track = (ObjectTrack *)(*track_it);
	track->overlap.clear();  // clear out soon-to-be-dead pointers
	if (track->segment_set) {
	  if (sset) {
	    if (track->segment_set != sset) {
	      // have to merge sets.
	      SegmentSet *victim = track->segment_set;
	      sset->points.insert(sset->points.end(), victim->points.begin(),
				  victim->points.end());
	      sset->tracks.insert(sset->tracks.end(), victim->tracks.begin(),
				  victim->tracks.end());
	      sset->raw_segs.insert(sset->raw_segs.end(), victim->raw_segs.begin(),
				    victim->raw_segs.end());
	      for (unsigned int i = 0; i < victim->tracks.size(); i++)
		victim->tracks[i]->segment_set = sset;
	      victim->points.clear();
	      victim->tracks.clear();
	      victim->raw_segs.clear();
	    }
	  } else {
	    // Found an existing set, but we don't have one yet.  Just add our
	    // points to that set.
	    sset = track->segment_set;
	    sset->points.insert(sset->points.end(), seg->points.begin(),
				seg->points.end());
	  }
	} else {
	  if (sset) {
	    // Add this track to our segment set.
	    sset->tracks.push_back(track);
	    track->segment_set = sset;
	  } else {
	    // Make new segment set and add our points and this track to it.
	    sset = new SegmentSet;
	    sset->id = _segment_sets.size();
	    sset->points = seg->points;
	    sset->tracks.push_back(track);
	    track->segment_set = sset;
	    _segment_sets.push_back(sset);
	  }
	}
      }
      sset->raw_segs.push_back(seg);
    }
  }
}

// Return the segment closest to p.  To improve performance with segments that
// fit their model very poorly, we also consider a point to be closest to
// prev_seg if it is closer to the point before p,
//   prev_seg->points[prev_ix - 1]
// than to any other segment model.  This way, with segments that are
// reasonably smooth, but have poor line fit, we are reluctant to move the
// point to another segment.
Segment *Datmo::closest_segment
    (Segment *prev_seg, unsigned int prev_ix, const PointVec &p, SegmentSet *sset) 
{
  Segment *close_seg = NULL;
  double close_dist = DBL_MAX;

  if (prev_seg && prev_ix >= 1) {
    close_seg = prev_seg;
    close_dist = (prev_seg->points[prev_ix - 1].pos - p).length();
  }

  for (unsigned int i = 0; i < sset->tracks.size(); i++) {
    Segment *seg = (Segment *)(sset->tracks[i]->best_match);
    if (seg) {
      double dist = seg->distance(p);
      if (dist < close_dist) {
	close_seg = seg;
	close_dist = dist;
      }
    }
  }
  ASSERT(close_seg, "no segments?");
  return close_seg;
}

static int pointdata_scan_order (const void *x, const void *y) {
  PointData *pdx = (PointData *)x;
  PointData *pdy = (PointData *)y;
  return pdx->scan_pos > pdy->scan_pos ? 1 : -1;
}


// K-means iteration process.  Fit current point assignments, then reassign
// any points which are no longer closest to their current segment.
void Datmo::model_segment_1 (SegmentSet *sset) {
  bool anything_changed;
  int iter = 0;
  do {
    anything_changed = false;
    for (unsigned int i = 0; i < sset->tracks.size(); i++) {
      Segment *seg = (Segment *)(sset->tracks[i]->best_match);
      if (seg) {
	qsort(&seg->points.front(), seg->points.size(), sizeof(PointData),
	      pointdata_scan_order);
	seg->shape_classify();
      }
    }

    iter++;
    if (iter > _max_segment_iterations) {
      LOG(sset->tracks[0], "max_segment_iterations exceeded.");
      break;
    }

    // Move points to the new closest segment, but never move the last two
    // points so that the segment doesn't become empty.
    for (unsigned int seg_ix = 0; seg_ix < sset->tracks.size(); seg_ix++) {
      Segment *seg = (Segment *)(sset->tracks[seg_ix]->best_match);
      if (seg) {
	for (unsigned int i = 0;
	     i < seg->points.size() && seg->points.size() > 2;
	     ) {
	  Segment *cseg = closest_segment(seg, i, seg->points[i].pos, sset);
	  if (cseg != seg) {
	    anything_changed = true;
	    cseg->points.push_back(seg->points[i]);
	    vector<PointData>::iterator it = seg->points.begin();
	    advance(it, i);
	    seg->points.erase(it);
	  } else {
	    i++;
	  }
	}
      }
    }
  } while (anything_changed);
}


// Split a track in two.  The existing track is associated with the closest
// new segment.  If a split-off part has < 2 points, it is just discarded.
// Any discarded points are effectively removed from the data we are fitting
// because we only fit to the points for segments currently in the SegmentSet.
// The actual creation of the new track is delayed to associate_tracks (as
// normal).  At least for this associate cycle, any new track is not part of
// our segment set.
//
void Datmo::split_tracks (SegmentSet *sset, Segment *seg, unsigned int gap_pos) {
  ObjectTrack *bm = (ObjectTrack *)seg->best_match;
  if (seg->points.size() <= 2) {
    // Only has two points, so neither side is viable.
    ASSERT(seg->points.size() == 2, "segment w/ <2 points?");
    bm->best_match = NULL;
    _segments.remove(seg);
    ignore_points(seg->points);
    delete seg;
  } else if (gap_pos < 2) {
    // first chunk has only one point, just drop it.
    ASSERT(gap_pos == 1, "split w/ no points in first chunk?");
    ignore_point(seg->points[0]);
    seg->points.erase(seg->points.begin());
  } else if (seg->points.size() - gap_pos < 2) {
    // last chunk has only one point, just drop it.
    ignore_point(seg->points[gap_pos]);
    seg->points.resize(gap_pos);
  } else {
    vector<PointData> new_points;
    for (unsigned i = gap_pos; i < seg->points.size(); i++)
      new_points.push_back(seg->points[i]);
    seg->points.resize(gap_pos);
    Segment *new_seg = new Segment(seg->datmo, seg->scanner, new_points);
    _segments.push_back(new_seg);
    seg->shape_classify();
    new_seg->shape_classify();

    // If new_seg is better match for track than old, then give old
    // track to new seg.
    if (bm->closeness(new_seg) > bm->closeness(seg)) {
      new_seg->best_match = bm;
      bm->best_match = new_seg;
      new_seg->id = bm->id;
      seg->id = 0;
      seg->best_match = NULL;
      seg->split_from = bm;
    } else {
      new_seg->split_from = bm;
    }
    bm->ever_split = true;
  }
}

bool Datmo::model_segment_ok (SegmentSet *sset) {
  if ((int)sset->tracks.size() > _max_segment_set_size) {
    if (_scan_log) *_scan_log <<"(skipping " <<sset->tracks.size() <<") ";
    return false;
  }
  return true;
}

// Find segment sets, do initial fit, then check if any of the segments should
// be split due to an excessive internal gap.  If so, split, and refit.  Each
// track is split at most once on any tracker cycle.
void Datmo::model_segment () {
  find_segment_sets();

  if (_scan_log) *_scan_log <<"ssets: ";
  for (unsigned int sset_ix = 0; sset_ix < _segment_sets.size(); sset_ix++) {
    SegmentSet *sset = _segment_sets[sset_ix];
    if (sset->tracks.size() && model_segment_ok(sset)) {
      for (unsigned int i = 0; i < sset->raw_segs.size(); i++) {
	// Remove this seg from segments list, since points now in sset.
	Segment *seg = sset->raw_segs[i];
	_segments.remove(seg);
	delete seg;
      }

      if (_scan_log) *_scan_log <<sset->tracks.size() <<" ";
      // assign initial segment models for tracks.
      for (unsigned int i = 0; i < sset->tracks.size(); i++) {
	ObjectTrack *track = sset->tracks[i];
	_segments.push_back(new Segment(track));
      }

      // assign initial points to the closest segment.
      qsort(&sset->points.front(), sset->points.size(), sizeof(PointData),
	    pointdata_scan_order);
      Segment *prev_seg = NULL;
      unsigned int prev_ix = 0;
      for (unsigned int i = 0; i < sset->points.size(); i++) {
	Segment *cseg = closest_segment(prev_seg, prev_ix, sset->points[i].pos, sset);
	cseg->points.push_back(sset->points[i]);
	prev_seg = cseg;
	prev_ix = prev_seg->points.size();
      }

      // If seg intially has less than 2 points, delete it.
      for (unsigned int i = 0; i < sset->tracks.size(); i++) {
	ObjectTrack *track = sset->tracks[i];
	Segment *seg = (Segment *)(track->best_match);
	if (seg->points.size() < 2) {
	  track->best_match = NULL;
	  _segments.remove(seg);
	  ignore_points(seg->points);
	  delete seg;
	}
      }

      model_segment_1(sset);

      // Split any segments that need it (then refit.)
      bool did_split = false;
      for (unsigned int i = 0; i < sset->tracks.size(); i++) {
	Segment *seg = (Segment *)(sset->tracks[i]->best_match);
	if (seg) {
	  ASSERT(seg->kind != ObjectBase::DELETED, "deleted?");
	  ASSERT(seg->best_match->best_match == seg, "bad best_match");
	  unsigned int gap_pos;
	  double gap_size;
	  seg->find_gap(gap_pos, gap_size);
	  if (gap_size > 1) {
	    split_tracks(sset, seg, gap_pos);
	    did_split = true;
	  }
	}
      }
      if (did_split)
	model_segment_1(sset);
    }
  }
}


//// Datmo::associate_tracks


// Initialize the overlap lists in all the tracks and segments by doing the
// N * M pairwise overlap tests.
void Datmo::find_overlaps () {
  for (list<ObjectTrack *>::iterator track_it = _tracks.begin();
       track_it != _tracks.end();
       track_it++) {
    ObjectTrack *track = *track_it;
    track->overlap.clear();
    track->best_match = NULL;
    for (list<Segment *>::iterator seg_it = _segments.begin();
	 seg_it != _segments.end();
	 seg_it++) {
      Segment *seg = *seg_it;
      if (track->overlapping(seg)) {
	track->overlap.push_back(seg);
	seg->overlap.push_back(track);
      }
    }
  }
}


// First do our associate pass, then create new tracks for unassociated
// segments.
void Datmo::associate_tracks () {
  {
    list<Segment *>::iterator next;
    ft_poll_before("Segment::init");
    for (list<Segment *>::iterator seg_it = _segments.begin();
	 seg_it != _segments.end();
	 seg_it = next) {
      Segment *seg = *seg_it;
      next = seg_it;
      next++;
      seg->init();
      if (ft_poll_after("Segment::init")) {
	_segments.erase(seg_it);
	if (seg->best_match && seg->best_match->best_match == (ObjectBase *)seg)
	  seg->best_match->best_match = NULL;
      }
    }
  }

  {
    ft_poll_before("match_and_associate");
    list<ObjectTrack *>::iterator next;
    for (list<ObjectTrack *>::iterator track_it = _tracks.begin();
	 track_it != _tracks.end();
	 track_it = next) {
      ObjectTrack *track = *track_it;
      next = track_it;
      next++;
      track->match_and_associate();
      if (ft_poll_after("match_and_associate")) {
	_tracks.erase(track_it);
	ft_kill_track(track);
      }
    }
  }

  // create new tracks from unassociated segments.
  for (list<Segment *>::iterator seg_it = _segments.begin();
       seg_it != _segments.end();
       seg_it++) {
    Segment *seg = *seg_it;
    if (!seg->best_match && seg->good_new_track()) {
      _tracks.push_back(new ObjectTrack(seg));
    }
  }
}


//// Datmo::add_scan

// Delete any tracks which are no longer supported by data.
void Datmo::reap_tracks () {
  list<ObjectTrack *>::iterator next;
  for (list<ObjectTrack *>::iterator track_it = _tracks.begin();
       track_it != _tracks.end();
       track_it = next) {
    ObjectTrack *track = *track_it;
    next = track_it;
    next++;
    if (_scan_time - track->last_associated_time
	> min((track->occlusion == ObjectTrack::TOTAL
		? _occluded_max_no_associate_time
		: _max_no_associate_time),
	      max(track->last_associated_time - track->create_time,
		  _min_no_associate_time))) {
      bool dwod = (!track->living_offspring()
		   && track->occlusion != ObjectTrack::TOTAL);
      if ((_print_moving_tracks && track->valid && track->moving)
	  || (_print_died_without_descendents && dwod)) {
	track->dump_time_and_id(cout);
	if (track->merged_with)
	  cout <<" merged with " <<track->merged_with <<".";
	else
	  cout <<" dead.";
	if (dwod)
	  cout <<" (no descendents)";
	cout <<endl;
      }
      LOG(track, "Track dead. "
	  <<(track->living_offspring() ? "(living offspring)" : ""));
      track->kind = ObjectTrack::DEAD_TRACK;
      _tracks.erase(track_it);
      _dead_tracks.push_back(track);
    }
  }
}


// Do retrospective validity test pass.  To bound the work for validation, we
// bound the number of tracks validated on each iteration to max_validate.
//
void Datmo::validate_tracks () {
  unsigned int nval = 0;
  unsigned int o_lastval = _last_validated;
  _last_validated = 0;
  unsigned int lastval = 0;
  ft_poll_before("validate");
  list<ObjectTrack *>::iterator next;
  for (list<ObjectTrack *>::iterator track_it = _tracks.begin();
       track_it != _tracks.end();
       track_it = next) {
    ObjectTrack *track = *track_it;
    next = track_it;
    next++;
    if (track->do_validate && (track->id > o_lastval)) {
      if (nval >= _max_validate) {
	_last_validated = lastval;
	// This is just here because I'm curious to know how much it happens.
#if 0
	if (_print_moving_tracks)
	  DEBUG("max_validate exceeded.");
#endif
	break;
      } else {
	track->validate();
	if (ft_poll_after("validate")) {
	  _tracks.erase(track_it);
	  ft_kill_track(track);
	}
	nval++;
	lastval = track->id;
      }
    }
  }
}


// Common action when track is killed due to float error.  Must erase from
// _tracks before this call.
void Datmo::ft_kill_track (ObjectTrack *track) {
  LOG(track, "Track killed for float error.");
  if (track->best_match && track->best_match->best_match == (ObjectBase *)track)
    track->best_match->best_match = NULL;
  track->kind = ObjectTrack::DEAD_TRACK;
  _dead_tracks.push_back(track);
  WARN("Track killed for float error: " <<track->id);
}


// Update all tracks for time increment.  This must be done before associating
// new data with any individual tracks.  It is assumed that the delta_t is the
// same across all tracks, which is a reasonable assumption, as long as we are
// happy with predicting the position only at the measurement times, and not
// trying to sync all tracks to the same instant.
void Datmo::update_tracks () {
  ObjectTrack::set_delta_t(_scan_time - _prev_scan_time);
  list<ObjectTrack *>::iterator next;
  for (list<ObjectTrack *>::iterator track_it = _tracks.begin();
       track_it != _tracks.end();
       track_it = next) {
    ObjectTrack *track = *track_it;
    next = track_it;
    next++;
    ft_poll_before("update");
    (*track_it)->update();
    if (ft_poll_after("update")) {
      _tracks.erase(track_it);
      ft_kill_track(track);
    }
  }
}


// Clear out information saved for visualization.  This is where segments and
// tracks are normally deleted, regardless of whether visualization is on or
// not.
void Datmo::begin_scans () {
  if (_ignored_points) _ignored_points->clear();

  // Delete dead tracks.
  for (list<ObjectTrack *>::iterator it = _dead_tracks.begin();
       it != _dead_tracks.end();
       it++)
    delete *it;
  _dead_tracks.clear();

  // Delete old segments.
  for (list<Segment *>::iterator seg_it = _segment_log.begin();
       seg_it != _segment_log.end();
       seg_it++) {
    Segment *seg = *seg_it;
    ASSERT(seg->kind == ObjectBase::SEGMENT, "not a segment?");
    delete seg;
  }
  _segment_log.clear();
}


// Do occlusion processing.
void Datmo::end_scans () {
  for (list<ObjectTrack *>::iterator track_it = _tracks.begin();
       track_it != _tracks.end();
       track_it++) {
    (*track_it)->update_occluded();
  }
}


// Main entry point for datmo.    
bool Datmo::add_scan (ScannerInfo *scanner) {
  if (!_inited) init();

  if (!import_data(scanner))
    return false;

  ft_poll_before("segment");
  segment();
  if (ft_poll_after("segment"))
    return false;

  update_tracks();
  find_overlaps();
  if (_model_based_segmentation)
    model_segment();
  associate_tracks();
  reap_tracks();
  validate_tracks();
  _iteration++;
  _segment_log.insert(_segment_log.end(),
		      _segments.begin(), _segments.end());
  if (_scan_log) *_scan_log <<endl;
  return true;
}


//// Debug dump stuff:


#define dump_tracks(tracks,type,out,track_id,dump_points) {\
  out.precision(8);\
  out <<"#Time "\
      <<_scan_time - _first_scan_time\
      <<", " <<tracks.size() <<" tracks:\n";\
  int track_ix = 0;\
  for (list<type *>::iterator track_it = tracks.begin();\
       track_it != tracks.end();\
       track_it++) {\
    if (!track_id || (*track_it)->id == track_id)\
      (*track_it)->dump(out, track_ix, dump_points);\
  }\
}


void Datmo::_dump_data () {
  {
    ofstream out("segs.dat");
    dump_tracks(_segments, Segment, out, 0, true);
  }
  {
    ofstream out("tracks.dat");
    dump_tracks(_tracks, ObjectTrack, out, 0, true);
  }
}

void dump_data () {
  gDatmo->_dump_data();
}
