#include "TrackLogger.h"
#include "utils/ConfigFile.h"


// Though not entirely robust, for our offline performance analysis, we assume
// that we log each and every scan at the specified scan rate (scans_per_sec.)
// datmo itself doesn't know or care about the scan rate.  This way a past
// state at a particular lag is a fixed index into the history.

// Parameter caches.
static int history_size;
static float scans_per_sec;
static float predict_step;
static const int max_horizons = 100;
static float test_horizons[max_horizons];
static int num_horizons; // number actually specified.

// horizons converted into cycle counts.
static int horizon_increments[max_horizons];

void TrackLogger::static_init (ConfigFile *config) {
  scans_per_sec = config->getFloat("scans_per_sec");
  predict_step = config->getFloat("predict_step");
  history_size = (int)(config->getFloat("history_duration") * scans_per_sec + 1);
  num_horizons = config->getFloats("test_horizons", test_horizons, max_horizons);
  for (int i = 0; i < num_horizons; i++) {
    horizon_increments[i] = (int)rint(test_horizons[i] * scans_per_sec);
    ASSERT(horizon_increments[i] < history_size,
	   "test horizon larger than history");
  }
}

TrackLogger::TrackLogger (ObjectTrack *track_arg) :
  track(track_arg),
  file(NULL)
{
}
 
TrackLogger::~TrackLogger () {
  if (file)
    delete file;
}

void TrackLogger::update () {
  // Enough history to test anything?
  if (true || (int)history.size() >= horizon_increments[0]) {
    if (!file) {
      char buf[80];
      snprintf(buf, sizeof(buf), "log/%d.test", track->id);
      file = new ofstream(buf);
      ASSERT(*file, "couldn't open " <<buf <<" for writing.");
    }
    file->precision(6);
    *file <<track->id <<" " // 1
	  <<(track->datmo->_scan_time - track->datmo->_first_scan_time) <<" " // 2
	  <<track->shape_class <<" " // 3
	  <<track->disoriented <<" " // 4
	  <<track->compact <<" " // 5
	  <<track->limiting_dv_dt <<" " // 6
	  <<track->valid <<" " // 7
	  <<track->moving <<" " // 8
	  <<track->do_validate <<" " // 9
	  <<track->associated_count <<" " // 10
	  <<track->split_from() <<" " // 11
	  <<track->living_offspring() <<" " // 12
	  <<track->merged_with <<" "; // 13
    Vec2d pos = track->position();
    *file <<pos[0] <<" " <<pos[1] <<" "; // 14, 15
    Vec2d vel = track->state_part(track->kf_motion.state, ObjectTrack::V_X);
    *file <<vel[0] <<" " <<vel[1] <<" "; // 16, 17
    *file <<track->velocity_correction[0] <<" "
	  <<track->velocity_correction[1] <<" "; // 18, 19
    for (int i = ObjectTrack::A_X; i <= ObjectTrack::A_Y; i++) // 20, 21
      *file <<track->kf_motion.state[i] <<" ";
    double theta, dum;
    track->get_theta(theta, dum);
    *file <<theta <<" " // 22
	  <<track->kf_rotation.state[ObjectTrack::V_THETA] <<" "; // 23

    for (int i = 0; i < num_horizons; i++) {
      int ix = history.size() - horizon_increments[i];
      if (ix >= 0)
	testOne(ix);
      else
	*file <<"0 0 "; // 24, 25...
    }
    *file <<endl;
  }

  if ((int)history.size() >= history_size)
    history.pop_front();

  ObjectTrack copy = *track;
  copy.log_file = NULL;
  copy.logger = NULL;
  history.push_back(copy);
}

// This works somewhat similarly to ObjectTrack::process_history, and borrows
// some of its internal mechanism.  The main difference is that we find the
// match error at only one time.
//
void TrackLogger::testOne (int index) {
  ObjectTrack copy = history[index];
  copy.log_file = NULL;
  copy.limiting_dv_dt = false;

  // Apply V correction (if any).
  ObjectTrack::set_state_part(copy.kf_motion.state, ObjectTrack::V_X,
			      copy.velocity());

  // Project state forward to now in steps no larger than predict_step.
  double back_time = (history.size()-index)/scans_per_sec;
  double t_offset = back_time;
  while (t_offset > predict_step) {
    copy.set_delta_t(predict_step);
    copy.update();
    t_offset -= predict_step;
  }
  copy.set_delta_t(predict_step);
  copy.update();

  // At least for now, compare to current estimate rather than observation, as
  // there may or not be any observation.  On the plus side, this should
  // result in somwhat less noise.  On the minus side, this could tend to
  // spuriously reduce the error because the same error processes could influence
  // the current estimate as well as the past estimate.  In practice, I don't
  // think this is a big effect with feature positions because the filter
  // response to position measurement is pretty fast.
  TrackHistory hist;

  // scan time is randomly hacked to make process_1_history not choke.
  hist.scan_time = track->datmo->_scan_time + back_time;
  hist.shape_class = track->shape_class;
  for (int i = 0; i < ObjectTrack::NUM_FEATURES; i++)
    hist.features[i] = track->features[i];

  double min_dist, rms_dist;
  ObjectTrack::alt_process_1_history(copy, hist, min_dist, rms_dist);

  *file <<min_dist <<" " <<rms_dist <<" ";
}
