/// Segment contains the state and methods related to segmentation.

#ifndef SEGMENT_H
#define SEGMENT_H

#include "ObjectBase.h"
class ObjectTrack;

class Segment : public ObjectBase {
  friend class PointLogger;

 public:
  // Create a segment from points.
  Segment (Datmo *datmo, ScannerInfo *scanner, const vector<PointData> &points);

  // Create a segment from a track as a model for model-based segmentation.
  // It has no points initially, but the same features as the track.
  Segment (ObjectBase *track);

  virtual ~Segment ();

  // Call once to init static vars.
  static void static_init ();

  // Call to set up or refine linear features.
  void shape_classify ();

  // Call on each segment before association.
  void init ();

  // Call this to test if we should make a new track.
  bool good_new_track ();

  void find_gap (unsigned int &gap_pos, double &gap_size);

  // Mean-square error of linear fit, adjusted by size factor.  Compares
  // directly to _line_tolerance^2.
  double squared_fit_quality;

  // Track that this segment appears to be split from.
  ObjectTrack *split_from;

  // Debug stuff.
  virtual void dump_state (ostream &out);
  void render ();

 private:
  // Segmentation.
  static PointVec *_fit_buf;
  static double *_fit_weight;
  static double *_fit_error;
  static double *_fit_error_sorted;

  void find_bounds ();

  // In a corner segment, the index of the "knuckle" point used as the split
  // point between the two sides.
  int corner_ix;

  void find_knuckle (int first, int last, int &max_ix);
  void fit_linear
    (int start, int end, double &sumsq, int &npoints, Vec2d &base, Vec2d &dir);
  void fit_linear_aux (int npoints, double &sumsq, Vec2d &base, Vec2d &dir);
  void fit_drop_outliers
    (const Vec2d &base, const Vec2d &norm, int full_npoints, int &npoints);
  void fill_fit_buf (int start, int npoints);
  void fit_side (int npoints, const Vec2d &big_pos, const Vec2d &dir_big,
		 Vec2d &norm, Vec2d &corner_pos, double &side_mag);
  void fit_corner (int big_start, int big_end, int small_start, int small_end,
		   Vec2d &corner_pos, Vec2d &big_pos, Vec2d &small_pos,
		   double &mean_square_err, bool &success);
  bool try_fit_corner (unsigned int eff_ignore, double mse_line,
		       const Vec2d &dir_line);
  const Vec2d &adjacent_point (int increment);
  void check_1_scan_angle
    (FeaturePoint &end, const FeaturePoint &other, int increment,
     const Vec2d &dir, const Vec2d &norm,
     unsigned int end_ix, unsigned int npoints,
     double &along_sigma, double &lateral_sigma) ;
  void maybe_extend (int feature_ix, Vec2d &end, const Vec2d &other);
  void find_center ();
  void check_scan_angle ();
  void detect_occlusion ();
};

#endif
