// ScannerInfo is a quasi-abstract bundle of stuff related to scanner creation
// and calibration data.

#ifndef SCANNERINFO_H
#define SCANNERINFO_H

#include <ConfigSource/ConfigSource.h>
#include <ConfigSource/Module.h>
#include <LineScanner/LineScanner.h>
#include <utils/Linear.h>
#include <utils/Vector.h>

class PointData {
public:
  // Position of this point.
  utils::Vec2d pos;

  // True if this point may be a false object edge due to either a
  // shadowing object in front or due to adacent missing return.
  bool occluded;

  // Raw range value.
  float range;

  // Index in scan of this point.
  unsigned int scan_pos;
};


class ScannerInfo {
 public:
  // String scanner name.
  const char *name;

  // Maximum size in points possible for scan (parameter cache.)  This is in
  // common to all scanners, so is the max across all scanners.
  static unsigned int max_scan_size;

  // scan source object.
  LineScanner *ls;

  // Location of scanner in vehicle coordinates.
  double x_offset;
  double y_offset;
  double yaw_offset;
  

  // The remaining fields are various data that must be per-scanner.  These
  // fields are used both in datmomod and datmo.

  // Location of scanner in smooth (world) coordinates at end of current scan.
  utils::Vec2d location;

  // Timestamp of start of last scan.
  double last_time;

  // Timestamp of scan just read.  If cur_time != last_time, then there is new
  // data to be processed.
  double cur_time;

  // Buffer of last read scanner data.
  utils::Vector<LineScanElem> lse;
  
  // PointData for each point in last scan, converted to smooth coordinates,
  // allocated at max_scan_size.  num_points is the actual number of points
  // (same as lse.numElems().)
  unsigned int num_points;
  PointData *points;

  // Set to 1 if point was ignored on last scan (not put in any segment) due
  // to there being too few points in the segment, being in an ignore region,
  // or whatever.
  unsigned char *ignored;

  // Get next line into lse, updating cur_time.
  bool getLine ();

  double resolution () {
    return fabs(ls->resolution());
  }

  // Convert a bearing angle into the bearing bin index that bearing
  // would appear in.  If -1, the bearing angle does not fall in the
  // field of view.
  int bearing_index (float bearing);

  ScannerInfo (Module *module, const char *name_arg, ConfigSource *config);
  ~ScannerInfo ();
};

#endif
