#include "ScannerInfo.h"
#include <utils/Linear.h>
#include <utils/ConfigFile.h>
#include <MapBase/MapBase.h>
#include "datmomod.h"
#include <unistd.h>

unsigned int ScannerInfo::max_scan_size;

ScannerInfo::ScannerInfo
    (Module *module, const char *name_arg, ConfigSource *config) :
  name(name_arg),
  last_time(-1),
  cur_time(-1),
  num_points(0)
{
  max_scan_size = config->getInt("max_scan_size");
  char spec_buf[256];
  snprintf(spec_buf, sizeof(spec_buf), "%s_spec", name);
  char pose_buf[256];
  snprintf(pose_buf, sizeof(pose_buf), "%s_pose", name);

  const char *spec = config->getString(spec_buf);
  ls = module->create<LineScanner>(spec);
  ASSERT(ls, "Spec " <<spec <<" failed to create a valid LineScanner.");

  ConfigFile transform_struct;
  config->getStruct(pose_buf, transform_struct);
  Transform transform;
  transform.set(transform_struct);
  Vec3d trans;
  Rotation rot;
  Vec3d axis;
  double angle;
  // ### this is only correct as long as the only scanner rotation is about
  // the yaw axis.  Of course, if there is a 3d transform, we can't reduce it
  // to a 2d one.  But the failure may be particularly interesting here
  // because this is not extracting the yaw part of the rotation...  It would
  // be somewhat more correct to use Transform::toRPY, but this works for now.
  transform.getTransform(trans, rot);
  rot.getValue(axis, angle);
  x_offset = trans[0];
  y_offset = trans[1];
  yaw_offset = angle;
  if (axis[2] < 0)
    yaw_offset = -yaw_offset;

  lse.setNumElems(max_scan_size);
  points = new PointData[max_scan_size];
  ignored = new (unsigned char)[max_scan_size];
  memset(ignored, 0, max_scan_size);
}

ScannerInfo::~ScannerInfo () {
  delete [] points;
  delete [] ignored;
}

bool ScannerInfo::getLine () {
  if (ls->getLine(lse)) {
    // If scanner status is bad or no scan elements, treat this like read
    // suceeded but we didn't get any new data.
    if (ls->getStatus() >= 0 && lse.numElems() > 0) {
      cur_time = lse[0].secs +  lse[0].usecs * 1e-6;
      ASSERT(lse.numElems() <= (int)max_scan_size,
	     "Actual num elements " <<lse.numElems()
	     <<" exceeds max_scan_size.");
    }
    return true;
  } else {
    return false;
  }
}

// Really this or something similar should be exported from
// LineScanner.  As it is, the various LineScanners have internally
// the information about the field-of-view which we have to infer from
// the data returned.
int ScannerInfo::bearing_index (float bearing) {
  if (lse.numElems() < 2 || cur_time == -1) return -1;
  float start_angle = lse[0].bearing;
  float end_angle = lse[lse.numElems() - 1].bearing;

  // modulo sign, this is should be the same as resolution().  It
  // seems that on startup we sometimes have bad data in here, which
  // can cause divide by zero.
  float incr = (end_angle - start_angle) / (lse.numElems() - 1);
  if (fabs(fabs(incr) - fabs(ls->resolution())) > 0.001) {
    WARN("Bad start end/angle on " <<name <<", got " <<incr <<", expected "
	 <<ls->resolution());
    return -1;
  }
  
  // A little tricky, but I think this works even if incr is negative.
  int res = (int)rint((bearing - start_angle) / incr);
  if (res < 0 || res >= lse.numElems())
    return -1;
  else
    return res;
}
