#include "PointLogger.h"
#include "utils/ConfigFile.h"


// Parameter caches.
static double sample_interval;

void PointLogger::static_init (ConfigFile *config) {
  sample_interval = config->getDouble("sample_interval");
}

PointLogger::PointLogger (ObjectTrack *track_arg) :
  track(track_arg),
  file(NULL),
  point_ix(0),
  track_file(NULL),
  track_ix(0),
  last_output_time(0)
{
}
 
PointLogger::~PointLogger () {
  if (file) delete file;
  if (track_file) delete track_file;
}

static int pointvec_order(const void *x, const void *y) {
  return (*(PointVec *)x)[0] < (*(PointVec *)y)[0] ? -1 : 1;
}

// Convert 2d point data for a segment into something resembling a 1d
// time series that we can do frequency domain analysis on.  The
// general idea is to make position along the line be X and deviation
// from the line be Y.  We fold the two sides of the corner to one
// line, so we need to offset by the length of the top arm (center to
// last.)
//
// X values may not be monotonic, especially near the corner.
// Non-monotonicity can also happen with a poor line fit.  In either
// case, we enforce monotonicity by dropping points with lower X.
// Sorting the points would tend to create HF content not really
// there.  Dropping questionable points tends more to smooth out the
// transition.
//
void PointLogger::points_to_1d (Segment *seg, vector<PointVec> &res) {
  PointMat rot;
  PointVec min_b, max_b;
  res.clear();
  ASSERT(seg->shape_class == ObjectBase::LINE
	 || seg->shape_class == ObjectBase::CORNER,
	 "Eh?");
  bool is_line = (seg->shape_class == ObjectBase::LINE);
  set_cols(rot, seg->direction, seg->norm_dir);
  rot.transpose(rot); // inverse
  PointVec fpos = rot * seg->features[ObjectBase::FIRST].state;
  PointVec lpos = rot * seg->features[ObjectBase::LAST].state;
  PointVec cpos = rot * seg->features[ObjectBase::CORNER_F].state;

  // If a line, the features should now lie on a horizonal line, with
  // FIRST farther from the origin.  If a corner, FIRST and LAST
  // should be the bottom right and top left corners, respectively,
  // with CORNER_F the bottom left.
  PointVec origin = is_line ? lpos : cpos;
  double last_x = -1e10;
  bool need_sort = false;
  for (int i = seg->points.size() - 1; i >= 0 ; i--) {
    PointVec ppos = (rot * seg->points[i].pos) - origin;
    PointVec rp;
    if (is_line) {
      rp = ppos;
    } else {
      double offset = lpos[1] - cpos[1];
      if (ppos[0] > ppos[1]) // i < seg->corner_ix
	rp = PointVec(ppos[0] + offset, ppos[1]);
      else
	rp = PointVec(-ppos[1] + offset, ppos[0]);
    }
#if 1
    if (last_x < rp[0]) {
      last_x = rp[0];
      res.push_back(rp);
    }
#else
    if (last_x >= rp[0]) need_sort = true;
    last_x = rp[0];
    res.push_back(rp);
#endif
  }
  if (need_sort)
    qsort(&res[0], res.size(), sizeof(PointVec),
	  pointvec_order);
  
  if (res.size() >= 2) {
    unsigned int ix_mid = res.size() / 2;
    double x_median = (res.size() & 1)
      ? res[ix_mid][0]
      : (res[ix_mid - 1][0] + res[ix_mid][0]) * 0.5;
    for (unsigned int i = 0; i < res.size(); i++)
      res[i][0] -= x_median;
  }
}

void PointLogger::update () {
  if (track->best_match
      && track->datmo->_scan_time - last_output_time > sample_interval) {
    if (!file) {
      char buf[80];
      snprintf(buf, sizeof(buf), "log/%d.points", track->id);
      file = new ofstream(buf);
      ASSERT(*file, "couldn't open " <<buf <<" for writing.");
    }
    vector<PointVec> res;
    Segment *seg = (Segment *)(track->best_match);
    points_to_1d(seg, res);
    last_output_time = track->datmo->_scan_time;
    unsigned int np = res.size();
    *file <<"# " <<track->datmo->_scan_time - track->datmo->_first_scan_time <<" "
	  <<np <<" " <<point_ix++ <<endl;
    for (unsigned int i = 0; i < res.size(); i++)
      *file <<res[i][0] <<" " <<res[i][1] <<endl;
    *file <<endl <<endl;

    if (!track_file) {
      char buf[80];
      snprintf(buf, sizeof(buf), "log/%d.track", track->id);
      track_file = new ofstream(buf);
      ASSERT(*track_file, "couldn't open " <<buf <<" for writing.");
      // create dummy "update" entry to keep in sync with full dump format.
      track->dump(*track_file, track_ix, false);
    }
    seg->dump(*track_file, track_ix, true);
    track->dump(*track_file, track_ix, false);
  }
}
