/// The ObjectTrack contains the state and methods related to tracks and tracking.

#ifndef OBJECTTRACK_H
#define OBJECTTRACK_H

class ObjectTrack;
class TrackHistory;
class HistoryStats;
class SegmentSet;

#include <deque>
#include "KalmanFilter.h"
#include "utils/MedianFilter.h"
#include "ObjectBase.h"
#include "Segment.h"

static const int ROT_FILTER_ORDER = 3;
typedef LinearVector<double, ROT_FILTER_ORDER> RotFilterVec;
typedef SquareMatrix<double, ROT_FILTER_ORDER> RotFilterMat;

// Kalman filter types.
typedef KalmanFilter<double, FILTER_ORDER, POINT_ORDER> KFLinear;
typedef KalmanFilter<double, ROT_FILTER_ORDER, POINT_ORDER> KFRotation;

class ObjectTrack : public ObjectBase {
  friend class TrackLogger;
  friend class PointLogger;
  friend class Datmo;
  friend class datmomod;

 public:
  ObjectTrack (Segment *seg);

  // Copy constructor creates a copy for running predictions, with many fields
  // such as the log_file being restored to the default initial value.
  ObjectTrack (const ObjectTrack &track);

  virtual ObjectTrack::~ObjectTrack ();

  // Timestamp for track creation.
  double create_time;

  // Timestamp from the last scan that we associated data with this
  // track.  Used to time out dead tracks.
  double last_associated_time;

  // Actual number of cycles this track has had data associated.
  int associated_count;

  // Max of associated_count for features currently contributing the motion
  // estimate.  Max is computed in this slot during associate(), so may be
  // spuriously low then.  Forced to 1 for COMPLEX tracks. 
  int feature_associated_count;
  
  // Vector listing IDs of all tracks we are split from, starting from
  // oldest.  This is sorted by increasing track ID because the split track
  // must necessarily be created after the parent track. 
  vector<unsigned int> split_from_vec;

  // Track ID of track that this track split from.  0 if not a split.
  unsigned int split_from () {
    return split_from_vec.size() ? split_from_vec.back() : 0;
  };

  // Return true if this track has any live tracks that split from it,
  // including indirectly via multiple splits or if this track is split
  // (possibly indirectly) from any track that is still alive.
  bool living_offspring ();

  // Track ID of track that this track merged with.  0 if not merged.
  unsigned int merged_with;

  // valid is true if we feel that we have a good velocity estimate on this
  // track.  moving is true if the track appears to be moving.  A track will
  // never be moving but not valid, so there are these three combinations:
  //   valid && !moving: definitely not moving
  //    valid && moving: definitely moving
  //  !valid && !moving: not sure if moving or not.
  bool valid;
  bool moving;

  // True if the object was ever moving (moving ever set.)
  bool ever_moving;

  // Call once to init static vars.
  static void static_init ();

  // Call before update to set the delta_t on this iteration.
  static void set_delta_t (double delta_t);

  // The time update (prediction) step in the filter.
  void update ();

  // Find best segment for this track and associate with it.
  void match_and_associate ();

  // Retrospective validity test.
  void validate ();

  // Call once per group of scans on all scanners to update occluded status of
  // tracks not successfully associated.
  void update_occluded ();

  // Accessor functions for track state components.
  Vec2d position () const {
    return features[CENTER].pos();
  }
    
  Mat2d position_covariance () {
    PointMat sub;
    kf_center.get_submatrix(kf_center.covariance(), sub);
    return Mat2d((double (*)[2])sub.getValue());
  }

  Vec2d velocity () const {
    return state_part(kf_motion.state, V_X) + velocity_correction;
  }

  PointMat velocity_covariance_sq () {
    PointMat sub;
    kf_motion.get_submatrix(kf_motion.covariance(), sub, V_X);
    if (sub.at(0, 0) + sub.at(1, 1)
	< history_v_covariance.at(0, 0) + history_v_covariance.at(1, 1))
      sub = history_v_covariance;
    return sub;
  }

  Mat2d velocity_covariance () {
    return Mat2d((double (*)[2])(velocity_covariance_sq().getValue()));
  }

  Vec2d acceleration () const {
    return state_part(kf_motion.state, A_X);
  }

  Mat2d acceleration_covariance () {
    PointMat sub;
    kf_center.get_submatrix(kf_motion.covariance(), sub, A_X);
    return Mat2d((double (*)[2])sub.getValue());
  }

  Vec2d incremental_motion () {
    return state_part(kf_motion.state, P_X);
  }

  Mat2d incremental_motion_covariance () {
    PointMat sub;
    kf_center.get_submatrix(kf_motion.covariance(), sub, P_X);
    return Mat2d((double (*)[2])sub.getValue());
  }

  // Get heading and heading variance.  Use whichever of THETA_HEADING,
  // THETA_ORIENTATION has lower variance.
  void get_theta (double &theta, double &v_theta) {
    double cov_orient = kf_rotation.covariance()
      .at(THETA_ORIENTATION, THETA_ORIENTATION);
    double cov_head = kf_rotation.covariance().at(THETA_HEADING, THETA_HEADING);
    if (cov_orient < cov_head) {
      theta = kf_rotation.state[THETA_ORIENTATION];
      v_theta = cov_orient;
    } else {
      theta = kf_rotation.state[THETA_HEADING];
      v_theta = cov_head;
    }
  }

  // Debug visualization method.
  void render ();

  // If logging data, the log file for this track.
  ostream *log_file;

  // Number of data blocks written to file (delimited by \n\n.)
  int log_index;

  // Print time and track ID to out for status logging.
  void dump_time_and_id (ostream &out);

  // If logging path, this is the path buffer.
  deque<PointVec> *path;

  // Time of path->back()
  double last_path_time;

  // debug/visualization.
  int color16 ();

 private:
  // delta_t increment set by set_delta_t to set the amount by which dynamic
  // model advances.  Also implicit in static vars, e.g. phi.
  static double update_delta_t;

  // Time increment from previous association to current association.  This
  // will often be larger than update_delta_t.
  double associate_delta_t;

  // Indices in state vector.
  static const int P_X = 0;
  static const int P_Y = 1;
  static const int V_X = 2;
  static const int V_Y = 3;
  static const int A_X = 4;
  static const int A_Y = 5;

  static Vec2d state_part (const FilterVec &state, int ix) {
    return Vec2d(state[ix], state[ix+1]);
  }

  static void set_state_part (FilterVec &state, int ix, const Vec2d &part) {
    state[ix] = part[0];
    state[ix+1] = part[1];
  }

  // Filter for relative motion and velocity.  The relative motion is the
  // aggregate motion of the (non-center) features since track creation.
  KFLinear kf_motion;

  // Filter for center position.  We don't actually use the state of this
  // filter, only its covariance.  The position of the center feature is the
  // track position.
  KFLinear kf_center;

  // Indices in rotational state.  THETA_xxx is what we think the
  // tracked object's yaw angle is.  V_THETA is the angular velocity.
  //
  // THETA_ORIENTATION is derived from the orientation of the track's
  // lines, and can easily be off by a multiple of 90 degrees.
  // THETA_ORIENTATION may jump by a multiple of PI/2 if we change our
  // idea of which side is the track front, however this doesn't have
  // any effect on the V_THETA.
  //
  // THETA_HEADING is derived from the direction of our linear
  // velocity vector.  This is an independent estimate of heading
  // available when we are moving.  It may differ significantly from
  // THETA_ORIENTATION due the unmodelled steering geometry, and
  // perhaps other factors.  However, the change in THETA_HEADING is a
  // reasonable angular velocity estimate, and that is all we care
  // about.
  //
  // The THETA_xxx are not canonicalized r.e. 2 PI.  If the object
  // rotates past +/- 180 degrees, the rotation just keeps going up or
  // down.  In principle, if the object rotates many times, this can
  // become arbitrarily large, though this is unlikely to be seen in
  // practice.
  //
  static const int THETA_ORIENTATION = 0;
  static const int THETA_HEADING = 1;
  static const int V_THETA = 2;

  // Filter for angular motion.
  KFRotation kf_rotation;

  // Add this to kf_rotation.state[THETA_ORIENTATION] to get the angle
  // of our direction vector.  Always a multiple of PI/2.  This
  // changes as our view moves from one side of the object to the
  // next.
  double heading_offset;

  // Set in the parent track of a possible split.  Necessary but not
  // sufficient for this track to have been split.
  bool ever_split;

  // If true, this track is potentially moving, and we need to do
  // ObjecTrack::validate.
  bool do_validate;

  // Buffer of track history data.  A pointer so that we can copy an
  // ObjectTrack without having to copy all the history.  NULL in
  // segments.
  deque<TrackHistory> *history;

  // Time this track was last validated in ObjectTrack::validate.
  // Used to determine if there is any new history data, so validation
  // needs to be done.
  double last_validated_time;

  // Velocity correction derived from history data.  Add this to KF
  // estimate.  This is kept seperate from the KF state so as not to interfere
  // with acceleration estimation.  In theory we could also fold the
  // correction info the KF by treating the corrected velocity as a velocity
  // measurement, however the way that we use KalmanFilter.h this is not easy
  // because the measured state always has to be at the start of the state
  // vector.
  PointVec velocity_correction;

  // Estimate of velocity error covariance derived from history data.
  PointMat history_v_covariance;

  // These filters smooth the inputs to the history validity test to make it
  // less noisy.
  static const int history_filter_order = 21;
  MedianFilter<double, history_filter_order> fixed_info2;
  MedianFilter<double, history_filter_order> fixed_sigma2;
  MedianFilter<double, history_filter_order> moving_info2;
  MedianFilter<double, history_filter_order> moving_sigma2;

  // True if we limited dv/dt on the last update, thus the filter is slew-rate
  // limiting and the KF loop is effectively open.  When limiting dv_dt, we
  // effectively hold the velocity covariance constant.
  bool limiting_dv_dt;

  // Sum of kf_motion.covariance().at(i, i) for V_X, V_Y as of the last time
  // that we did *not* limit dv_dt.  This is the goal at which we hold this
  // value during the limiting period.
  double old_v_cov;

  // The R_k (measurement noise) and innovation for the current measurement
  // cycle.  This info is accumulated here because the individual parts of the
  // innovation are computed in two seperate locations that don't always get
  // executed.
  PointMat R_k_rotation;
  RotFilterVec innov_rotation;

  // Segment set for this track, or NULL if none.
  SegmentSet *segment_set;

  void update_state_only ();
  void update_aux (FilterMat &phi, bool update_residue, bool update_points);

  // Tracking
  void new_track (Segment *seg);
  void process_dv_dt_limiting ();
  void update_residue (FeaturePoint &feature, const FeaturePoint &measurement,
		       const PointVec &innovation);
  PointVec feature_normal (int f_ix);
  bool maybe_reset
    (int f_ix, Segment *seg, KFLinear &filter, bool motion_feature, bool reset,
     bool ripe, const FilterVec &state_inc);
  void associate_feature (int f_ix, Segment *seg, KFLinear &filter,
			  bool motion_feature, bool reset);
  void set_direction ();
  void check_direction ();
  void associate_rotation ();
  void measure_rotation (double d_theta, bool reset);
  static bool find_match_dir
    (ObjectBase *line, ObjectBase *corner, double &d_theta, double &offset_incr,
     int &corner_ix, int &match_ix);
  void reset_rotation (Segment *seg);
  void measure (Segment *seg, bool reset);
  bool limit_d_dt (ObjectTrack *save_state, int ix, double limit, const char *what);
  void limit_acceleration ();
  void test_motion (bool &motion_significant, bool &fast_enough);
  void check_moving ();
  void heading_from_motion ();
  void limit_d_omega_dt (ObjectTrack *save_state);
  bool associate (Segment *seg, bool reset);
  void fail_associate (Segment *match);
  double closeness (Segment *seg);
  void match_and_associate_aux ();

  // History analysis.
  void find_corresponding
    (const TrackHistory &hist, int c_ix[NUM_FEATURES]);

  void process_1_history (/* const */ ObjectTrack *copy, const TrackHistory &hist,
			  ofstream *vlog, int &vlog_ix, HistoryStats &stats);
  void process_history (bool assume_fixed, ofstream *vlog, int &vlog_ix,
			HistoryStats &stats);

  // Debug
  void dump_TrackHistory (ostream &out, int &counter, const TrackHistory &hist);
  void predict_positions ();
  virtual void dump_state (ostream &out);

  // Number of future positions to predict for visualization.
  static const int NUM_PREDICT = 20;
  static Vec2d predictions[NUM_PREDICT];

  // Used for offline data analysis.  If null, this is suppressed.
  TrackLogger *logger;

  // For offline texture analysis.  If null, this is suppressed.
  PointLogger *point_logger;

  static void ObjectTrack::alt_process_1_history
    (/* const */ ObjectTrack &copy, const TrackHistory &hist,
     double &min_dist, double &rms_dist);
};


// Summary of a segment (measurement) associated with a track, logged for
// retrospective validation of motion estimate.
struct TrackHistory {
  double scan_time;
  ObjectTrack::TrackShape shape_class;
  FeaturePoint features[ObjectTrack::NUM_FEATURES];
};

// Used during history analysis to summarize results.
class HistoryStats {
 public:
  HistoryStats () {
    num_sumsq = num_mean = 0;
    sumsq = 0;
  }

  // The sum of the information across all of the features in correspondence,
  // and the eigenvalues of this matrix.
  PointMat info_sum;
  double info_eigenvalues[2];

  // The mean_square Mahalanobis distance, and the values used to compute it.
  double mean_square;
  int num_sumsq;
  double sumsq;

  // The mean and covariance of the velocity error.  num_mean is the number of
  // values that statistics are based on.
  int num_mean;
  PointVec mean;
  PointMat covariance;
};


// Datastructure used during segmentation to represent a group of nearby
// tracks.
struct SegmentSet {
  // ID of this segment set.
  int id;

  // Tracks in this set.  Normally the best_match is the segment that
  // will be associated, but if there is no such segment, then the
  // best_match is NULL.
  vector<ObjectTrack *> tracks;

  // Initial points in the segment set.  This vector is not used or
  // updated after the K-means iteration is initialized.
  vector<PointData> points;

  // The raw segments providing points to this set.  If we decide to
  // proceed with model-based segmentation of this set, then these
  // segments must be deleted from datmo::_segments.
  vector<Segment *> raw_segs;
};


#define LOG(track,stuff) {\
  if ((track)->log_file) {\
     (track)->log_file->precision(6);\
     *(track)->log_file <<"# " \
       <<(track)->datmo->_scan_time - (track)->datmo->_first_scan_time \
       <<" " <<stuff <<endl;\
  }\
}

#endif
