#include "ObjectTrack.h"
#include "TrackLogger.h"
#include "PointLogger.h"
#include "params.h"
#include "datmo.h"
#include <fstream>
#include <float.h>

// Multiply normal angle noise by this to get value for angle that is not known.
static const double unknown_angle = 100;

ObjectTrack::ObjectTrack (Segment *seg) :
  ObjectBase(seg->datmo, seg->scanner, TRACK),
  create_time(datmo->_scan_time),
  last_associated_time(datmo->_scan_time),
  associated_count(1),
  feature_associated_count(1),
  merged_with(0),
  valid(false),
  moving(false),
  ever_moving(false),
  log_file(NULL),
  log_index(0),
  path(NULL),
  last_path_time(0),
  associate_delta_t(0),
  heading_offset(0),
  ever_split(false),
  do_validate(false),
  history(NULL),
  last_validated_time(0),
  limiting_dv_dt(false),
  old_v_cov(0),
  segment_set(NULL),
  logger(NULL),
  point_logger(NULL)
{
  copy_features(seg);
  seg->id = id = datmo->_next_track_id++;
  points = seg->points;
  best_match = seg;
  seg->best_match = this;

  if (_log_format && _log_format[0]) {
    char buf[80];
    snprintf(buf, sizeof(buf), _log_format, id);
    log_file = new ofstream(buf);
    ASSERT(*log_file, "couldn't open " <<buf <<" for writing.");
  } else {
    ASSERT(!log_file, "Non-null log file?");
  }

  if (_path_size > 0)
    path = new deque<PointVec>;

  history = new deque<TrackHistory>;

  if (_tracklogger_configured)
    logger = new TrackLogger(this);

  if (_pointlogger_configured)
    point_logger = new PointLogger(this);

  new_track(seg);
}

ObjectTrack::~ObjectTrack () {
  if (kind != DELETED) {
    if (log_file) {
      delete log_file;
      log_file = NULL;
    }
    if (path) {
      delete path;
      path = NULL;
    }
    if (history) {
      delete history;
      history = NULL;
    }
    if (logger) {
      delete logger;
      logger = NULL;
    }
    if (point_logger) {
      delete point_logger;
      point_logger = NULL;
    }
    kind = DELETED;
  }
}

// This copy constuctor is intended to be used in places where we want to make
// a copy of a track so that we can extrapolate its motion without affecting
// the actual track state.  We deliberately don't copy various track state
// that is related to tracking the modifications of the actual track:
// log_file, path, history, logger, etc.  We also don't copy the points, which
// is more questionable, but saves some CPU.
//
ObjectTrack::ObjectTrack (const ObjectTrack &track) :
  ObjectBase(track.datmo, track.scanner, TRACK),
  create_time(track.create_time),
  last_associated_time(track.last_associated_time),
  associated_count(track.associated_count),
  feature_associated_count(track.feature_associated_count),
  merged_with(track.merged_with),
  valid(track.valid),
  moving(track.moving),
  ever_moving(track.ever_moving),
  log_file(NULL),
  log_index(0),
  path(NULL),
  last_path_time(0),
  associate_delta_t(track.associate_delta_t),
  heading_offset(track.heading_offset),
  ever_split(track.ever_split),
  do_validate(track.do_validate),
  history(NULL),
  last_validated_time(0),
  limiting_dv_dt(track.limiting_dv_dt),
  old_v_cov(track.old_v_cov),
  logger(NULL),
  point_logger(NULL)
{
  copy_features(&track);
  kf_motion = track.kf_motion;
  kf_center = track.kf_center;
  kf_rotation = track.kf_rotation;
  velocity_correction = track.velocity_correction;
}


/// ObjectTrack:: static_init, init, new_track, update, associate_feature
//
// These methods implement the tracking Kalman filter.

// State transition function (dynamic model.)  This is a function of
// the delta T and the angular velocity, and is updated on each
// iteration.  This is a constant linear acceleration, constant
// angular velocity model.
// 
// Phi is the product of the linear system and a rotation.  The
// rotation rotates the linear velocity and acceleration according to
// the incremental rotation.
//
// Since the rotation part depends on the individual track angular velocity,
// it must be computed seperately for each track, whereas the translational
// part depends only on update_delta_t, which is (currently) the same for all
// tracks.  So we compute the translational part in phi_linear once per
// iteration in set_delta_t, and the rotation part in update for each track.
/*
  {1, 0, update_delta_t, 0, 0, 0,
   0, 1, 0, update_delta_t, 0, 0,
   0, 0, 1, 0, update_delta_t, 0,
   0, 0, 0, 1, 0, update_delta_t,
   0, 0, 0, 0, 1, 0,
   0, 0, 0, 0, 0, 1}

   cdt = cos(update_delta_t * omega);
   sdt = sin(update_delta_t * omega);

  {1, 0, 0, 0, 0, 0,
   0, 1, 0, 0, 0, 0,
   0, 0, cdt, -sdt, 0, 0,
   0, 0, sdt, cdt, 0, 0,
   0, 0, 0, 0, cdt, -sdt,
   0, 0, 0, 0, sdt, cdt}
*/
static FilterMat phi_linear;

// The process noise covariance.  Computed from process noise parameters,
// normalized to 1 second.
static FilterMat Q;

// Scaled for current update_delta_t.
static FilterMat Lambda_Q_Lambda_transpose;

// Rotational KF matrices.

// phi_rot looks like:
/*
  {1, 0, update_delta_t,
   0, 1, update_delta_t,
   0, 0, 1}
*/
static RotFilterMat phi_rot;
static RotFilterMat Q_rot;
static RotFilterMat Lambda_Q_Lambda_transpose_rot;

// Init ObjectTrack static vars.
void ObjectTrack::static_init () {
  for (int i = P_X; i <= P_Y; i++)
    Q.at(i, i) = fsqr(_process_position_noise);

  for (int i = V_X; i <= V_Y; i++)
    Q.at(i, i) = fsqr(_process_velocity_noise);

  for (int i = A_X; i <= A_Y; i++)
    Q.at(i, i) = fsqr(_process_acceleration_noise);

  for (int i = 0; i < FILTER_ORDER; i++)
    phi_linear.at(i, i) = 1;

  for (int i = 0; i < ROT_FILTER_ORDER; i++)
    phi_rot.at(i, i) = 1;

  Q_rot.at(THETA_ORIENTATION, THETA_ORIENTATION)
    = fsqr(_process_angular_position_noise);
  Q_rot.at(THETA_HEADING, THETA_HEADING)
    = fsqr(_process_angular_position_noise);
  Q_rot.at(V_THETA, V_THETA) = fsqr(_process_angular_velocity_noise);
}


// This method does the non-storage-related part of construction of a track
// from a segment (in lieu of data association.)  Only kf_rotation needs its
// state initialized to non-zero.  zero is the correct init for kf_motion, and
// the state of kf_center is not used.
//
// If split_from, we add copy the state from the original track, add the
// initial velocity and acceleration sigmas to the covariance, and append the
// ID to split_from_vec.
//
void ObjectTrack::new_track (Segment *seg) {
  double v_var = fsqr(_initial_velocity_noise);
  double a_var = fsqr(_initial_acceleration_noise);
  ObjectTrack *split = seg->split_from;

  if (split) {
    split_from_vec = split->split_from_vec;
    split_from_vec.push_back(split->id);

    // So that the split_from_vec can't become arbitarily large.
    if (split_from_vec.size() > 100) {
      vector<unsigned int>::iterator it = split_from_vec.begin();
      advance(it, 50);
      split_from_vec.erase(split_from_vec.begin(), it);
    }

    if (path && split->ever_moving) *path = *(split->path);
    kf_motion = split->kf_motion;
    kf_center = split->kf_center;
    kf_rotation = split->kf_rotation;

    reset_rotation(seg);
    FilterMat cov = kf_motion.covariance();
    double v_var = fsqr(_initial_velocity_noise);
    for (int i = V_X; i <= V_Y; i++)
      cov.at(i, i) += v_var;

    double a_var = fsqr(_initial_acceleration_noise);
    for (int i = A_X; i <= A_Y; i++)
      cov.at(i, i) += a_var;
    kf_motion.set_covariance(cov);
    kf_center.set_covariance(cov);
  } else {
    // If not a linear feature, give an arbitrary large value to
    // THETA_ORIENTATION sigma.  Then when we get a heading reading,
    // we converge rapidly and without inferring a lot of velocity.
    // THETA_HEADING sigma is always large, since there is no way the
    // track can be moving at creation.
    double theta_noise = _angular_pos_noise * unknown_angle;
    if (shape_class == LINE || shape_class == CORNER) {
      // This is the right heading if we are now seeing the back, which
      // is basically a random guess.  It doesn't matter for angular
      // velocity estimation.
      double theta = atan2(norm_dir[1], norm_dir[0]);
      kf_rotation.state = RotFilterVec(theta, 0, 0);
      heading_offset = (normal(norm_dir).dot(direction) > 0)
	? M_PI/2
	: -M_PI/2;
      if (!disoriented)
	theta_noise = _angular_pos_noise;
    }

    RotFilterMat cov_rot;
    cov_rot.at(THETA_ORIENTATION, THETA_ORIENTATION) = fsqr(theta_noise);
    cov_rot.at(THETA_HEADING, THETA_HEADING)
      = fsqr(_angular_pos_noise * unknown_angle);
    cov_rot.at(V_THETA, V_THETA) = fsqr(_initial_angular_velocity_noise);
    kf_rotation.set_covariance(cov_rot);

    FilterMat cov;

    // We need some non-zero initial position covariance, exact value is not
    // critical, as process noise dominates.
    kf_motion.set_submatrix(cov, features[CENTER].residue_covariance);
  
    for (int i = V_X; i <= V_Y; i++)
      cov.at(i, i) = v_var;

    for (int i = A_X; i <= A_Y; i++)
      cov.at(i, i) = a_var;

    kf_motion.set_covariance(cov);
    kf_center.set_covariance(cov);
  }

  LOG(this, "new_track");
  // This dumps basically the same info twice to mimic update cycle so that
  // each time entry has 3 data blocks.
  if (log_file) {
    dump(*log_file, log_index, false);
    seg->dump(*log_file, log_index, true);
  }
}


// Where we save the delta_t currently set by set_delta_t (for
// ObjectTrack::update.)
double ObjectTrack::update_delta_t;

// Call this to set update_delta_t before calling update and associate.  Can be
// called with negative update_delta_t to predict backwards.
void ObjectTrack::set_delta_t (double delta_t_arg) {
  update_delta_t = delta_t_arg;

  Lambda_Q_Lambda_transpose = Q;
  Lambda_Q_Lambda_transpose *= fabs(update_delta_t);
  Lambda_Q_Lambda_transpose_rot = Q_rot;
  Lambda_Q_Lambda_transpose_rot *= fabs(update_delta_t);

  // Compute phi.  First fill in the linear system.
  for (int i = 0; i <= V_Y; i++)
    phi_linear.at(i, i + V_X) = update_delta_t;
  
  // Set up phi for rotation system.
  phi_rot.at(THETA_ORIENTATION, V_THETA) = update_delta_t;
  phi_rot.at(THETA_HEADING, V_THETA) = update_delta_t;
}


// If dv_dt_limiting, add excess velocity noise to keep the velocity
// covariance magnitude fixed at its former value.  We multiplicitivly scale
// up the velocity portion of the covariance.  This preserves any correlation
// in the velocity covariance.
void ObjectTrack::process_dv_dt_limiting () {
  const FilterMat &P_k_minus = kf_motion.covariance();
  double sum = 0;
  for (int i = V_X; i <= V_Y; i++)
    sum += P_k_minus.at(i, i);
  
  if (limiting_dv_dt) {
    double frac = old_v_cov/sum;
    if (frac > 1) {
      PointMat v_cov;
      kf_motion.get_submatrix(P_k_minus, v_cov, V_X);
      v_cov *= frac;
      FilterMat cov(P_k_minus);
      kf_motion.set_submatrix(cov, v_cov, V_X);
      kf_motion.set_covariance(cov);
      kf_center.set_covariance(cov);
    }
  } else {
    old_v_cov = sum;
  }
}


// ObjectTrack::update_aux computes the phi state transform matrix and updates
// the track state.
//
void ObjectTrack::update_aux (FilterMat &phi, bool update_residue,
			      bool update_points) 
{
  // Actually update for rotation only if there is significant motion or
  // non-negligible angular velocity.  This avoids considerble computation in
  // common cases.
  double omega = kf_rotation.state[V_THETA];
  const bool need_rot = (do_validate || omega > 1.0*M_PI/180);

  // Rotation matrix to rotate vectors from old to new angular orientation.
  // Ignore if not need_rot.
  PointMat d_theta_transform;

  if (need_rot) {
    // Now find d_theta_transform, the 2d rotation transform.
    double d_theta = omega * update_delta_t;
    set_rotation(d_theta_transform, d_theta);

    // Now fill in the rotation part of phi.
    FilterMat rot;
    rot.at(0, 0) = 1;
    rot.at(1, 1) = 1;
    kf_motion.set_submatrix(rot, d_theta_transform, V_X);
    kf_motion.set_submatrix(rot, d_theta_transform, A_X);

    // Compose linear system and rotation.  I don't have any theory
    // about the correct order to compose the two.  It shouldn't make
    // much difference.
    phi.mult(rot, phi);
  }

  // state update for track.
  FilterVec &x_hat = kf_motion.state;
  PointVec old_pos = state_part(x_hat, P_X);
  x_hat = phi * x_hat;
  kf_rotation.state = phi_rot * kf_rotation.state;
  set_direction();

  // Update all the features by the same relative motion.
  PointVec inc = state_part(x_hat, P_X) - old_pos;
  for (int i = 0; i < NUM_FEATURES; i++)
    features[i].state = features[i].state + inc;

  // Now rotate corners about the track center.
  if (need_rot) {
    const PointVec &cpos = features[CENTER].state;
    for (int i = FIRST; i <= CORNER_F; i++)
      if (valid_feature(i)) {
	PointVec disp = features[i].state - cpos;
	features[i].state = (d_theta_transform * disp) + cpos;
      }
    
    // Also rotate the measurement residue statistics (noise estimate)
    // for corners and center.
    if (update_residue) {
      PointMat dt3;
      dt3.transpose(d_theta_transform);
      for (int i = FIRST; i <= CENTER; i++)
	if (valid_feature(i)) {
	  features[i].residue_mean = d_theta_transform * features[i].residue_mean;
	  // rot * cov * rot'
	  PointMat &cov = features[i].residue_covariance;
	  cov.mult(cov, dt3);
	  cov.mult(d_theta_transform, cov);
	}
    }
  }

  if (update_points) {
    // Update all of the raw points as well to help data association.  Also, the
    // track points are an output, so we should keep them somewhat up to
    // date.  We could rotate too, but that would take longer, and it's not clear
    // it's necessary.
    for (unsigned int i = 0; i < points.size(); i++) 
      points[i].pos = points[i].pos + inc;
  }
}


// ObjectTrack::update_state_only is for prediction uses outside of the main KF.
// In comparison to ObjectTrack::update, we suppress update of the track
// covariances, feature residues and position of the raw points.
void ObjectTrack::update_state_only () {
  FilterMat phi(phi_linear);
  update_aux(phi, false, false);
}

    
// Update this track's features (predict state) and update estimate
// covariances.  We also do various debug logging.
void ObjectTrack::update () {
  FilterMat phi(phi_linear);
  update_aux(phi, true, true);

  // Do process covariance update.
  kf_motion.process(phi, Lambda_Q_Lambda_transpose);
  kf_center.process(phi, Lambda_Q_Lambda_transpose);
  kf_rotation.process(phi_rot, Lambda_Q_Lambda_transpose_rot);

  process_dv_dt_limiting();
  
  if (log_file) {
    LOG(this, "update");
    dump(*log_file, log_index, false);
  }

  if (path && datmo->_scan_time - last_path_time > _path_interval) {
    if (_path_incremental_motion) 
      path->push_back(incremental_motion());
    else
      path->push_back(position());
    last_path_time = datmo->_scan_time;
    if (path->size() >= _path_size)
      path->pop_front();
  }
}


// Update statistics on measurement residue.  This is just the statistics of
// the innovation, since H is an identity.  When finding the covariance, we
// add in a fraction of the prior measurement noise to set a floor on the
// measurement noise estimate.  This is particularly useful for vague line
// ends.
//
// Residue update has two phases: during track startup, we compute
// recursive mean and covariance with equal sample weighting.  After
// enough time has elapsed, we switch to a mode where we have a first
// order response (exponential decay.)

// The xxx_prescale determine the % contribution to the result of the
// current measurement.  Note that:
//    1 - exp(-x) ~= x, for 0 < x << 1
//
// We don't compute it this way (in case of large associate_delta_t), but we can
// see the cutover from equal weighting to exponential decay happens
// at associated_count = _residue_time_constant/associate_delta_t.
//
// With all features, the measurement noise comes from the feature measurement
// residue, combined with a noise floor model.
//
// In theory, for computing an adaptive measurement noise, we should reduce
// the noise residue by the estimate covariance to represent the contribution
// to the measurement residue from the estimate uncertainty.  However, this
// risks creating small or even negative measurement noise when the estimate
// covariance is too high.  Since we are knowingly inflating process and
// measurement noise to deal with time-varying behavior and modelling error,
// this subtracting out the state covariance would to cause problems.
//
void ObjectTrack::update_residue
    (FeaturePoint &feature, const FeaturePoint &measurement,
     const PointVec &innovation)
{
  double exp_prescale = 1 - exp(-associate_delta_t/_residue_time_constant);
  double mean_prescale =  max(1/(double)feature.associated_count,
			      exp_prescale);
  feature.residue_mean *= (1 - mean_prescale);
  feature.residue_mean = feature.residue_mean + innovation*mean_prescale;

  double cov_prescale = max(feature.associated_count == 1
			    ? 0.0
			    : 1/(feature.associated_count - 1.0),
			    exp_prescale);
  feature.residue_covariance *= (1 - cov_prescale);
  PointVec diff = innovation - feature.residue_mean;
  PointMat cov_inc;
  cov_inc.outer_product(diff, diff);
  PointMat cov_floor(measurement.residue_covariance);
  cov_floor *= 1/fsqr(_min_noise_factor);
  cov_inc.add(cov_inc, cov_floor);
  cov_inc *= cov_prescale;
  feature.residue_covariance.add(feature.residue_covariance, cov_inc); // S_hat
}


// Return a direction vector normal to a FIRST or LAST feature.
PointVec ObjectTrack::feature_normal (int f_ix) {
  if (f_ix == FIRST)
    return norm_dir;
  else {
    ASSERT(f_ix == LAST, "Bad feature index.");
    if (shape_class == CORNER)
      return direction;
    else
      return norm_dir;
  }
}


// ObjectTrack::maybe_reset
//
// Check for various situations where we should reset the feature
// (reinitialize from the measurement.)  If we reset, return true, suppressing
// normal state update.
// 
// We reset only the feature position if:
//  1] The residue mean is too large (indicating consistent tracking error
//     over time.)
//  2] The "reset" arg is true, and the Mahalanobis distance of the state
//     change is too large.
//
// In case [1], we probably have a wedged feature, are just trying to maintain
// the feature in approximately the correct place.  We zero the mean also, so
// that we don't keep resetting continuously, and it is possible that the
// feature will start tracking well again.  Or a vague->non-vague transition
// may force a full reset.
//
// In case [2], the reset flag indicates to expect a spurious position jump,
// and we don't want that to be interpreted as motion.
//
bool ObjectTrack::maybe_reset
    (int f_ix, Segment *seg, KFLinear &filter, bool motion_feature, bool reset,
     bool ripe, const FilterVec &state_inc)
{
  FeaturePoint &feature = features[f_ix];
  FeaturePoint &measurement = seg->features[f_ix];

  double residue_mag = feature.residue_mean.length();
  if (ripe && residue_mag > _residue_mean_reset_threshold) {
    feature.state = measurement.state;
    feature.residue_mean = Vec2d(0, 0);
    LOG(this, "maybe_reset.residue_reset " <<feature_names[f_ix]
	<<" " <<residue_mag <<(feature.vague ? " [vague]" : ""));
    return true;
  }

  if (reset) {
    const FilterMat &Iota = filter.information();
    double mahalanobis_2 = state_inc.dot(Iota * state_inc);
    if (mahalanobis_2 > fsqr(_feature_reset_sigmas)) {
      feature.state = measurement.state;
      LOG(this, "maybe_reset.mahalanobis_reset " <<feature_names[f_ix]
	  <<" " <<sqrt(mahalanobis_2)
	  <<(feature.vague ? " [vague]" : ""));
      return true;
    }
  }

  return false;
}


// ObjectTrack::associate_feature
//      
// Measurement update step for old feature with new measurement.  Filter is
// the filter that we associate with.  motion_feature is true if this is a
// feature that is contributing to the motion estimate, and thus should also
// be maxed into to the track feature_associated_count.
//
// This is a somewhat mutated version of the standard KF update because each
// feature has some characteristics of an independent track, but is not fully
// independent.  This is done mainly to avoid the computational overhead of
// maintaining seperate K and P for each feature, but also because it makes
// some kind of sense.
//
// The features are not fully independent from the track.  In particular, it
// is assumed the features have no velocity independent of the track. They do
// however have a position independent of the track.  We want to have a
// filtered feature position for assessing relative motion from, as this
// should be less noisy than subtracting two sequential raw measurements.
//
// We apply the state adjustment to both the feature's state (position) and to
// the KalmanFilter state (relative motion, velocity.)  This way the track
// gets relative motion and velocity which is the fused result of the feature
// motion.  The feature still gets an independent position, because there is
// no feedback from the center to the feature except via velocity.  For
// features that don't contribute to the motion estimate (center, somtimes
// bounds), the KalmanFilter is a dummy, discarding that part of the update.
//
// Measurement noise:
//   We compute and use the measurement residue statistics as a noise-adaptive
//   measurement noise. In addition, we add in the full prior measurement noise
//   to this value when then feature is not ripe.  This prevents excessive
//   optimisim about noise when not enough data has been collected yet.  Adding
//   allows a preliminary bad noise estimate to come though.
//
// Vague lines:
//   When associating a vague line end, only the normal component of the
//   innovation is used in computing the velocity and acceleration change.  This
//   minimizes spurious velocity due to vague features, but still allows us to
//   be surprised by the longitudinal component of the motion, possibly causing
//   the association to fail if it is excessive.
//
void ObjectTrack::associate_feature
    (int f_ix, Segment *seg, KFLinear &filter,
     bool motion_feature, bool reset)
{
  FeaturePoint &feature = features[f_ix];
  FeaturePoint &measurement = seg->features[f_ix];
  
  feature.associated_count++;
  if (motion_feature && feature.associated_count > feature_associated_count)
    feature_associated_count = feature.associated_count;

  bool ripe = (feature.associated_count >= _min_associate_count);
  bool any_vague = feature.vague || measurement.vague;

  PointMat R_k = measurement.residue_covariance;
  if (!ripe) R_k.add(R_k, feature.residue_covariance);
#if 0
  PointMat dum;

  ASSERT(dum.spd_inverse(measurement.residue_covariance), "meas cov not SPD!");
  ASSERT(dum.spd_inverse(feature.residue_covariance), "feat cov not SPD!");
  ASSERT(dum.spd_inverse(R_k), "R_k not SPD!");
#endif

  const FilterMat &K = filter.gain(R_k);

  double kw = (1/K.at(P_X, P_X) + 1/K.at(P_Y, P_Y))/2;
  if (kw > 30)
    LOG(this, "associate_feature.k_wedged " <<feature_names[f_ix] <<" " <<kw
	<<(any_vague ? " [vague]" : ""));

  // Refine state estimate based on measurement.
  PointVec &z_k = measurement.state;
  PointVec &x_hat = feature.state; // Feature state (position) H * x_hat
  PointVec innovation = z_k - x_hat;
  FilterVec state_inc = K * filter.make_state(innovation);

  // Modify state_inc if vague line end.
  if (any_vague && (f_ix == FIRST || f_ix == LAST)) {
    PointVec norm = feature_normal(f_ix);
    PointVec norm_innov = norm * innovation.dot(norm);
    FilterVec norm_inc = K * filter.make_state(norm_innov);
    for (int i = V_X; i <= A_Y; i++)
      state_inc[i] = norm_inc[i];
  }

  if (!maybe_reset(f_ix, seg, filter, motion_feature, reset, ripe, state_inc)) {
    // Track State Update
    filter.state = filter.state + state_inc;
    
    // feature state update
    x_hat = x_hat + state_part(state_inc, P_X);

    update_residue(feature, measurement, innovation);
    filter.measure(R_k);
  }

  feature.vague = measurement.vague;
}


//// ObjectTrack::associate: feature correspondence and track result analysis


// Check our direction invariants.
void ObjectTrack::check_direction () {
  if (shape_class == CORNER) {
    ASSERT((features[FIRST].state - features[CORNER_F].state)
	   .dot(direction) > 0,
	   "Corner direction bad.");
    Vec2d side_dir = features[LAST].state - features[CORNER_F].state;
    ASSERT(side_dir.dot(norm_dir) > 0, "Corner norm_dir bad.");
  } else {
    if (shape_class == LINE) {
      ASSERT((features[FIRST].state - features[LAST].state).dot(direction)
	     > 0,
	     "Line direction bad.");
    } else {
      Vec2d scan_dir = position() - scanner->location;
      ASSERT(scan_dir.dot(norm_dir) > 0, "norm_dir not away from scanner.");
    }
  }
}


// Set track direction from rotatation filter state if is reasonably close to
// the feature orientation.  If we are disoriented, or if THETA_ORIENTATION is
// too far off, set direction from the features.
// 
// Whatever we choose as direction, the normal is normal to direction, but
// with sign chosen to point toward LAST in a corner and away from the scanner
// otherwise.
//
// We also do some maintenance of heading_offset. If we are moving, so
// THETA_HEADING is defined, and mismatch between orientation and heading is >
// 45 degrees, change our orientation in 90 degree increments while preserving
// the track direction vector.  Also keep heading_offset within +/- 2 pi.
//
void ObjectTrack::set_direction () {
  double &theta_o = kf_rotation.state[THETA_ORIENTATION];
  const double &theta_h = kf_rotation.state[THETA_HEADING];
  double h_o_diff = theta_h - theta_o;
  while (moving && fabs(h_o_diff) > M_PI/4) {
    double incr = signum(h_o_diff) * M_PI/2;
    h_o_diff -= incr;
    theta_o += incr;
    heading_offset -= incr;
    LOG(this, "set_direction.moving " <<theta_o*180/M_PI
	<<" " <<heading_offset*180/M_PI);
  };

  if (heading_offset > 2*M_PI)
    heading_offset -= 2*M_PI;
  else if (heading_offset < -2*M_PI)
    heading_offset += 2*M_PI;

  double theta = kf_rotation.state[THETA_ORIENTATION] + heading_offset;
  Vec2d new_dir = Vec2d(cos(theta), sin(theta));
  if (shape_class == LINE || shape_class == CORNER) {
    Vec2d f_dir = (shape_class == LINE)
      ? features[FIRST].state - features[LAST].state
      : features[FIRST].state - features[CORNER_F].state;
    f_dir.normalize();
    if (disoriented) {
      direction = f_dir;
    } else if (new_dir.dot(f_dir) > cos(_angle_reset_threshold)) {
      direction = new_dir;
    } else {
      direction = f_dir;
      LOG(this, "set_direction.ignoring "
	  <<angle_diff(f_dir, new_dir)*180/M_PI);
    }
  } else {
    direction = new_dir;
  }

  Vec2d norm = normal(direction);
  if (shape_class == CORNER) {
    Vec2d side_dir = features[LAST].state - features[CORNER_F].state;
    norm_dir = side_dir.dot(norm) < 0 ? norm * -1 : norm;
  } else {
    Vec2d scan_dir = position() - scanner->location;
    norm_dir = scan_dir.dot(norm) < 0 ? norm * -1 : norm;
  }
  
  check_direction();
}


// Measurement step for rotation.  This uses the innovation and measurement
// noise in the innov_rotation, R_k_rotation.
void ObjectTrack::associate_rotation ()
{
  double &d_theta_o = innov_rotation[THETA_ORIENTATION];

  // If we get a smaller angle by flipping 180 degrees, do it.  This
  // changes direction while preserving heading.
  if (fabs(d_theta_o) > M_PI/2) {
    double incr = signum(d_theta_o) * M_PI;
    d_theta_o -= incr;
    heading_offset += incr;
    LOG(this, "associate_rotation.flip180 " <<heading_offset * 180/M_PI);
  }

  LOG(this, "associate_rotation " <<innov_rotation[0] * 180/M_PI
      <<" " <<innov_rotation[1] * 180/M_PI);
  const RotFilterMat &K = kf_rotation.gain(R_k_rotation);
  kf_rotation.state = kf_rotation.state + K * innov_rotation;
  kf_rotation.measure(R_k_rotation);
}


// Find the best angular match between the side of a corner track and
// a line track, and check if this is better than the default match of
// the ends to ends.  The match is considered better if the corner is closer
// to the identified end than the corresponding end->end distance.
//
// If it is a good match, return true and set:
// d_theta:
//     Amount that you would have to rotate line to make it match up with the
//     corner (radians.)
// offset_incr:
//     Increment to heading_offset to match on line->corner association.
//     direction = heading + heading_offset, if the change in direction from
//     line->direction to corner->direction is e.g. +90 degrees, then to
//     preserve heading, offset_incr must be also be +90 degrees.
// corner_ix:
//     The line feature index that should be associated with the corner
//     feature.
// match_ix:
//     The other line end (which should be associated as the same feature
//     in the two tracks.)
//
// We use a static method because *this might be either the line or the
// corner.
//
bool ObjectTrack::find_match_dir
    (ObjectBase *line, ObjectBase *corner, double &d_theta, double &offset_incr,
     int &corner_ix, int &match_ix)
{
  Vec2d match_dir;
  double direct_match = line->direction.dot(corner->direction);
  double norm_match = -line->direction.dot(corner->norm_dir);
  if (fabs(direct_match) > fabs(norm_match)) {
    if (direct_match < 0) return false;
    match_dir = corner->direction;
    corner_ix = LAST;
    match_ix = FIRST;
    offset_incr = 0;
  } else {
    if (norm_match < 0) return false;
    match_dir = corner->norm_dir * -1;
    corner_ix = FIRST;
    match_ix = LAST;
    // From our invariants, we know that -norm_dir will match up with
    // direction, but we don't know which sign of rotation that is, so compare
    // to the normal of line->direction.  If line +90 (normal) matches corner,
    // then offset_incr is also +90.
    offset_incr = (normal(line->direction).dot(corner->direction) > 0)
      ? M_PI/2
      : -M_PI/2;
  }

  Vec2d line_corner = line->features[corner_ix].pos();
  double d_end_corner = length2(corner->features[CORNER_F].pos() - line_corner);
  double d_end_end = length2(corner->features[corner_ix].pos() - line_corner);
  if (d_end_corner < d_end_end) {
    d_theta = angle_diff(line->direction, match_dir);
    return true;
  } else {
    return false;
  }
}

// This is a reinitializing version of rotation measurement.  Assume the old
// heading_offset (and direction) are meaningless, but that the heading may be
// meaningful.  Find a new heading_offset that minimizes the heading error mod
// 90 degrees.  If the resulting rotation innovation is still too large, then
// set the heading to the measurement.
//
// If new seg is non-disoriented, measure the new rotation.  Disoriented seg
// heading measurement is done in heading_from_motion.
void ObjectTrack::reset_rotation (Segment *seg) {
  if (shape_class == COMPLEX) return;
  double theta = kf_rotation.state[THETA_ORIENTATION];
  Vec2d h_dir(cos(theta), sin(theta));
  double d_theta = angle_diff(h_dir, seg->direction);
  heading_offset = 0;
  while (fabs(d_theta) > M_PI/4) {
    double incr = signum(d_theta) * M_PI/2;
    d_theta -= incr;
    heading_offset += incr;
    LOG(this, "reset_rotation.heading_offset " <<heading_offset * 180/M_PI);
  }
  if (fabs(d_theta) > _angle_reset_threshold) {
    RotFilterMat cov = kf_rotation.covariance();
    double noise = (seg->disoriented
		    ? _angular_pos_noise * unknown_angle
		    : _angular_pos_noise);
    cov.at(THETA_ORIENTATION, THETA_ORIENTATION) += fsqr(noise);
    kf_rotation.set_covariance(cov);
    kf_rotation.state[THETA_ORIENTATION] += d_theta;
    LOG(this, "reset_rotation.reset "
	<<kf_rotation.state[THETA_ORIENTATION]*180/M_PI);
  } else if (!seg->disoriented) {
    R_k_rotation.at(THETA_ORIENTATION, THETA_ORIENTATION)
      = fsqr(_angular_pos_noise);
    innov_rotation[THETA_ORIENTATION] = d_theta;
  }
}


// If reset is true or either seg or track is disoriented, then call
// reset_rotation.  Otherwise set orientation measurement.
void ObjectTrack::measure_rotation (double d_theta, bool reset) {
  if (!disoriented && !best_match->disoriented && !reset) {
    R_k_rotation.at(THETA_ORIENTATION, THETA_ORIENTATION)
      = fsqr(_angular_pos_noise);
    innov_rotation[THETA_ORIENTATION] = d_theta;
  } else {
    reset_rotation((Segment *)best_match);
  }
}


// Determine which features correspond, and do measurement update on them.
//
// We special-case associating line and corner features, since we often flip
// between the two on a track, and we don't want to lose the track info on the
// features in common between the line and corner.
//
// Otherwise, we just associate the same features in the two tracks as long as
// both are valid.  This covers same-kind association (LINE -> LINE, etc.) and
// association with COMPLEX, where we need to fall back on the bounding box.
//
// We also exercise some control of the update process here by saying which
// KalmanFilter the results of association of a given feature should
// contribute to: the center position, motion, or no contribution (only update
// the feature position itself.)
// 
// When updating the center itself, we use kf_center.  This way, the motion of
// the center contributes nothing to the velocity estimate (or the relative
// motion.)  This comes from the application-specific knowledge that the
// location of the center is poorly known, thus its motion is a poor measure
// of velocity.
//
// Similarly, when we have a linear shape, we update the bound features, but
// use a discarded copy of the motion filter.  As with the center, we choose
// not to make use of the bound features when we have better info.  However,
// we want to keep the features up to date so that we have valid residue in
// case we become complex later on.
//
// In theory we should also discard the result of association of the
// corner feature in corner->corner association.  This is because the corner
// is really derived from the sides, and doesn't contain any independent
// information.  However, we seem to get better results by using the corner.
//
void ObjectTrack::measure (Segment *seg, bool reset) {
  bool associate_line_corner = false;
  int old_shape_class = shape_class;
  shape_class = seg->shape_class;
  bool disoriented_reset = reset || (disoriented && !seg->disoriented);

  // If switching between CORNER and LINE, probably one of the ends in the
  // LINE is the corner feature in the CORNER track.
  double d_theta, offset_incr;
  int corner_ix, match_ix;
  if (old_shape_class == CORNER && shape_class == LINE) {
    if (find_match_dir(seg, this, d_theta, offset_incr, corner_ix, match_ix)) {
      d_theta = -d_theta;
      offset_incr = -offset_incr;
      features[corner_ix] = features[CORNER_F];
      associate_feature(corner_ix, seg, kf_motion, true, disoriented_reset);
      associate_line_corner = true;
    }
  } else if (old_shape_class == LINE && shape_class == CORNER) {
    if (find_match_dir(this, seg, d_theta, offset_incr, corner_ix, match_ix)) {
      features[CORNER_F] = features[corner_ix];
      associate_feature(CORNER_F, seg, kf_motion, true, disoriented_reset);
      features[corner_ix] = seg->features[corner_ix];
      associate_line_corner = true;
    }
  }

  // Used to discard motion of bounds when we have a linear feature.
  KFLinear discard(kf_motion);

  if (associate_line_corner) {
    associate_feature(match_ix, seg, kf_motion, true, disoriented_reset);
    associate_feature(MIN_BOUND, seg, discard, false, reset);
    associate_feature(MAX_BOUND, seg, discard, false, reset);
    associate_feature(CENTER, seg, kf_center, false, disoriented_reset);

    if (offset_incr != 0) {
      heading_offset += offset_incr;
      LOG(this, "measure.heading_offset " <<heading_offset * 180/M_PI);
    }
    measure_rotation(d_theta, disoriented_reset);
  } else {
    // all associations except line->corner or corner->line (and some of
    // those.)
    bool either_complex =
      (old_shape_class == COMPLEX || shape_class == COMPLEX);

    for (int feature_ix = 0; feature_ix < NUM_FEATURES; feature_ix++)
      if (valid_feature(feature_ix, old_shape_class)) {
	if (!seg->valid_feature(feature_ix)) {
	} else if (feature_ix == CENTER) {
	  associate_feature(CENTER, seg, kf_center, false, disoriented_reset);
	} else if (feature_ix == CORNER_F) {
	  associate_feature(feature_ix, seg, discard, true, disoriented_reset);
	} else if (feature_ix > MAX_BOUND) { // FIRST, LAST
	  associate_feature(feature_ix, seg, kf_motion, true, disoriented_reset);
	} else { // MIN_BOUND, MAX_BOUND
	  // If complex, associate bounds, and apply result to motion.
	  // Discard result of bounds when we have a linear feature.
	  if (either_complex)
	    associate_feature(feature_ix, seg, kf_motion, true, reset);
	  else
	    associate_feature(feature_ix, seg, discard, false, reset);
	}
      } else {
	if (seg->valid_feature(feature_ix))
	  features[feature_ix] = seg->features[feature_ix];
      }

    measure_rotation(angle_diff(direction, seg->direction),
		     disoriented_reset || old_shape_class != shape_class);
  }
}


// Limit change in velocity or acceleration.  ix is the state index, limit is
// the physical limit, and what is a string used in log messages.  The limit
// used is the max of the parameter arg and a value derived from the estimate
// covariance and _initial_d_dt_sigmas_per_sec.  This allows us to adapt
// faster when error is dominated by estimate uncertainty rather than
// modelling error.
//
// As a somewhat ad-hoc hack, we only limit when we are not moving if the
// change results in an increased magnitude.  This reduces problems with
// scanner angular resolution artifacts ratcheting up the velocity because the
// points drift in one direction until a new ray hits the leading edge of the
// object, at which point we jump back.  This jump back is more rapid, and may
// d/dt limit when the other direction doesn't.  Of course, this also happens
// when we *are* moving, but it's not clear how to generalize this hack.
//
bool ObjectTrack::limit_d_dt (ObjectTrack *save_state, int ix,
			      double limit, const char *what)
{
  const FilterMat &cov = kf_motion.covariance();
  limit = max(sqrt((cov.at(ix, ix) + cov.at(ix+1, ix+1)) * 0.5)
	      * _initial_d_dt_sigmas_per_sec,
	      limit);

  Vec2d old = state_part(save_state->kf_motion.state, ix);
  Vec2d nval = state_part(kf_motion.state, ix);
  Vec2d inc = nval - old;
  double max_diff = limit * associate_delta_t;
  double d_dt = inc.length();
  if (d_dt > max_diff
      && (moving || nval.length() > old.length())) {
    LOG(this, what <<" " <<limit <<" "
	<<(associate_delta_t ? d_dt/associate_delta_t : 1e9));
    inc.normalize();
    set_state_part(kf_motion.state, ix, old + inc*max_diff);
    return true;
  } else {
    return false;
  }
}


// If a vague line, zero the longitudinal component of the acceleration, since
// this is not measured.  Then limit acceleration to _max_acceleration.  This
// is done even on unripe tracks, as silly accelerations are always, well,
// silly.
void ObjectTrack::limit_acceleration () {
  Vec2d acc = acceleration();
  if (vague_line())
    acc = norm_dir * acc.dot(norm_dir);
  double speed_dot = acc.length();
  if (speed_dot > _max_acceleration) {
    acc * (1/speed_dot);
    acc = acc * _max_acceleration;
  }
  set_state_part(kf_motion.state, A_X, acc);
}


// Check whether the motion is statistically significant and if the magnitude
// is great enough.  Both tests have hysteresis bands favoring tracks that
// were ever_moving.
//
// The motion is statistically significant if the Mahalanobis distance between
// the velocity and zero exceeds a threshold.  If a vague line, only consider
// normal component of velocity, as only that is measured.
//
void ObjectTrack::test_motion (bool &motion_significant, bool &fast_enough) {
  Vec2d vel = velocity();
  PointMat v_info = velocity_covariance_sq();
  v_info.spd_inverse(v_info);
  Vec2d n_vel = ((!ever_moving && vague_line())
		 ? norm_dir * norm_dir.dot(vel)
		 : vel);
  double v_sigmas_2 = n_vel.dot(v_info * n_vel);
  double sigma_thresh = (ever_moving
			 ? fsqr(_min_moving_sigmas)
			 : fsqr(_max_fixed_sigmas));
  motion_significant = (v_sigmas_2 >= sigma_thresh);
  double v_thresh = (ever_moving
		     ? _min_moving_velocity
		     : _max_fixed_velocity);
  fast_enough = n_vel.length() > v_thresh;
}


// check_moving is responsible for determining if a track might be moving, or
// if it is definitely not moving.  If the motion is significant and fast
// enough, then we set the do_validate flag which causes ObjectTrack::validate
// to look at the track and do some more refined moving checks.  Otherwise, if
// the velocity and velocity covariance are small enough, then we say we are
// definitely not moving.
//
void ObjectTrack::check_moving ()
{
  if (do_validate) return;

  bool motion_significant, fast_enough;
  test_motion(motion_significant, fast_enough);
  do_validate = (motion_significant && fast_enough);
    
  if (do_validate) {
    valid = false;
  } else {
    velocity_correction = PointVec();
    history_v_covariance.clear();
    moving = false;
    // velocity variance threshold
    double rthresh = fsqr(valid ? _max_valid_velocity_deviation
			  : _max_invalid_velocity_deviation);
    const FilterMat P_k_plus = kf_motion.covariance();
    valid = ((P_k_plus.at(V_X, V_X) + P_k_plus.at(V_Y, V_Y))*0.5 < rthresh
	     && velocity().length() < _max_fixed_velocity);
  }
}


// Limit the change in angular velocity according to
// _max_angular_acceleration.  Analogous to limit_d_dt.  Also set direction
// vectors from theta.
//
void ObjectTrack::limit_d_omega_dt (ObjectTrack *save_state)
{
  double cov = kf_rotation.covariance().at(V_THETA, V_THETA);
  double limit = max(cov * _initial_d_dt_sigmas_per_sec,
		     _max_angular_acceleration);
  double old = save_state->kf_rotation.state[V_THETA];
  double d_dt = kf_rotation.state[V_THETA] - old;
  double max_diff = limit * associate_delta_t;
  if (fabs(d_dt) > max_diff) {
    LOG(this, "limit_d_omega_dt " <<limit*180/M_PI
	<<" "
	<<(associate_delta_t ? (d_dt/associate_delta_t)*180/M_PI : 1e9));
    kf_rotation.state[V_THETA] = old + signum(d_dt)*max_diff;
  }
}


// Estimate heading from motion if we are moving.  If not, then
// reinitialize heading if no other heading info (e.g. from linear
// shape.)
void ObjectTrack::heading_from_motion () {
  double &theta = kf_rotation.state[THETA_HEADING];
  Vec2d h_dir(cos(theta), sin(theta));

  // direction of our velocity is a heading estimate.
  Vec2d v_dir = velocity();
  v_dir.normalize();
  R_k_rotation.at(THETA_HEADING, THETA_HEADING)
    = fsqr(_velocity_angular_pos_noise);
  innov_rotation[THETA_HEADING] = angle_diff(h_dir, v_dir);
}


// Attempt to associate with seg.  If the Mahalanobis distance of the computed
// state increment is too large, fail.  Also fail if there was a numeric
// problem (couldn't invert some matrix.)  If the track is ripe, do tests for
// being valid and moving.
//
// Find the information increment, which is a measure of how much the track
// was influenced by this measurement.  If the information increment is low,
// then the track is not responding to the measurement because the
// measurement is considered too noisy relative to how certain we think we are
// of the current state.  If the track is not responding to the data, that is
// bad, because we are basically just coasting blind.
//
// Though I call this "information" increment, the computation is really based
// on the covariance, which more closely mimics the computation of the Kalman
// gain.  The Kalman gain is what actually determines the response speed of
// the filter.
//
// The info increment is a single number summarizing the decrease in
// covariance relative to the previous value, weighted to as to emphasize the
// contribution from the direction that is most pooly localized.  To
// understand the approach, assume that the eigenvectors are basically
// unchanged by this measurement cycle, so we can use the ratio of the
// eigenvalues before and after.
//
// If info increment is too low and reset is true, fail.  If reset is false,
// then pretend not to associate via last_associated_time, but actually do
// associate.  This cuts us some more slack in the case of 1-1 matched tracks.
// Low info increment can be caused by e.g. vague lines, but we also get zero
// info increment if we reset all the features.  This can happen if we have a
// bad association.
//
// Failure in association is most likely due to a bad measurement, though it's
// possible that match_and_associate might have picked the wrong segment to
// associate with.  If association fails, we still leave the segment marked as
// associated via best_match so that it doesn't create another competing
// track.  Whatever the cause, if we fail to associate for long enough the
// track will be killed, allowing a new track to be created.
//
// If the association succeeds, then limit d_dt on velocity and acceleration.
//
// If we limit dv_dt, hold acceleration fixed, as otherwise the acceleration
// slews wildly during the time that we are limiting dv_dt (because the KF
// loop is open.)  We also set the limiting_dv_dt flag to modify the process
// noise update so that the covariance magnitude is effectively fixed at the
// initial value.
//
// As kind of a cheesy approximation to a multi-model approach, we force
// acceleration to zero on compact objects (pedestrians.)  Pedestrians don't
// spend much time in a constant acceleration regime.  Also, for reasons not
// entirely clear, we were getting really bad acceleration estimates that hung
// around and crufted up the track prediction.  This notable fails if
// we have a pedestrian with too low point density to be COMPACT.
//
// Vague lines are treated specially, since their longitudinal motion is not
// measured.  We allow a constant longitudinal velocity component, but don't
// consider it in our test for significance of motion.
//
// If reset is true, then reset any features whose innovation is too large.
// This is used when tracks split or merge, possibly invalidating the old
// features.
//
bool ObjectTrack::associate (Segment *seg, bool reset) {
  best_match = seg;
  best_match->best_match = this;
  best_match->id = id;

  // Copy *this so that we can restore the state if we fail.  For efficency,
  // find motion information first so that it is saved, since it is used in
  // both copies.
  kf_motion.information();

  // This is kind of a gross hack, but assigning *this to a ObjectTrack local
  // using the default copy constructor seems to tickle a bug related to STL,
  // so we'll just do a bit copy.  This will be faster, and will work fine, as
  // long as none of the vector, etc., data that we're aliasing gets
  // reallocated in this time window.  It shouldn't.  Note also that there is
  // now an explict copy constructor used elsewhere which does not copy all
  // state, and cannot be used here because we want to back up to the exact
  // state.
  unsigned char save_bytes[sizeof(ObjectTrack)];
  memcpy(save_bytes, this, sizeof(ObjectTrack));
  ObjectTrack *save_state = (ObjectTrack *)save_bytes;

  feature_associated_count = 1; // we max feature associated_count into here
  scanner = seg->scanner;

  // Set rotation measurement to represent no measurement.  If we are
  // able to measure a THETA_xxx angle, then that is filled in later.
  R_k_rotation.at(THETA_ORIENTATION, THETA_ORIENTATION)
    = fsqr(_angular_pos_noise * unknown_angle);
  R_k_rotation.at(THETA_HEADING, THETA_HEADING)
    = fsqr(_velocity_angular_pos_noise * unknown_angle);
  innov_rotation = RotFilterVec(0, 0, 0);

  measure((Segment *)best_match, reset);
  disoriented = best_match->disoriented;

  // COMPLEX can only become non-compact.
  if (!(seg->shape_class == COMPLEX && seg->compact))
    compact = seg->compact;

  FilterVec state_inc = kf_motion.state - save_state->kf_motion.state;
  const FilterMat &Iota_minus = save_state->kf_motion.information();
  double mahalanobis_dist_2 = state_inc.dot(Iota_minus * state_inc);
  bool failure = kf_motion.failure || kf_center.failure || kf_rotation.failure;

  if (failure
      || mahalanobis_dist_2 > fsqr(_max_associate_sigmas)) {
    if (log_file) {
      const char *sc_name = shape_class_names[best_match->shape_class];
      if (failure) {
	LOG(this, "associate.failed_numerical." <<sc_name)
	  } else {
	    LOG(this, "associate.failed_sigma." <<sc_name <<" mahalanobis: "
		<<sqrt(mahalanobis_dist_2));
	  }
    }
    memcpy(this, save_bytes, sizeof(ObjectTrack));
    return false;
  }

  // Info increment computation.
  double W_minus[2], W_plus[2];
  get_submatrix_eigenvalues(W_minus, save_state->kf_motion.covariance());
  get_submatrix_eigenvalues(W_plus, kf_motion.covariance());
  
  double sum = 0;
  for (int i = 0; i < 2; i++)
    sum += max(W_minus[i]/W_plus[i] - 1, 1e-6);

  double info_inc = sum/2;

  // If info increment is too low, fail or pseudo-fail.
  // ### occlusion factor kind of a hack...  It seems that our process noise
  // related to occlusion is not high enough, so we are too sure of our
  // position when we reacquire.
  if (info_inc < _min_info_increment && occlusion != TOTAL) {
    if (reset) {
      LOG(this, "associate.failed_low_info " <<info_inc);
      memcpy(this, save_bytes, sizeof(ObjectTrack));
      return false;
    } else {
      LOG(this, "associate.ignoring_low_info " <<info_inc);
    }
  } else {
    associate_delta_t = datmo->_scan_time - last_associated_time;
    last_associated_time = datmo->_scan_time;
    associated_count++;
  }

  best_match->id = id;

  // Log measurement history.
  if (history->size() >= _history_size)
    history->pop_front();
  TrackHistory hist;
  hist.scan_time = datmo->_scan_time;
  hist.shape_class = seg->shape_class;
  for (int i = 0; i < NUM_FEATURES; i++)
    hist.features[i] = seg->features[i];
  history->push_back(hist);

  // d/dt limiting and moving/validity test.
  limit_acceleration();
  if (limit_d_dt(save_state, V_X, _max_acceleration, "limit_dv_dt")) {
    set_state_part(kf_motion.state, A_X,
		   state_part(save_state->kf_motion.state, A_X));
    limiting_dv_dt = true;
  } else {
    if (compact)
      set_state_part(kf_motion.state, A_X, PointVec());
    else 
      limit_d_dt(save_state, A_X, _max_jerk, "limit_da_dt");
    limiting_dv_dt = false;
  }

  if (moving || feature_associated_count > _min_associate_count) {
    check_moving();
  }

  if (moving) heading_from_motion();
  associate_rotation();
  limit_d_omega_dt(save_state);

  LOG(this, "associate mahalanobis: " <<sqrt(mahalanobis_dist_2)
      <<" info_inc: " <<info_inc);

  points = best_match->points;
  occlusion = best_match->occlusion;
  set_direction();
  merged_with = 0;

  return true;
}


//// ObjectTrack::match_and_associate: track correspondence

// Find the "closeness" between this track and seg, which is intended to be a
// measure of how similar seg is to this track.  In match_and_associate we
// know the tracks overlap, the main point is not so much to find which
// segments don't correspond to this track at all, but to determine in the
// case of track splitting, which segment will best support the existing
// track.  It also provides guidance in cases where there is not a real track
// split, but we falsely believe there is an overlap.
//
// Closeness is just the sum of the inverse distance between corresponding
// features.  It is not a mean.  The more close features, the merrier.
//
// With line/corner, it's not entirely clear how we will end up associating,
// so we take the max closenesses across all pairings of FIRST, LAST, CORNER_F.
//
// We ignore the center since it is derived from the other features.
static inline double point_closeness (const Vec2d &p1, const Vec2d &p2) {
  return 1/max((p1 - p2).length(), 1e-3);
}
//
double ObjectTrack::closeness (Segment *seg) {
  double res = 0;
  for (int i = MIN_BOUND; i <= MAX_BOUND; i++)
    res += point_closeness(features[i].state, seg->features[i].state);

  for (int i = FIRST; i <= CORNER_F; i++)
    if (valid_feature(i)) {
      double max_val = 0;
      for (int j = FIRST; j <= CORNER_F; j++)
	if (seg->valid_feature(j)) {
	  double val = point_closeness(features[i].state, seg->features[j].state);
	  if (val > max_val) max_val = val;
	}
      res += max_val;
    }

  return res;
}

// Called when we don't want to associate with our best match, but want to
// prevent making a new track.  
void ObjectTrack::fail_associate (Segment *match) {
  best_match = match;
  best_match->best_match = this;
  best_match->id = id;
  if (log_file)
    match->dump(*log_file, log_index, true);
}
  

// Update occlusion level for a track.  If the track was associated recently,
// it is not occluded, so there is nothing to do.
//
// If apparently totally occluded, upgrade to TOTAL.  If apparently totally
// visible, downgrade to NONE.
//
// If currently TOTAL, but should be partially visible, leave occlusion
// unchanged.  "Should" if the motion estimate was correct and the object has
// not significantly changed velocity.  Requiring the track to be fully
// visible gives us some slack w.r.t. bad prediction.  If we could somehow
// predict exactly where the object "should be", we would regain the track as
// soon as any part is visible.
//
void ObjectTrack::update_occluded () {
  if (last_associated_time == scanner->cur_time) return;

  // There is some scanner where all points are inside of the FOV and not
  // occluded.
  bool occlusion_none = false;

  // No point is visible in any scanner.  All points are either outside of the
  // FOV or occluded.
  bool occlusion_total = true;

  for (unsigned int scan_ix = 0; scan_ix < datmo->_scanners.size();
       scan_ix++) {
    ScannerInfo *scan = datmo->_scanners[scan_ix];
    if (scan->last_time == -1) continue; // never processed any scan yet.
    PointMat rot;
    set_rotation(rot, -(datmo->_states[0].yaw + scan->yaw_offset));

    occlusion_none = true;
    for (unsigned int i = 0; i < points.size(); i++) {
      // The point in scanner coordinates.
      PointVec p = rot * (points[i].pos - scan->location);
    
      float bearing = atan2(p[1], p[0]);
      int bi = scan->bearing_index(bearing);

      if (bi == -1) {
	occlusion_none = false;
	break; 
      }

      // If in the last scan, either the return was closer than where we
      // predict this point to be or the return was ignored (due to too
      // few points, ignore region, etc.), then the point is considered
      // occluded.
      const PointData &pd = scan->points[bi];
      if (scan->ignored[bi] || pd.range < p.length()) {
	occlusion_none = false;
      } else {
	occlusion_total = false;
      }
    }

    if (occlusion_none) break;
  }

  if (occlusion_total) {
    occlusion = TOTAL;
  } else if (occlusion_none) {
    occlusion = NONE;
  } else {
    if (occlusion != TOTAL)
      occlusion = PARTIAL;
  }
}

void ObjectTrack::match_and_associate () {
  match_and_associate_aux();
  if (logger) logger->update();
  if (point_logger) point_logger->update();
}


// Find the matching segment for a track and associate with it.  The
// complications occur when there is not a 1-to-1 match between overlap tracks
// and segments.
void ObjectTrack::match_and_associate_aux () {
  int n_overlap = overlap.size();
  if (n_overlap == 0 && !best_match) {
    if (log_file) {
      LOG(this, "match_and_associate.failed_no_overlaps, scanner "
	  <<datmo->_scanner->name);
      dump(*log_file, log_index, true);
    }
    return;
  }

  double max_cl = 0;
  Segment *bm = NULL;
  Segment *merged_seg = NULL;  

  if (best_match) {
    // match already forced by model based segmentation.
    LOG(this, "match_and_associate.forced");
    bm = (Segment *)best_match;
    max_cl = closeness(bm);
  } else if (n_overlap == 1 && overlap.front()->overlap.size() == 1) {
    // 1-to-1 match, the easy case.
    LOG(this, "match_and_associate.unique");
    bm = (Segment *)overlap.front();
    max_cl = closeness(bm);
  } else {
    LOG(this, "match_and_associate.multi " <<overlap.size()
	<<" " <<overlap.front()->overlap.size());
    // We have multiple overlaps (track split to multiple segments) and/or our
    // overlap segments do (tracks joined to single segment.)  First find our
    // closest overlap segment not already associated, which we will associate
    // with.
    for (list<ObjectBase *>::iterator seg_it = overlap.begin();
	 seg_it != overlap.end();
	 seg_it++) {
      Segment *seg = (Segment *)*seg_it;
      double cl = closeness(seg);
      if (cl > max_cl) {
	max_cl = cl;
	if (seg->best_match) {
	  merged_with = seg->id;
	  merged_seg = seg;
	} else {
	  bm = seg;
	}
      }
    }

    // For all overlap segs that aren't associated, set split_from.  If they
    // aren't associated by the end of the association pass, then they will be
    // made into new tracks with this state and split annotation.
    for (list<ObjectBase *>::iterator seg_it = overlap.begin();
	 seg_it != overlap.end();
	 seg_it++) {
      Segment *seg = (Segment *)*seg_it;
      if (seg != bm && !seg->best_match && !seg->split_from) {
	ever_split = true;
	seg->split_from = this;
      }
    }
  }

  if (!bm) {
    if (log_file) {
      LOG(this, "match_and_associate.failed_merged " <<merged_with);
      merged_seg->dump(*log_file, log_index, true);
    }
  } else if (max_cl < _min_associate_closeness) {
    LOG(this, "match_and_associate.failed_match_not_close " <<max_cl);
    fail_associate(bm);
  } else {
    if (!associate(bm, false)
	&& associated_count >= _min_associate_count) {
      LOG(this, "match_and_associate.reset_associate");
      associate(bm, true);
    }
    if (log_file)
      bm->dump(*log_file, log_index, true);
  }
}


//// ObjectTrack::validate -- retrospective motion validation

// Find corresponding features.  Two features are corresponding if they are
// mutual best-match.  If either complex, assume direct match of bounds.
// 
// c_ix maps the indices of features in the history entry to the corresponding
// features in the track.  -1 if no corresponding feature.
//
// First we find the closest feature in the opposite set for each feature.  We
// disallow correspondences between FIRST and LAST as they don't make
// topological sense.
//
// Then we find the features that are not mutual closest and invalidate them.
// 
// In theory CORNER_F CORNER_F associations don't contain any information not
// present in the FIRST/LAST, but we seem to get better results using them
// too.
//
void ObjectTrack::find_corresponding
    (const TrackHistory &hist, int c_ix[NUM_FEATURES]) 
{
  for (int i = 0; i < NUM_FEATURES; i++)
    c_ix[i] = -1;

  if (shape_class == COMPLEX || hist.shape_class == COMPLEX) {
    for (int i = MIN_BOUND; i <= MAX_BOUND; i++) 
      c_ix[i] = i;
  } else {
    // Squared distance of corresponding feature indexed by history feature index.
    double c_dist[NUM_FEATURES];

    // tk_c_ix and tk_c_dist is the inverse mapping from track feature indices
    // to history indices.
    int tk_c_ix[NUM_FEATURES];
    double tk_c_dist[NUM_FEATURES];

    for (int i = FIRST; i <= CORNER_F; i++) {
      tk_c_ix[i] = -1;
      tk_c_dist[i] = c_dist[i] = DBL_MAX;
    }

    for (int i = FIRST; i <= CORNER_F; i++) {
      if (valid_feature(i)) {
	for (int j = FIRST; j <= CORNER_F; j++) {
	  if (valid_feature(j, hist.shape_class)
	      && !(i == FIRST && j == LAST)
	      && !(i == LAST && j == FIRST)) {
	    double dist = length2(features[i].state - hist.features[j].state);
	    if (dist < c_dist[j]) {
	      c_dist[j] = dist;
	      c_ix[j] = i;
	    }
	    if (dist < tk_c_dist[i]) {
	      tk_c_dist[i] = dist;
	      tk_c_ix[i] = j;
	    }
	  }
	}
      }
    }

    for (int j = FIRST; j <= CORNER_F; j++) {
      if (c_ix[j] != -1 && tk_c_ix[c_ix[j]] != j)
	c_ix[j] = -1;
    }

    if (c_ix[CORNER_F] == CORNER_F)
      c_ix[CORNER_F] = -1;
  }
}


// Process a history entry for validity check.  We are accumulating two
// different streams of information about the difference between the
// retrodicted location and measured location:
//  1] Mean-square Mahalanobis distance according to prior position error.
//     This is used to evaluate moving and valid.
//  2] Metric distance error normalized by delta T from now.   We collect the
//     mean, which is a velocity correction, and the covariance, which is an
//     estimate of the velocity covariance.
//
void ObjectTrack::process_1_history
    (/* const */ ObjectTrack *copy, const TrackHistory &hist,
     ofstream *vlog, int &vlog_ix, HistoryStats &stats)
{
  int c_ix[NUM_FEATURES];
  copy->find_corresponding(hist, c_ix);
  PointMat rot;
  double theta, copy_theta, ignore;
  get_theta(theta, ignore);
  copy->get_theta(copy_theta, ignore);
  set_rotation(rot, theta - copy_theta);
  bool printed_any = false;
  if (vlog) {
    *vlog <<"#index " <<vlog_ix <<", time "
	  <<hist.scan_time - datmo->_first_scan_time <<endl;
    vlog_ix++;
  }

  for (int j = 0; j < NUM_FEATURES; j++) {
    if (c_ix[j] != -1) {
      const FeaturePoint &h_feature = hist.features[j];
      int this_ix = c_ix[j];
      const FeaturePoint &tk_feature = copy->features[this_ix];
      Vec2d diff = h_feature.state - tk_feature.state;

      // If vague line end, use only normal component of error.
      if ((h_feature.vague || tk_feature.vague) && (j == FIRST || j == LAST)) {
	int other_ix = (hist.shape_class == CORNER
			? CORNER_F
			: (j == FIRST ? LAST : FIRST));
	Vec2d norm = normal(hist.features[other_ix].state - h_feature.state);
	norm.normalize();
	diff = norm * norm.dot(diff);
      }

      PointMat info;
      info.spd_inverse(h_feature.residue_covariance);
      double mahal2 = diff.dot(info * diff);
      stats.sumsq += mahal2;
      stats.num_sumsq++;

      if (mahal2 < fsqr(_history_outlier_sigmas)) {
	stats.info_sum.add(stats.info_sum, info);
	Vec2d rot_diff = rot * diff;
	rot_diff *= 1/(datmo->_scan_time - hist.scan_time);
	if (vlog) {
	  PointVec f_pos = features[this_ix].state;
	  *vlog <<"# Correspondence " <<feature_names[this_ix] <<" "
		<<feature_names[j] <<", error " <<diff.length() <<endl;
	  *vlog <<f_pos <<endl;
	  *vlog <<(f_pos + rot_diff) <<endl <<endl;
	  printed_any = true;
	}
	
	stats.num_mean++;
	double mean_prescale =  1/(double)stats.num_mean;
	stats.mean *= (1 - mean_prescale);
	stats.mean = stats.mean + rot_diff * mean_prescale;

	double cov_prescale = (stats.num_mean == 1
			       ? 0.0
			       : 1/(stats.num_mean - 1.0));
	stats.covariance *= (1 - cov_prescale);
	PointVec diff2 = rot_diff - stats.mean;
	PointMat cov_inc;
	cov_inc.outer_product(diff2, diff2);
	cov_inc *= cov_prescale;
	stats.covariance.add(stats.covariance, cov_inc);
      }
    }
  }
  if (vlog) {
    if (!printed_any)
      *vlog <<"#Dummy data\n" <<features[MIN_BOUND].state <<endl;
    *vlog <<endl <<endl;
  }
}


// Hacked version of process_1_history for offline testing.  Doesn't
// compute V correction, and uses min Eclidian distance across all
// features, not mean-square Mahalanobis.
void ObjectTrack::alt_process_1_history
    (/* const */ ObjectTrack &copy, const TrackHistory &hist,
     double &min_dist, double &rms_dist)
{
  int c_ix[NUM_FEATURES];
  copy.find_corresponding(hist, c_ix);
  min_dist = 1e6;
  double mean_square = 0;
  int n_mean_square = 0;
  for (int j = 0; j < NUM_FEATURES; j++) {
    if (c_ix[j] != -1) {
      const FeaturePoint &h_feature = hist.features[j];
      int this_ix = c_ix[j];
      const FeaturePoint &tk_feature = copy.features[this_ix];
      Vec2d diff = h_feature.state - tk_feature.state;

      // If vague line end, use only normal component of error.
      if ((h_feature.vague || tk_feature.vague) && (j == FIRST || j == LAST)) {
	int other_ix = (hist.shape_class == CORNER
			? CORNER_F
			: (j == FIRST ? LAST : FIRST));
	Vec2d norm = normal(hist.features[other_ix].state - h_feature.state);
	norm.normalize();
	diff = norm * norm.dot(diff);
      }

      mean_square += diff.dot(diff);
      n_mean_square++;
      if (diff.length() < min_dist)
	min_dist = diff.length();
    }
  }
  if (min_dist == 1e6) min_dist = 0;
  if (n_mean_square == 0)
    rms_dist = 0;
  else
    rms_dist = sqrt(mean_square/n_mean_square);
}


// Retrodict the track path, and compare to the history data.  We use only the
// oldest 1/3 of the history for comparison so as to weight older data more
// heavily (and also save some computation.)
//
// To minimize the number of calls to ObjectTrack::update when processing the
// most recent 2/3 of the data, we skip by hist_stride.  We do want to process
// some of the recent elements so that we extrapolate a curved trajectory,
// rather than a straight-line jump.
//
// If assume_fixed is true, then we do the same computation, but assuming that
// we are actually not moving at all.  This is implemented by not doing any
// update.
//
void ObjectTrack::process_history
    (bool assume_fixed, ofstream *vlog, int &vlog_ix, HistoryStats &stats)
{
  ObjectTrack copy_obj = *this;
  ObjectTrack *copy = &copy_obj;

  // Apply V correction (if any).
  set_state_part(copy->kf_motion.state, V_X, velocity());

  double then = datmo->_scan_time;
  int hist_ix = history->size() - 2;
  int use_ix = min((int)rint(history->size() * _history_measure_fraction), hist_ix);
  int end_ix = use_ix; // end of chunk: predict or measure
  double interval = _history_predict_interval;

  while (hist_ix >= 0) {
    while (hist_ix >= end_ix && then - (*history)[hist_ix--].scan_time < interval);
    if (hist_ix < end_ix) hist_ix = end_ix;

    const TrackHistory &hist = (*history)[hist_ix];
    if (!assume_fixed) {
      copy->set_delta_t(hist.scan_time - then);
      copy->update_state_only();
    }
    
    if (vlog) {
      *vlog <<"# T " <<hist.scan_time - datmo->_scan_time <<endl;
      copy->dump(*vlog, vlog_ix, false);
      dump_TrackHistory(*vlog, vlog_ix, hist);
    }
    then = hist.scan_time;
    if (hist_ix <= use_ix) {
      process_1_history(copy, hist, vlog, vlog_ix, stats);
      interval = _history_measure_interval;
      end_ix = 0;
    } else {
      if (vlog) {
	*vlog <<"#Dummy data\n" <<features[MIN_BOUND].state <<endl;
	*vlog <<endl <<endl;
	vlog_ix++;
      }
    }
    if (hist_ix == end_ix) break;
  }

  double info[2][2];
  stats.info_sum.getValue(info);
  calc_eigenvalues(info, stats.info_eigenvalues);
  stats.mean_square = stats.num_sumsq ? stats.sumsq/stats.num_sumsq : 0;
}

// Called on moving tracks to update the valid flag.   Works by doing a
// historical check of past measurments to see if they are consistent with the
// current motion estimate.
//
// What we do is project the current track backwards, and then find the
// Mahalanobis distance of the corresponding features beween the old
// measurement and the backward projected track.  The RMS value of the
// distance must be less than a threshold (with hysteresis).
//
// Also, we sum the position information, and require that the smallest
// eigenvalue be greater than a bound.  This insures that there is enough
// evidence to localise the track.
//
void ObjectTrack::validate () {
  if (history->size() == 0 || last_validated_time == history->back().scan_time)
    return;
  
  last_validated_time = history->back().scan_time;
  
  ofstream *vlog = NULL;
  int vlog_ix = 0;
  if (id == _validate_trace_track
	&& datmo->_scan_time - datmo->_first_scan_time >= _validate_trace_time)
    vlog = new ofstream("validate.dat");

  HistoryStats moving_stats, fixed_stats;
  process_history(false, vlog, vlog_ix, moving_stats);
  process_history(true, NULL, vlog_ix, fixed_stats);

  double f_info2 = fixed_info2.filter(fixed_stats.info_eigenvalues[0]);
  double f_sigma2 = (f_info2
		     ? fixed_sigma2.filter(fixed_stats.mean_square)/f_info2
		     : 1e9);
  double m_info2 = moving_info2.filter(moving_stats.info_eigenvalues[0]);
  double m_sigma2 = (m_info2
		     ? moving_sigma2.filter(moving_stats.mean_square)/m_info2
		     : 1e9);

  LOG(this, "validate.fixed " <<sqrt(f_info2) <<" " <<sqrt(f_sigma2)
      <<" [" <<fixed_stats.mean <<"]");

  LOG(this, "validate.moving "  <<sqrt(m_info2) <<" " <<sqrt(m_sigma2)
      <<" [" <<moving_stats.mean <<"]");

  bool ovalid = valid;
  bool omoving = moving;
  double min_info2 = fsqr(_min_history_pos_info);

  // Velocity correction deliberately does not depend on track classification
  // as moving or fixed.  This allows the validity test thresholds to be
  // varied without changing the motion estimate, thus affecting track
  // creation, hence track IDs.  We do consider some of the same evidence.
  Vec2d new_vc = velocity_correction - moving_stats.mean;
  bool vc_bad = (new_vc.length() > _max_v_correction);
  if (vc_bad) LOG(this, "validate.vc_bad");
  if (!_use_velocity_correction || vc_bad
      || m_sigma2 > fsqr(_max_v_correction_history_distance)) {
    velocity_correction = PointVec();
    history_v_covariance.clear();
  } else {
    velocity_correction = new_vc;
    history_v_covariance = moving_stats.covariance;
  }
      
  bool motion_significant, fast_enough;
  test_motion(motion_significant, fast_enough);
  if (!motion_significant)
    LOG(this, "validate.motion_insignificant");
  if (!fast_enough)
    LOG(this, "validate.too_slow");

  if (f_sigma2 < fsqr(_min_invalid_history_distance)
      && f_info2 > min_info2
      && m_sigma2 > f_sigma2 * fsqr(_history_bad_v_factor)) {
    // Fixed hypothesis much better supported than moving (a bad velocity
    // fix.)  reset filter clear do_validate.
    //
    // Kicking down to !do_validate is mainly an efficency thing.  It helps to
    // reduce load in situations where there is a transient anomaly such as
    // unmodeled scanner motion.  However, this does cause the classification
    // to affect the motion estimate, which is a problem for sensitivity
    // testing.
    valid = true;
    moving = false;
    kf_motion.state = FilterVec();
    kf_rotation.state[V_THETA] = 0;
    velocity_correction = PointVec();
    history_v_covariance.clear();
    do_validate = false;
    LOG(this, "validate.bad_v_resetting");
  } else if (f_sigma2 < fsqr(valid
			      ? _max_valid_history_distance
			      : _min_invalid_history_distance)
	     && f_info2 > min_info2
	     && (!motion_significant || !fast_enough)) {
    // Fixed hypothesis well supported, and motion estimate not above the
    // noise floor.  Say it's definitely not moving.  The velocity is left at
    // whatever it is corrected to based on the moving estimate.  We don't
    // care about how well supported the moving hypothesis is, since the
    // supposed motion is so low.
    LOG(this, "validate.fixed");
    valid = true;
    moving = false;
  } else if (m_info2 < min_info2) {
    // Not enough measurement information to support moving hypothesis.
    LOG(this, "validate.unknown_low_m_info");
    valid = moving = false;
  } else if (m_sigma2
	     > fsqr(ever_moving
		    ? _max_valid_history_distance
		    : _min_invalid_history_distance)) {
    // Moving error exceeds absolute threshold.  If ever_moving we tolerate
    // more absolute error.
    LOG(this, "validate.unknown_high_m_sigma");
    valid = moving = false;
  } else if (m_sigma2
	     * fsqr(moving ? _history_unknown_factor
	     	           : _history_moving_factor)
	     > f_sigma2) {
    // Compare the relative evidence for moving v.s. fixed.  If currently
    // moving, we tolerate weaker evidence.
    LOG(this, "validate.unknown_not_greater");
    valid = moving = false;
  } else if (!motion_significant || !fast_enough || vc_bad
	     || (!ever_moving && disoriented && !compact)) {
    // Motion is too small or statistically insignificant, velocity correction
    // too large, or we are not currently moving and the segmentation is noisy.
    LOG(this, "validate.unknown_other");
    valid = moving = false;
  } else {
    // All those tests, passed, we conclude we are definitely moving.
    valid = ever_moving = moving = true;
  }

  if (_print_moving_tracks
      && (valid != ovalid || moving != omoving)
      && (moving || omoving)) {
    dump_time_and_id(cout);
    if (valid && moving)
      cout <<" moving";
    else if (valid && !moving)
      cout <<" fixed";
    else
      cout <<" unknown";

    cout <<", speed " <<velocity().length() <<endl;
  }

  if (vlog) {
    vlog->close();
    if (valid || _validate_trace_time > 0) {
      DEBUG("Logged validate.dat");
      _validate_trace_track = 0;
    }
  }
}


//// ObjectTrack::living_offspring
//
// Search both ways in the split_from relation to see if there is any track
// related by splitting that is still alive.
// 
bool ObjectTrack::living_offspring () {
  // Find if any track we are split_from is still alive, exploiting the fact
  // that both Datmo::_tracks and ObjectTrack::split_from_vec are sorted.
  list<ObjectTrack *>::iterator track_it = datmo->_tracks.begin();
  for (unsigned i = 0; i < split_from_vec.size(); i++) {
    unsigned int split_id = split_from_vec[i];
    while (track_it != datmo->_tracks.end()
	   && (*track_it)->id <= split_id) {
      if ((*track_it)->id == split_id)
	return true;
      track_it++;
    }
  }
    
  // If ever_split is set, look at all living tracks and see if any is split
  // from this track.
  if (!ever_split) return false;
  for (list<ObjectTrack *>::iterator track_it = datmo->_tracks.begin();
       track_it != datmo->_tracks.end();
       track_it++) {
    ObjectTrack *track = *track_it;
    if (track->split_from_vec.size()) {
      vector<unsigned int> &vec = track->split_from_vec;
      for (unsigned int i = 0; i < vec.size(); i++) {
	if (vec[i] == id)
	  return true;
      }
    }
  }
  return false;
}
  


//// Prediction
//
// For debugging visualization, we can predict a bunch of future
// positions based on the current state and dynamic model.

// Time horizon of prediction.
static const double PREDICT_TIME = 2;
Vec2d ObjectTrack::predictions[NUM_PREDICT];

void ObjectTrack::predict_positions () {
  ObjectTrack copy_obj = *this;
  ObjectTrack *copy = &copy_obj;

  copy->set_delta_t(PREDICT_TIME/(NUM_PREDICT-1));
  // Apply V correction (if any).
  set_state_part(copy->kf_motion.state, V_X, velocity());
  for (int i = 0; i < NUM_PREDICT; i++) {
    predictions[i] = copy->position();
    copy->update_state_only();
  }
}


//// Debug dump:

// dump track state & covariance.
void ObjectTrack::dump_state (ostream &out) {
  if (valid) out <<" valid";
  if (moving) out <<" moving";
  if (ever_moving) out <<" ever_moving";
  if (do_validate) out <<" do_validate";
  out <<endl;

  out <<"# associated " <<associated_count <<endl;

  if (split_from_vec.size()) {
    out <<"# split_from ";
    for (unsigned int i = 0; i < split_from_vec.size(); i++)
      out <<split_from_vec.at(i) <<" ";
    out <<endl;
  }

  if (merged_with)
    out <<"# merged_with: " <<merged_with <<endl;

  if (segment_set) {
    out <<"# segment_set ";
    for (unsigned int i = 0; i < segment_set->tracks.size(); i++)
      out <<segment_set->tracks[i]->id <<" ";
    out <<endl;
  }

  dump_time(out);
  out.precision(8);
  out <<"position " <<position();
  out.precision(4);
  out <<" range " <<(position() - scanner->location).length() <<"\n";
  dump_time(out);
  out <<"position.sigma ";
  PointMat c2;
  kf_motion.get_submatrix(kf_center.covariance(), c2);
  print_evals(out, c2);
  out <<endl;

  Vec2d motion = state_part(kf_motion.state, P_X);
  dump_time(out);
  out <<"motion " <<motion
      <<" distance " <<motion.length() <<"\n";
  dump_time(out);
  out <<"motion.sigma ";
  kf_motion.get_submatrix(kf_motion.covariance(), c2);
  print_evals(out, c2);
  out <<endl;

  dump_time(out);
  Vec2d vel = velocity();
  out <<"velocity " <<vel
      <<" speed " <<vel.length() <<"\n";
  dump_time(out);
  out <<"velocity.sigma ";
  print_evals(out, velocity_covariance_sq());
  out <<endl;

  if (do_validate) {
    dump_time(out);
    out <<"velocity.correction " <<velocity_correction
	<<" magnitude " <<velocity_correction.length() <<"\n";
    if (history_v_covariance.at(0, 0)) {
      dump_time(out);
      out <<"velocity.correction_sigma ";
      print_evals(out, history_v_covariance);
      out <<endl;
    }
  }

  dump_time(out);
  Vec2d acc = acceleration();
  double theta, ignore;
  get_theta(theta, ignore);
  Vec2d heading(cos(theta), sin(theta));
  out <<"acceleration " <<acc
      <<" forward " <<acc.dot(heading)
      <<" right " <<acc.dot(normal(heading)) <<"\n";

  dump_time(out);
  out <<"acceleration.sigma ";
  kf_motion.get_submatrix(kf_motion.covariance(), c2, A_X);
  print_evals(out, c2);
  out <<endl;

  dump_time(out);
  out <<"rotation ";
  for (int i = 0; i < ROT_FILTER_ORDER; i++)
    out <<kf_rotation.state[i] * (180/M_PI) <<" ";
  out <<endl;

  const RotFilterMat &c3 = kf_rotation.covariance();
  dump_time(out);
  out <<"rotation.sigma ";
  for (int i = 0; i < ROT_FILTER_ORDER; i++) 
    out <<sqrt(c3.at(i, i)) * 180/M_PI <<" ";
  out <<endl;
}


void ObjectTrack::dump_TrackHistory
    (ostream &out, int &counter, const TrackHistory &hist) {
  out <<"#index " <<counter
      <<", time " <<hist.scan_time - datmo->_first_scan_time <<endl;
  out <<"#History " <<shape_class_names[hist.shape_class] <<endl;

  counter++;
  for (int tab_ix = 0; tab_ix < NUM_FEATURES; tab_ix++) {
    int feature_ix = f_ix_tab[tab_ix];
    if (valid_feature(feature_ix, hist.shape_class) && feature_ix != CENTER) 
      dump_feature(out, feature_ix, hist.features, hist.shape_class);
  }
  out <<endl <<endl;
}

  
void ObjectTrack::dump_time_and_id (ostream &out) {
  out.precision(12);
  out <<datmo->_scan_time;
  out.precision(6);
  out <<" (" <<datmo->_scan_time - datmo->_first_scan_time <<")"
      <<": Track " <<id;
  if (split_from())
    out <<":" <<split_from();
}
