/// The ObjectBase class is an abstract base class for both ObjectTrack
/// (tracks) and Segment, and holds state which is common to both.

#ifndef OBJECTBASE_H
#define OBJECTBASE_H
#include "ScannerInfo.h"
#include "utils.h"
#include <list>
#include <vector>

class Datmo;

// Represents the track of a point feature.  Used for both the vertices in
// linear features and the bounding box corners in all features.
class FeaturePoint {
public:

  // True if we think that our position is vague due to sensing concerns
  // (scanner critical angle or occlusion), and the true extent may be larger.
  // Corners will never be vague.
  bool vague;

  // Indices in state vector.
  static const int X = 0;
  static const int Y = 1;

  // Position of this feature.
  PointVec state;

  // Number of cycles on which we associated data with this track.
  // Controls initialization of residue filter.
  int associated_count;

  // Statistics of the measurement residue for this feature, which is how much
  // the measured location differed from the predicted one.  If the residue is
  // small, then [1] the actual measurement noise is low, and [2] the process
  // noise is also low, meaning that the object is following a constant
  // velocity trajectory.
  //
  // For initialization, the residue is updated recursively to compute the
  // statistics of all residue up to a certain amount of time, after which we
  // begin to decay past data according to residue_time_constant.  The cutover
  // happens at the point where the expotential decay rule gives larger
  // weighting to current data.
  //
  // If this FeaturePoint represents a measurement in a raw segment, then the
  // residue_covariance is instead a minimum value for measurement noise, from
  // parameters and feature characteristics.
  PointVec residue_mean;
  PointMat residue_covariance;

  Vec2d pos () const {
    return state;
  }

  void init (const PointData &p);

};


class ObjectBase {
 public:
  // Our Datmo object. 
  Datmo *datmo;

  // Scanner that was the data source.
  ScannerInfo *scanner;

  // unique ID of this track.  If 0, no ID assigned yet.
  unsigned int id;

  // Kinds of ObjectBase
  typedef enum {SEGMENT = 0, TRACK, DEAD_TRACK, DELETED, NUM_KINDS} TrackKind;
  // Maps kind to string name.  Please update if you change enum.
  static const char *kind_names[NUM_KINDS];

  // Segment or track.  DEAD_TRACK is a track that has died and will be
  // deleted on next iteration, but is being kept around so that we can do a
  // post-mortem report.  DELETED is set when we actually deleted the track.
  //
  TrackKind kind;

  // Track shape classifications.
  typedef enum {LINE = 0, CORNER, COMPLEX, NUM_SHAPES} TrackShape;
  static const char *shape_class_names[NUM_SHAPES];

  // Classification of object shape based on how well it can be fit by
  // one or two lines.
  //
  // LINE: fits a single line.
  //
  // CORNER: a convex corner.  See try_fit_corner for discussion of "convex".
  //
  // COMPLEX: anything else.
  //
  TrackShape shape_class;

  // disoriented and compact are boolean flags giving additional info
  // about the shape.
  //
  // disoriented means the angular orientation is not well defined, due to
  // poor fit or over-fitting.
  //
  // compact is a mitigating factor w.r.t. disoriented and COMPLEX.  The
  // segment is small and has sufficient point density that we think we can
  // track it using the bounding box or poor linear fit.
  //
  // Not all combinations of shape_class, disoriented and compact are
  // possible.  COMPLEX is always disoriented, as are all compact segments:
  //			!d !c	!d c	d !c	d c
  // LINE/CORNER	y	n	y	y
  // COMPLEX		n	n	y	y
  //
  // All small segments are disoriented, regardless of whether they pass the
  // density test and become compact.  A small object is more subject to
  // angular motion unrelated to heading, both for purely mechanical reasons,
  // and because it may be a pedestrian.  This is true even (perhaps
  // especially) if there are not enough points to track it well.
  //
  // In tracks, these are copied from the last associated segment, except that
  // non-compact is somewhat sticky, see ObjectTrack::associate.
  bool disoriented;
  bool compact;

  // Components of line (and corner) segments and tracks.  Direction always
  // points toward FIRST from LAST (or CORNER_F).  norm_dir points away from
  // the scanner in LINEs, and points from CORNER_F toward LAST in corner
  // tracks.  In a corner, we are guaranteed that both vectors point away from
  // the scanner due to the convexity requirement.
  //
  // In segments, these are precisely the directions of the sides.  In a
  // track, these are set from kf_rotation on each cycle, and the relation
  // between the these basis vectors and the feature orientation is loosened.
  // We guarantee that they will point within angle_reset_threshold of the
  // direction of FIRST in linear features.
  //
  // In moving COMPLEX tracks, the direction and norm_dir are still set from
  // the heading, direction being the direction of heading and norm_dir being
  // the normal that points away from the scanner.
  PointVec direction;
  PointVec norm_dir;

  // We have a fixed allocation of feature FeaturePoints for each track.
  // depending on the shape_class, some of these may be invalid.
  typedef enum {
    MIN_BOUND = 0, // min XY bounding corner
    MAX_BOUND = 1, // max XY bounding corner
    FIRST = 2, // start point of first segment in linear feature
    LAST = 3, // last point in linear feature.
    CORNER_F = 4, // corner in corner feature
    CENTER = 5, // object center
    NUM_FEATURES = 6}
  TrackFeature;

  // Maps feature index to name, please update if you change enum.
  static const char *feature_names[NUM_FEATURES];
  static const int f_ix_tab[NUM_FEATURES];

  // The features of this track.
  FeaturePoint features[NUM_FEATURES];

  // True if this is a valid feature index, given our shape_class.
  // shape_cl is an optional arg if for some reason, we want to know
  // w.r.t. some other shape class.
  bool valid_feature (int i, int shape_cl) const {
    return (i <= MAX_BOUND
	    || i == CENTER
	    || (shape_cl == CORNER)
	    || (shape_cl == LINE && i != CORNER_F));
  }

  bool valid_feature (int i) const {
    return valid_feature(i, shape_class);
  }

  // During data association this holds the track/segment overlap.  For a
  // track, this is the segments that overlap it.  For a segment, this is the
  // tracks that overlap it.  This is not a simple circular linkage, since in
  // general overlap is not one-to-one and not transitive.
  list <ObjectBase *> overlap;

  // During data association, this points circularly between a track and the
  // segment that it is associated with.  If this is non-NULL in a segment,
  // this indicates that that segment has been associated.  For old segments
  // in Datmo::_segment_log, the track best_match may not point back to the
  // segment, since the track may have subsequently been associated with
  // another segment.
  ObjectBase *best_match;

  // points in this track on last update.
  vector<PointData> points;

  // To what degree is this track occluded:
  //   NONE: the track should be totally visible in some scanner.
  //   PARTIAL: the track is at least partially occluded in some scanner.
  //   TOTAL: the track is believed to be completely occluded.
  //
  // In tracks this value is rather approximate.  For one thing, if
  // an object is actually occluded, we don't really know where it is,
  // so we don't know if it should be visible or not.
  //
  // A segment never has TOTAL occlusion, since we had to see something
  // to create it.
  enum {NONE, PARTIAL, TOTAL} occlusion;

  virtual ~ObjectBase ();
  ObjectBase (Datmo *datmo_arg, ScannerInfo *scanner_arg,
	      TrackKind kind_arg);
  void copy_features (const ObjectBase *obj, bool pos_only = false);

  // Return true if other overlaps this track.
  bool overlapping (ObjectBase *other);

  // Return distance of an arbitrary point from our closest feature.
  double distance (const PointVec &p);

  // Debug data logging and visualization support.
  void dump (ostream &out, int &counter, bool dump_points);
  void render (unsigned int color);

 protected:
  // dump subclass specific state.
  virtual void dump_state (ostream &out) = 0;

  // Dump # <time> to log
  void dump_time (ostream &out);

  // Dump data for one feature.
  void dump_feature (ostream &out, int feature_ix,
		     const FeaturePoint fvec[NUM_FEATURES],
		     TrackShape s_class);

  // Return true if this track is a vague line.  If both ends are vague, then it
  // is not localized longitudinally.
  bool vague_line () {
    return (shape_class == LINE && features[FIRST].vague && features[LAST].vague);
  }

 private:
  double distance_aux (const PointVec &p, int f_ix1, int f_ix2);
  bool overlapping_aux (ObjectBase *other);
};

#endif
