#include "ObjectBase.h"
#include "params.h"
#include "datmo.h"

const char *ObjectBase::kind_names[ObjectBase::NUM_KINDS] =
  {"SEGMENT", "TRACK", "DEAD_TRACK", "DELETED"};

const char *ObjectBase::shape_class_names[ObjectBase::NUM_SHAPES] =
  {"LINE", "CORNER", "COMPLEX"};

const char *ObjectBase::feature_names[ObjectBase::NUM_FEATURES] =
  {"MIN_BOUND", "MAX_BOUND", "FIRST", "LAST", "CORNER", "CENTER"};

const int ObjectBase::f_ix_tab[ObjectBase::NUM_FEATURES] =
  {ObjectBase::CENTER, ObjectBase::MIN_BOUND, ObjectBase::MAX_BOUND, ObjectBase::FIRST,
   ObjectBase::CORNER_F, ObjectBase::LAST};


//// FeaturePoint class

// Initialize new FeaturePoint from PointData
void FeaturePoint::init (const PointData &p) {
  vague = p.occluded;
  state = p.pos;

  double var = vague
    ? fsqr(_max_segment_distance * _point_spacing_sigma_factor)
    : fsqr(_disoriented_pos_noise);
  for (int i = X; i <= Y; i++)
    residue_covariance.at(i, i) = var;

  associated_count = 1;
  residue_mean = PointVec();
}

/// ObjectBase::ObjectBase

ObjectBase::ObjectBase (Datmo *datmo_arg, ScannerInfo *scanner_arg,
			TrackKind kind_arg) :
  datmo(datmo_arg),
  scanner(scanner_arg),
  id(0),
  kind(kind_arg),
  shape_class(LINE),
  disoriented(false),
  compact(false),
  best_match(NULL),
  occlusion(NONE)
{
}

ObjectBase::~ObjectBase () {};

// Copy over most ObjectBase state that is not an arg to the constructor.
// points and best_match are left alone (default init value empty and NULL.)
// If pos_only is true, only copy the feature positions, leaving the other
// data with the default values.
void ObjectBase::copy_features (const ObjectBase *obj, bool pos_only) {
  id = obj->id;
  shape_class = obj->shape_class;
  disoriented = obj->disoriented;
  compact = obj->compact;
  direction = obj->direction;
  norm_dir = obj->norm_dir;
  if (pos_only) 
    for (int i = 0; i < NUM_FEATURES; i++)
      features[i].state = obj->features[i].state;
  else
    for (int i = 0; i < NUM_FEATURES; i++)
      features[i] = obj->features[i];
  occlusion = obj->occlusion;
}


/// ObjectBase::overlapping: track correspondence

// Do half-test for overlapping.  The method is to transform all the raw
// points from "other" into the coordinate system of "this" (as defined by our
// basis vectors), then check whether they fall within our outline rectangle.
// If any do, then we overlap.  To be precise, we rotate the both our
// features and the points of other to align with the XY axes.  There
// is no translation done because this is rolled into the bounds check.
bool ObjectBase::overlapping_aux (ObjectBase *other) {
  bool need_rot = false;
  PointMat rot;
  Vec2d min_b, max_b;
  Vec2d margin(_overlap_margin, _overlap_margin);

  if (shape_class == LINE || shape_class == CORNER) {
    set_cols(rot, direction, norm_dir);
    rot.transpose(rot); // inverse
    need_rot = true;
    Vec2d fpos = rot * features[FIRST].state;
    Vec2d lpos = rot * features[LAST].state;
    if (shape_class == LINE) {
      // The features should now lie on a horizonal line, with FIRST farther
      // from the origin.

      // ### debug assertion, not really guaranteed.
      if (!(fabs(lpos[1] - fpos[1]) < max(0.5*fabs(lpos[0] - fpos[0]), 0.5)))
	WARN("line not horizontal, heading error?");

      max_b = fpos;
      min_b = lpos;
      // Force horizontal line at mean Y position.
      min_b[1] = max_b[1] = (min_b[1] + max_b[1])*0.5;
    } else {
      assert(shape_class == CORNER);
      // First and last should be the bottom right and top left corners,
      // respectively.
      max_b = Vec2d(fpos[0], lpos[1]);
      min_b = Vec2d(lpos[0], fpos[1]);
    }

    // If bounds are swapped, fall back to bounding box.
    if (min_b[0] > max_b[0] || min_b[1] > max_b[1]) {
      if (kind == TRACK)
	LOG(((ObjectTrack *)this), "overlapping_aux.bad_bounds");
      need_rot = false;
    }
  }
    
  if (!need_rot) {
    min_b = features[MIN_BOUND].state;
    max_b = features[MAX_BOUND].state;
  }

  min_b = min_b - margin;
  max_b = max_b + margin;

  for (unsigned int i = 0; i < other->points.size(); i++) {
    Vec2d p = other->points[i].pos;
    if (need_rot) p = rot * p;
    bool success = true;
    for (int j = 0; j < 2; j++)
      if (!(min_b[j] <= p[j] && p[j] <= max_b[j])) {
	success = false;
	break;
      }
    if (success) return true;
  }
  return false;
}

    
// Return true if "other" overlaps with this track.  Two tracks overlap if
// they appear to be occupying the same space.  We effectively grow the track
// by _overlap_margin to allow for tracking error.
//
// Though the actual implementation is necessarily approximate given that we
// don't know the true object extent, we do guarantee symmetry:
//    B.overlaps(A) if and only if A.overlaps(B)
//
// A moments though will reveal that overlapping is not transitive:
//   A.overlaps(B) & B.overlaps(C) does *not* imply A.overlaps(C)
//
// As it happens, our implementation (overlapping_aux) is not inherently
// symmetric, so we do both orders and combine the results with AND.
//
// Before doing the general test, we do a quick test to see if the bounding
// boxes overlap.  If they don't, then the two tracks can't overlap.  Since
// most tracks don't overlap with most segments, this is worth optimizing.
// Boxes overlap if both the X and Y ranges independently overlap.
//
bool ObjectBase::overlapping (ObjectBase *other) {
  Vec2d min_b = features[MIN_BOUND].state
    + Vec2d(-_overlap_margin, -_overlap_margin);
  Vec2d max_b = features[MAX_BOUND].state
    + Vec2d(_overlap_margin, _overlap_margin);
  const Vec2d &min_b_oth = other->features[MIN_BOUND].state;
  const Vec2d &max_b_oth = other->features[MAX_BOUND].state;
  for (int i = 0; i < 2; i++) {
    if (!(// our bounds enclose other bound
	  (min_b[i] <= min_b_oth[i] && max_b[i] >= max_b_oth[i])
	  // bounds straddle or other encloses us: our min in other bounds
	  || (min_b[i] > min_b_oth[i] && min_b[i] < max_b_oth[i])
	  // bounds straddle or other encloses us: our max in other bounds
	  || (max_b[i] > min_b_oth[i] && max_b[i] < max_b_oth[i])))
      return false;
  }
  return overlapping_aux(other) && other->overlapping_aux(this);
}


//// ObjectBase::distance
//
// Return distance from an arbitrary point our nearest feature.

double ObjectBase::distance_aux (const PointVec &p, int f_ix1, int f_ix2) {
  Vec2d cp = segClosestPoint(p, features[f_ix1].pos(), features[f_ix2].pos());
  return (p - cp).length();
}

double ObjectBase::distance (const PointVec &p) {
  switch (shape_class) {
  case LINE:
    return distance_aux(p, FIRST, LAST);

  case CORNER:
    return min(distance_aux(p, FIRST, CORNER_F),
	       distance_aux(p, CORNER_F, LAST));

  default:
    ERROR("Bad shape class: " <<shape_class);
  }
}
	      

//// Debug dump

void ObjectBase::dump_time (ostream &out) {
  out.precision(6);
  out <<"# " <<datmo->_scan_time - datmo->_first_scan_time <<" ";
}

// dump feature state and residue.
void ObjectBase::dump_feature
    (ostream &out, int feature_ix, const FeaturePoint fvec[NUM_FEATURES],
     TrackShape s_class)
{
  const FeaturePoint &feature = fvec[feature_ix];
  out <<"#Feature " <<feature_names[feature_ix]
      <<", associated " <<feature.associated_count;
  if (feature.vague) out <<" vague";
  out <<endl;

  out.precision(8);
  out <<"#  state " <<feature.state <<endl;

  if (kind == TRACK) {
    out.precision(4);
    out <<"#  residue mean ";
    for (int i = 0; i < POINT_ORDER; i++) {
      out <<" " <<feature.residue_mean[i];
    }
    out <<endl;
  }

  out <<"#  residue sigma ";
  print_evals(out, feature.residue_covariance);
  out <<endl;

  // Print feature data for gnuplot.
  Vec2d minp = fvec[MIN_BOUND].pos();
  Vec2d maxp = fvec[MAX_BOUND].pos();
  double incr = (maxp - minp).length() * 0.1;
  out.precision(8);
  if (feature_ix == CENTER) {
    Vec2d side1 = feature.pos() + norm_dir*incr;
    Vec2d side2 = feature.pos() + norm_dir*-incr;
    Vec2d tip = feature.pos() + direction*incr;
    if (kind == SEGMENT) {
      if (s_class == LINE || s_class == CORNER) {
	// draw a tee thingie with rotation.
	out <<side1 <<endl;
	out <<side2 <<endl;
	out <<endl;
	out <<feature.pos() <<endl;
	out <<tip <<endl;
	out <<endl;
      } else {
	// no rotation, draw a +.
	double dincr = sqrt(2.0) * incr * 0.5;
	out <<feature.pos() + Vec2d(-dincr, -dincr) <<endl;
	out <<feature.pos() + Vec2d(dincr, dincr) <<endl;
	out <<endl;
	out <<feature.pos() + Vec2d(-dincr, dincr) <<endl;
	out <<feature.pos() + Vec2d(dincr, -dincr) <<endl;
	out <<endl;
      }
    } else {
      // draw an arrowhead 
      out <<side1 <<endl;
      out <<tip <<endl;
      out <<side2 <<endl;
      out <<endl;
    }
  } else if (feature_ix <= MAX_BOUND) {
    double side = (feature_ix == MIN_BOUND) ? incr : -incr;
    // Draw crop marks for bounding box complex.
    if (s_class == COMPLEX) {
      out <<feature.pos() + Vec2d(side, 0) <<endl;
      out <<feature.pos() <<endl;
      out <<feature.pos() + Vec2d(0, side) <<endl;
      out <<endl;
    }
  } else {
    // end or corner.
    out <<feature.pos() <<endl;
    if (feature_ix == LAST)
      out <<endl;
  }
}

// Dump track data to out.  Counter is the data block index in the file, and
// dump_points says whether to dump the points.
void ObjectBase::dump (ostream &out, int &counter, bool dump_points) 
{
  out <<"#index " <<counter
      <<", time " <<datmo->_scan_time - datmo->_first_scan_time
      <<", scanner " <<scanner->name <<endl;
  out <<"#" <<kind_names[kind] <<" "
      <<shape_class_names[shape_class]
      <<", id " <<id;
  out <<", size " <<points.size() <<": " <<endl;
  counter++;

  out <<"#   ";
  if (disoriented) out <<" disoriented";
  if (compact) out <<" compact";
  if (occlusion == PARTIAL) out <<" partial_occlusion";
  if (occlusion == TOTAL) out <<" total_occlusion";

  dump_state(out);

  for (int tab_ix = 0; tab_ix < NUM_FEATURES; tab_ix++) {
    int feature_ix = f_ix_tab[tab_ix];
    if (valid_feature(feature_ix)) 
      dump_feature(out, feature_ix, features, shape_class);
  }

  if (dump_points) {
    out.precision(8);
    out <<"\n\n#scan data, index " <<counter++ <<endl;
    for (unsigned int i = 0; i < points.size(); i++) {
      out <<points[i].pos <<" # " <<points[i].scan_pos;
      if (points[i].occluded)
	out <<" (occluded)";
      out <<endl;
    }
    out <<endl;
  }
    
  out <<endl;
}

