#ifndef KALMANFILTER_H

#define KALMANFILTER_H

#include <utils/SquareMatrix.h>
using namespace utils;

template <class T, int filter_ord, int meas_ord>
class KalmanFilter {
 public:
  KalmanFilter () :
    failure(false),
    _covariance_valid(false),
    _information_valid(false)
    {};
    
  // System state, a.k.a. x_hat.
  LinearVector<T, filter_ord> state;

  // Set when an operation fails due to numerical problems (i.e. matrix not
  // symmetric positive definite.)  This is sticky, it must be explicitly
  // cleared.
  bool failure;

  // Set diagonal submatrix at offset.
  static void set_submatrix (SquareMatrix<T, filter_ord> &m,
			     const SquareMatrix<T, meas_ord> &p,
			     int offset = 0) {
    for (int i = 0; i < meas_ord; i++)
      for (int j = 0; j < meas_ord; j++)
	m.at(i + offset, j + offset) = p.at(i, j);
  }

  // Get diagonal submatrix at offset.
  static void get_submatrix (const SquareMatrix<T, filter_ord> &m,
			     SquareMatrix<T, meas_ord> &p,
			     int offset = 0) {
    for (int i = 0; i < meas_ord; i++)
      for (int j = 0; j < meas_ord; j++)
	p.at(i, j) = m.at(i + offset, j + offset);
  }

  // Make state vector from measurement vector by zero pad.
  static LinearVector<T, filter_ord> make_state
      (const LinearVector<T, meas_ord> &v)
    {
      const T *pd = v.getValue();
      T data[filter_ord];
      for (int i = 0; i < meas_ord; i++)
	data[i] = pd[i];
      for (int i = meas_ord; i < filter_ord; i++)
	data[i] = 0;
      return LinearVector<T, filter_ord>((T *)&data);
    }
    
  // Pre-measurement covariance estimate:
  //     P_k- = phi * P_k+(delayed) * phi' + Lambda * Q * Lambda'
  //
  // This reflects the loss of information due to process noise and propagates
  // state correlations from phi to P_k.
  void process (const SquareMatrix<T, filter_ord> &phi,
		const SquareMatrix<T, filter_ord> &Lambda_Q_Lambda_transpose)
    {
      covariance();
      SquareMatrix<T, filter_ord> new_cov;
      new_cov.transpose(phi); // phi'
      new_cov.mult(_covariance, new_cov); // P_k+ *
      new_cov.mult(phi, new_cov); // phi *
      new_cov.add(new_cov, Lambda_Q_Lambda_transpose); // + Lambda * Q * Lambda'
      _information_valid = false;
      _covariance = new_cov;
    }

  // Update covariance estimate for measurement.
  //   P_k+ = [(P_k-)^(-1) + (H' * R_k^(-1) * H)]^(-1)
  void measure (const SquareMatrix<T, meas_ord> &R_k) {
    information();

    SquareMatrix<T, meas_ord> R_k_inverse;
    if (!R_k_inverse.spd_inverse(R_k)) {
      WARN("Couldn't invert R_k:\n" <<R_k);
      failure = true;
    }
  
    SquareMatrix<T, filter_ord> tmp;
    set_submatrix(tmp, R_k_inverse);  // H' * ... * H
    _information.add(_information, tmp); // (P_k-)^(-1) +
    _covariance_valid = false;
  }

  // The Kalman gain, as computed on the last call to gain.
  SquareMatrix<T, filter_ord> K;

  // Compute Kalman gain for measurement:
  //    K = P_k- * H' * (H * P_k- * H' + R_k)^(-1)
  const SquareMatrix<T, filter_ord> &gain
      (const SquareMatrix<T, meas_ord> &R_k)
    {
      covariance();
      SquareMatrix<T, meas_ord> tmp;
      get_submatrix(_covariance, tmp); // H * P_k- * H'
      tmp.add(tmp, R_k); // + R_k
      if (!tmp.spd_inverse(tmp)) { // quantity inverse
	failure = true;
	WARN("Couldn't invert H * P_k- * H' + R_k:\n" <<tmp);
      }

      K.clear();
      set_submatrix(K, tmp); // H' *
      K.mult(_covariance, K); // Pk- *
      return K;
    }

  // Return a const reference to the covariance, converting from information
  // representation if necessary.
  const SquareMatrix<T, filter_ord> &covariance () {
    if (!_covariance_valid) {
      if (_covariance.spd_inverse(_information)) {
	_covariance_valid = true;
      } else {
	_covariance_valid = false;
	WARN("Couldn't invert information to covariance:\n" <<_information);
	failure = true;
      }
    }
    return _covariance;
  }

  // Return a const reference to the information, converting from covariance
  // representation if necessary.
  const SquareMatrix<T, filter_ord> &information () {
    if (!_information_valid) {
      if (_information.spd_inverse(_covariance)) {
	_information_valid = true;
      } else {
	_information_valid = false;
	WARN("Couldn't invert covariance to information:\n" <<_covariance);
	failure = true;
      }
    }
    return _information;
  }

  // Set the covariance.
  void set_covariance (const SquareMatrix<T, filter_ord> &cov) {
    _covariance = cov;
    _covariance_valid = true;
    _information_valid = false;
  }

  // Set the information.
  void set_information (const SquareMatrix<T, filter_ord> &info) {
    _information = info;
    _information_valid = true;
    _covariance_valid = false;
  }

 private:
  bool _covariance_valid;
  SquareMatrix<T, filter_ord> _covariance;

  bool _information_valid;
  SquareMatrix<T, filter_ord> _information;

};

#endif
