#ifndef _PORT_H
#define _PORT_H

/*
 * Standard portability stuff - byte ordering, fixed width types, etc.
 */

#include <utils/formatting/FormConfig.h>
#include <stdio.h>

#if BYTE_ORDER==LITTLE_ENDIAN
#define HOST_IS_LITTLE_ENDIAN
#elif BYTE_ORDER==BIG_ENDIAN
#define HOST_IS_BIG_ENDIAN  
#endif

#if (defined HOST_IS_LITTLE_ENDIAN) && (defined HOST_IS_BIG_ENDIAN)
#error "Both endian flags defined!  What do you think this is, ARM?"
#endif
#if (!defined HOST_IS_LITTLE_ENDIAN) && (!defined HOST_IS_BIG_ENDIAN)
#error "No endian flag defined.  What do you think this is, Switzerland?"
#endif


#include <stdint.h>

static inline uint64_t swap_endian_64(uint64_t data) 
{
	return (data >> 56) 
		| (data << 56)
		| ((data & 0xff00) << 40)
		| ((data >> 40) & 0xff00)
		| ((data & 0xff0000) << 24)
		| ((data >> 24) & 0xff0000)
		| ((data & 0xff000000) << 8)
		| ((data >> 8) & 0xff000000);
}

static inline uint32_t swap_endian_32(uint32_t data)
{
	return (data >> 24)
		| (data << 24)
		| ((data >> 8) & 0xff00)
		| ((data & 0xff00) << 8);
}

static inline uint16_t swap_endian_16(uint16_t data)
{
	return (data >> 8)
		| (data << 8);
}

static inline double swap_endian_double(double data)
{
	uint64_t tmp = swap_endian_64(*(uint64_t *)&data); 
	return *(double *)&tmp;
}

static inline double swap_endian_float(float data)
{
	uint32_t tmp = swap_endian_32(*(uint32_t *)&data); 
	return *(float *)&tmp;
}

/*
  These literal and printf conversion macros assume 
   sizeof(int) == 4
   sizeof(long) == 4
   sizeof(long long) == 8
*/

#define UINT64(lit) (lit##ULL)
#define INT64(lit) (lit##LL)

#define PF_64U "%llu"
#define PF_64I "%lli"
#define PF_64X "%llx"
#define PF_32U "%u"
#define PF_32I "%i"
#define PF_32X "%x"


#ifdef HOST_IS_LITTLE_ENDIAN
#define host_to_le_double(data)  (data)
#define host_to_le_float(data)  (data)
#define host_to_le_int64(data)  (data)
#define host_to_le_int32(data)  (data)
#define host_to_le_int16(data)  (data)
#define host_to_le_uint64(data) (data)
#define host_to_le_uint32(data) (data)
#define host_to_le_uint16(data) (data)
#define le_to_host_double(data)  (data)
#define le_to_host_float(data)  (data)
#define le_to_host_int64(data)  (data)
#define le_to_host_int32(data)  (data)
#define le_to_host_int16(data)  (data)
#define le_to_host_uint64(data) (data)
#define le_to_host_uint32(data) (data)
#define le_to_host_uint16(data) (data)

#define host_to_be_double(data)  swap_endian_double((double)(data))
#define host_to_be_float(data)   swap_endian_float((float)(data))
#define host_to_be_int64(data)   swap_endian_64((uint64_t)(data))
#define host_to_be_int32(data)   swap_endian_32((uint32_t)(data))
#define host_to_be_int16(data)   swap_endian_16((uint16_t)(data))
#define host_to_be_uint64(data)  swap_endian_64(data)
#define host_to_be_uint32(data)  swap_endian_32(data)
#define host_to_be_uint16(data)  swap_endian_16(data)
#define be_to_host_double(data)  swap_endian_double((double)(data))
#define be_to_host_float(data)   swap_endian_float((float)(data))
#define be_to_host_int64(data)   swap_endian_64((uint64_t)(data))
#define be_to_host_int32(data)   swap_endian_32((uint32_t)(data))
#define be_to_host_int16(data)   swap_endian_16((uint16_t)(data))
#define be_to_host_uint64(data)  swap_endian_64(data)
#define be_to_host_uint32(data)  swap_endian_32(data)
#define be_to_host_uint16(data)  swap_endian_16(data)

#else
/* We're big endian.  */
#define host_to_be_double(data)  (data)
#define host_to_be_float(data)  (data)
#define host_to_be_int64(data)  (data)
#define host_to_be_int32(data)  (data)
#define host_to_be_int16(data)  (data)
#define host_to_be_uint64(data) (data)
#define host_to_be_uint32(data) (data)
#define host_to_be_uint16(data) (data)
#define be_to_host_double(data)  (data)
#define be_to_host_float(data)  (data)
#define be_to_host_int64(data)  (data)
#define be_to_host_int32(data)  (data)
#define be_to_host_int16(data)  (data)
#define be_to_host_uint64(data) (data)
#define be_to_host_uint32(data) (data)
#define be_to_host_uint16(data) (data)

#define host_to_le_double(data)  swap_endian_double((double)(data))
#define host_to_le_float(data)   swap_endian_float((float)(data))
#define host_to_le_int64(data)   swap_endian_64((uint64_t)(data))
#define host_to_le_int32(data)   swap_endian_32((uint32_t)(data))
#define host_to_le_int16(data)   swap_endian_16((uint16_t)(data))
#define host_to_le_uint64(data)  swap_endian_64(data)
#define host_to_le_uint32(data)  swap_endian_32(data)
#define host_to_le_uint16(data)  swap_endian_16(data)
#define le_to_host_double(data)  swap_endian_double((double)(data))
#define le_to_host_float(data)   swap_endian_float((float)(data))
#define le_to_host_int64(data)   swap_endian_64((uint64_t)(data))
#define le_to_host_int32(data)   swap_endian_32((uint32_t)(data))
#define le_to_host_int16(data)   swap_endian_16((uint16_t)(data))
#define le_to_host_uint64(data)  swap_endian_64(data)
#define le_to_host_uint32(data)  swap_endian_32(data)
#define le_to_host_uint16(data)  swap_endian_16(data)

#endif

static inline void save_double(double val, FILE *file)
{
	val = host_to_le_double(val);
	fwrite(&val, 8, 1, file);
}

static inline double load_double(FILE *file)
{
	double tmp;
	fread(&tmp, 8, 1, file);
	return(le_to_host_double(tmp));
}


static inline void save_float(float val, FILE *file)
{
	val = host_to_le_float(val);
	fwrite(&val, 4, 1, file);
}

static inline float load_float(FILE *file)
{
	float tmp;
	fread(&tmp, 4, 1, file);
	return(le_to_host_float(tmp));
}

static inline void save_uint32(uint32_t val, FILE *file)
{
	val = host_to_le_uint32(val);
	fwrite(&val, 4, 1, file);
}

static inline uint32_t load_uint32(FILE *file)
{
	uint32_t val;
	fread(&val, 4, 1, file);
	return le_to_host_uint32(val);
}



#endif /* _PORT_H */
