// This file defines macros for text messaging, primarily as a placeholder
// for verbosity control and multiprocessing versions later on.

// The plan is to make these macros reference a variable which should
// contain an object that is a connection to message clearinghouse
// server.  This could be either a global or instance variable.  Each
// process (and potentially subactivities in a process) would open a
// connection to the server.  The server can then filter and present
// messages based on the originating process, the severity, and the
// specific message (the line number.)  It is (for example) usually
// not interesting to see 10,000 printouts of the same message.

#ifndef MESSAGE_H
#define MESSAGE_H

#include "utils/Basic.h"

// None of this actually works if iostream is absent, but it will compile.
#ifdef HAVE_IOSTREAM
#include <iostream>
#endif

#include <string.h>
#include <errno.h>

__UTILS_BEGIN_NAMESPACE

void _print_time ();

#define __MESSAGE(stuff) {\
  utils::_print_time();\
  cerr <<__FILE__ <<":" <<__LINE__ <<": " <<stuff <<endl;\
}

// DEBUG is used for output that usually doesn't indicate any sort of
// problem, and that may later be deleted from the source.
#define DEBUG(stuff) __MESSAGE("debug: " <<stuff)

// Indicates a condition that is somewhat unexpected or noteworthy,
// but may not be a problem.
#define NOTE(stuff) __MESSAGE("note: " <<stuff)

// Indicates a definite problem, but one that we think we can proceed
// from in some way.
#define WARN(stuff) __MESSAGE("warning: " <<stuff)

// Indicates a fatal error in the calling code.  Does not return.
#define ERROR(stuff) {\
  __MESSAGE("error: " <<stuff);\
  abort();\
}


// Like perror.
#define PWARN(stuff) WARN(stuff <<": " <<strerror(errno))

// Like perror, only a fatal error.
#define PERROR(stuff) ERROR(stuff <<": " <<strerror(errno))


// If exp is false, signal an ERROR (don't return.) 
#define ASSERT(exp, stuff) {\
  if (!(exp))\
    ERROR(stuff);\
}


// printf compatible interface.  Since the preprocessor doesn't have
// varargs, we use a function.  For convenience, macros are defined to
// supply the first three args analogously to the stream versions.
// So we can say:
//    mprintf(MSG_WARN, "horrible lossage %d", x)

enum __MprintfSeverity {__MP_DEBUG, __MP_NOTE, __MP_WARN,
			 __MP_ERROR};

#define MSG_DEBUG __FILE__, __LINE__, __MP_DEBUG
#define MSG_NOTE __FILE__, __LINE__, __MP_NOTE
#define MSG_WARN __FILE__, __LINE__, __MP_WARN
#define MSG_ERROR __FILE__, __LINE__, __MP_ERROR

void mprintf (char *file, int line, __MprintfSeverity severity, char *format,
	      ...);

__UTILS_END_NAMESPACE

#endif
