///////////////////////////////////////////////////////////////////////////////
//
//                               UnpackSizeAction.h
//
// Implements an action which takes a flat buffer of data with no alignment 
// padding and builds it into the correct data structure
//
// Classes defined for export:
//     UnpackSizeAction
//     
///////////////////////////////////////////////////////////////////////////////

#ifndef utils_unpack_size_action_h
#define utils_unpack_size_action_h

#include <string.h>

#include <utils/formatting/FormatAction.h>
#include <utils/formatting/FormConfig.h>

__UTILS_BEGIN_NAMESPACE

class UnpackSizeAction : public FormatAction {
  UTILS_BASE_HEADER(UnpackSizeAction);
 public:
  UnpackSizeAction(unsigned char* buffer=0, int buf_index=0,
                   int byte_order=BYTE_ORDER);

  void setBuffer(unsigned char* buffer, int buf_index, int byte_order);

  int getIndex() const { return _buf_index; }
  unsigned char* getBuffer() const { return _buffer; }
  void setIndex(int i) { _buf_index = i; }

  virtual bool actInt();
  virtual bool actShort();
  virtual bool actLong();
  virtual bool actFloat();
  virtual bool actDouble();
  virtual bool actChar();
  virtual bool actLength(int size);

  virtual bool actString();
  virtual bool actPtr(FormatSpec*);

  virtual bool actFixedArray(const Vector<int>& sizes, FormatSpec* fmt);
  virtual bool actVarArray(StructSpec*,
                           const Vector<int>&, FormatSpec*);

  static int unpackSize(FormatSpec*, unsigned char*, int byte_order);
  static void initClass();

  int getInt();

  // transfer from the buffer to the dest
  void fromBuffer(void* dest, int size) {
    if (dest != _buffer)
      memcpy((char*) dest, (char*) _buffer+_buf_index, size);
    _buf_index += size;
  }

  // advance the buffer index by length
  void advanceBuffer(int length) { _buf_index += length; }

 private:
#if ((BYTE_ORDER == LITTLE_ENDIAN) || (BYTE_ORDER == BIG_ENDIAN))

  void revIntBytes(unsigned char* dest) {
    unsigned char* src = _buffer+_buf_index;
    if (src == dest) {
      unsigned char s0 = src[0];
      unsigned char s1 = src[1];
      unsigned char s2 = src[2];
      dest[0] = src[3];
      dest[1] = s2;
      dest[2] = s1;
      dest[3] = s0;
    } else {
      dest[0] = src[3];
      dest[1] = src[2];
      dest[2] = src[1];
      dest[3] = src[0];
    }
  }

  void bytesToInt(unsigned char* dest) {
    if (_byte_order == BYTE_ORDER) 
      fromBuffer(dest, sizeof(int));
    else {
      revIntBytes(dest);
      _buf_index += 4;
    }
  }

#else /* not little endian or big endian, must be pdp endian.*/

  void revIntBytes(unsigned char* dest) {
    *((int*) dest) = htonl(*(int*) (_buffer+_buf_index));
  }

  void bytesToInt(unsigned char* dest) {
    fromBuffer(dest, sizeof(int));
    if (_byte_order != BYTE_ORDER)
      *((int*) dest) = htonl(*(int*) dest);
  }

#endif /* LITTLE_ENDIAN || BIG_ENDIAN */

 private:
  int _buf_index;            // current index into the buffer
  unsigned char* _buffer;    // the buffer
  int _byte_order;           // the byte order of the data
};

__UTILS_END_NAMESPACE

#endif
