///////////////////////////////////////////////////////////////////////////////
//
//                               PackAction.h
//
// Implements an action which packs data into a flat buffer with all 
// alignment padding stripped out
//
// Classes defined for export:
//     PackAction
//     
///////////////////////////////////////////////////////////////////////////////

#ifndef utils_pack_action_h
#define utils_pack_action_h

#include <utils/Basic.h>
#include <string.h>
#include <utils/formatting/DataAction.h>

__UTILS_BEGIN_NAMESPACE

class PackAction : public DataAction {
    UTILS_BASE_HEADER(PackAction);
  public:
    PackAction(unsigned char* buffer=0, int buf_size=0, void* data=NULL,
                 int index=0);

    // pack into a buffer of length buf_size starting at index
    void setBuffer(unsigned char* buffer, int buf_size, int index=0);

    // copy len bytes from data into the buffer and increment the index
    bool toBuffer(unsigned char* data, int len) {
        if (!overflow()) {
            bool bad = false;
            if (_buf_index + len > _buf_size) // make sure we don't overflow
                len = _buf_size-_buf_index;
            memcpy(_buffer+_buf_index, data, len);
            if (bad) _buf_index = _buf_size+1; else _buf_index += len;
            return true;
        } else
            return false;
    }
    // return if we have overflowed the buffer
    bool overflow() const { return _buf_index > _buf_size; }

    int getIndex() const { return _buf_index; }
    unsigned char* getBuffer() const { return _buffer; }
    int getBufSize() const { return _buf_size; }
    void setIndex(int i) { _buf_index = i; }

    virtual bool actInt();
    virtual bool actShort();
    virtual bool actLong();
    virtual bool actFloat();
    virtual bool actDouble();
    virtual bool actChar();
    virtual bool actLength(int size);

    virtual bool actString();
    virtual bool actPtr(FormatSpec*);

    virtual bool actFixedArray(const Vector<int>& sizes, FormatSpec* fmt);
    virtual bool actVarArray(StructSpec*,
                             const Vector<int>&, FormatSpec*);

    static bool pack(FormatSpec*, unsigned char*, int, void*, int=0);
    static void initClass();

  private:
    void packPtr(FormatSpec* fmt);

  private:
    int _buf_size;
    int _buf_index;
    unsigned char* _buffer;
};

__UTILS_END_NAMESPACE

#endif
