///////////////////////////////////////////////////////////////////////////////
//
//                              FormatAction.h
//
// Defines the foFormat action.  A foFormat action implements member functions
// which will process the standard foFormat specifiers to perform an action,
// such as calculating the size of the data defined by a foFormat specifier, or 
// printing out that data, etc.  Custom specifiers can key on the class type
// to implement their own responses to that action.
//
// Classes defined for export:
//   FormatAction - the base foFormat action class
//
///////////////////////////////////////////////////////////////////////////////

#ifndef utils_foFormat_action_h
#define utils_foFormat_action_h

#include <utils/BaseSub.h>

__UTILS_BEGIN_NAMESPACE

class FormatSpec;
class StructSpec;
template <class T> class Vector;
struct FormatSymbol;

class FormatAction : public Base {
    UTILS_BASE_ABSTRACT_HEADER(FormatAction);

  public:
    FormatAction();
    virtual ~FormatAction();

    // act on primitive types
    virtual bool actInt() = 0;
    virtual bool actShort() = 0;
    virtual bool actLong() = 0;
    virtual bool actFloat() = 0;
    virtual bool actDouble() = 0;
    virtual bool actChar() = 0;
    virtual bool actString() = 0;
    virtual bool actLength(int) = 0;

    // act on pointer to self referencing structure as specified by parent
    virtual bool actSelfPtr(StructSpec* parent);
    // act data as specified by fmt, but which is pointed to
    virtual bool actPtr(FormatSpec* fmt) = 0;

    // act on a data structure with size elements and elements elems
    virtual bool actStruct(int size, FormatSpec** elems);
    // default actStruct is built on these functions
    virtual bool startStruct() { return true; }
    virtual bool endStruct() { return true; }
    // act on the ith elemen of the size elems
    virtual bool actStructElem(int i, int size, FormatSpec** elems);

    // act on a fixed array of fmt's with dimensions in the list indices
    virtual bool actFixedArray(const Vector<int>& indices,
                               FormatSpec* fmt) = 0;
    // act on a variable size array embedded in parent with dimensions found
    // at indices within parent and made of fmt's
    virtual bool actVarArray(StructSpec* parent,
                             const Vector<int>& indices, FormatSpec* fmt) = 0;

    // return the size of a fixed array given its list of indices
    int fixedArraySize(const Vector<int>& indices);

    static void initClass();
    static void initClasses(); 

    void* get(const char*) const;
    bool set(const char*, void*);
    void inherit(const FormatAction&);
    void clear();

  private:
    FormatSymbol* _head;  // optional tag/value pair table
};    

__UTILS_END_NAMESPACE

#endif
