///////////////////////////////////////////////////////////////////////////////
//
//                               Format.h
//
// Defines standard format specifiers.  All standard format specifiers have
// corresponding member functions of FormatAction to process them 
// appropriately for different subclasses of FormatAction.  Custom format
// specifiers should use the facilities defined by CustomSpec
//
// Classes defined for export:
//   FormatSpec - the format specifier base class
//   IntSpec, ShortSpec, LongSpec, FloatSpec, CharSpec, DoubleSpec,
//     StringSpec - primitive, standard format specifiers
//   StructSpec - a collection of data types, i.e. a struct
//   FixedArraySpec - a fixed number of a data type
//   VarArraySpec - a variable number of a data type
//   PtrSpec - specifies pointer to data
//   SelfPtrSpec - pointer to the StructSpec that contains it
//   LengthSpec - specified an unstructured run of bytes with a fixed length
//   
///////////////////////////////////////////////////////////////////////////////

#ifndef utils_format_h
#define utils_format_h

#include <utils/BaseSub.h>

__UTILS_BEGIN_NAMESPACE

class FormatAction;
class FormatParser;
template <class T> class Vector;
class TokenReader;
class StructSpec;

// class will do something with an action according to the format spec.
class FormatSpec : public Base {
    UTILS_BASE_ABSTRACT_HEADER(FormatSpec);

  public:
    FormatSpec();

    // This defines how a format reacts to various format actions
    virtual bool act(FormatAction*) = 0;

    // returns true if format is a simple, primitive data type, i.e., is not 
    // a pointer to other data like a string or is not a struct
    virtual bool isSimpleType() const { return false; }

    // returns true if format is already completely packed, i.e., the data
    // could just be copied to a buffer to pack it in
    bool isPackedType() const;

    // returns true if format specifes a flat data structure
    bool isFlatType() const;  
    int dataSize() const;  // returns size of data structure specified

    // used for parameterized user defined format types
    virtual FormatSpec* processParameters(FormatParser*) { return 0L; }

    static void initClass();

  private:
    int _size;      // data structure size cache
    bool _packed; // alignment padding cache
    int _flat;     // flatness cache
};

// macros for declaring primitive specifiers
#define UTILS__SPEC_NAME(name) UTILS__CONCAT(name, Spec)
#define UTILS__DEF_SPEC(name, type, simpleType) \
class UTILS__SPEC_NAME(name) : public FormatSpec { \
  UTILS_BASE_HEADER(UTILS__SPEC_NAME(name)); \
  public: \
    UTILS__SPEC_NAME(name)(); \
    virtual bool act(FormatAction* action); \
    virtual bool isSimpleType() const { return simpleType; } \
    static void initClass(); \
}

// declare primitive specifiers
UTILS__DEF_SPEC(Int, int, true);        //  IntSpec
UTILS__DEF_SPEC(Short, short, true);    //  ShortSpec
UTILS__DEF_SPEC(Long, long, true);      //  LongSpec
UTILS__DEF_SPEC(Float, float, true);    //  FloatSpec
UTILS__DEF_SPEC(Char, char, true);      //  CharSpec
UTILS__DEF_SPEC(Double, double, true);  //  DoubleSpec
UTILS__DEF_SPEC(String, string, false); //  LongSpec

// declare the complex format specifiers

class StructSpec : public FormatSpec {
    UTILS_BASE_HEADER(StructSpec);
  public:
    StructSpec();
    virtual ~StructSpec();

    int getSize() const { return _size; }
    FormatSpec* getFormat(int i) {
        return ((i>=0&&i<_size)?_fmts[i]:((FormatSpec*) NULL)); }

    virtual bool act(FormatAction* action);
    virtual bool isSimpleType() const { return false; }

    static void initClass();

    void setStruct(const Vector<FormatSpec*>& fmts);
    int numElems() const { return _size; }
    FormatSpec** formats() const { return _fmts; }

  private:
    void init(int, FormatSpec**);

  private:
    int _size;
    FormatSpec** _fmts;
};

class FixedArraySpec : public FormatSpec{
    UTILS_BASE_HEADER(FixedArraySpec);
  public:
    FixedArraySpec(Vector<int>* sizes, FormatSpec* fmt);
    FixedArraySpec();
    virtual ~FixedArraySpec();

    const Vector<int>* getSizes() const { return _sizes; }
    FormatSpec* getSubFormat() const { return _fmt; }

    virtual bool act(FormatAction* action);
    virtual bool isSimpleType() const { return false; }

    static void initClass();

  private:
    Vector<int>* _sizes;
    FormatSpec* _fmt;
};

class VarArraySpec : public FormatSpec {
    UTILS_BASE_HEADER(VarArraySpec);
  public:
    VarArraySpec(StructSpec* parent, Vector<int>* indices,
                   FormatSpec* fmt);
    VarArraySpec();
    virtual ~VarArraySpec();

    const Vector<int>* getIndices() const { return _indices; }
    FormatSpec* getSubFormat() const { return _fmt; }

    virtual bool act(FormatAction* action);
    virtual bool isSimpleType() const { return false; }

    static void initClass();

  private:
    Vector<int>* _indices;
    StructSpec* _parent;
    FormatSpec* _fmt;
};

class SelfPtrSpec : public FormatSpec {
    UTILS_BASE_HEADER(SelfPtrSpec);
  public:
    SelfPtrSpec();
    SelfPtrSpec(StructSpec* parent);
    
    virtual bool act(FormatAction* action);
    virtual bool isSimpleType() const { return false; }

    static void initClass();
    
  private:
    StructSpec* _parent;
};

class PtrSpec : public FormatSpec {
    UTILS_BASE_HEADER(PtrSpec);
  public:
    PtrSpec(FormatSpec* fmt);
    PtrSpec();
    virtual ~PtrSpec();

    FormatSpec* getPointerFormat() const { return _fmt; }

    virtual bool act(FormatAction* action);
    virtual bool isSimpleType() const { return false; }

    static void initClass();
    
  private:
    FormatSpec* _fmt;
};

// and the really simple format specifier

class LengthSpec : public FormatSpec {
    UTILS_BASE_HEADER(LengthSpec);
  public:
    LengthSpec(int n=0);

    int getLength() const { return _length; }
    
    virtual bool act(FormatAction* action);
    virtual bool isSimpleType() const { return true; }

    static void initClass();

  private:
    int _length;
};

__UTILS_END_NAMESPACE

#endif
