///////////////////////////////////////////////////////////////////////////////
//
//                               DataSizeAction.h
//
// Implements an abstract format action class which maintains an index into
// a data structure.  It can be used by itself to find the size of a formatted
// data structure, but it is also used as a superclass by classes which 
// need to process data structures (specifically DataAction).  It can also
// be used to see if a data structure is "pre-packed," i.e., if a data 
// structure has no alignment padding it is considered pre-packed for 
// flattening
//
// Classes defined for export:
//     DataSizeAction
//     
///////////////////////////////////////////////////////////////////////////////

#ifndef utils_data_size_action_h
#define utils_data_size_action_h

#include <utils/formatting/FormatAction.h>

__UTILS_BEGIN_NAMESPACE

class DataSizeAction : public FormatAction {
    UTILS_BASE_HEADER(DataSizeAction);
  public:
    DataSizeAction();

    void resetSize() { _index = 0; _packed = true; }
    int getSize() const { return _index; }
    void addSize(int s) { _index += s; }
    bool isPacked() const { return _packed; }
    void needsPacking() { _packed = false; }
    void setIndex(int index) { _index = index; }

    virtual bool actInt() { _index += sizeof(int); return true; }
    virtual bool actShort() { _index += sizeof(short); return true; }
    virtual bool actLong() { _index += sizeof(long);  return true; }
    virtual bool actFloat() { _index += sizeof(float); return true; }
    virtual bool actDouble() { _index += sizeof(double); return true; }
    virtual bool actChar() { _index += sizeof(char); return true; }
    virtual bool actString() {
        _index += sizeof(char*); _packed = false;
        return true;
    }

    virtual bool actLength(int size) { _index += size; return true; }
    virtual bool actPtr(FormatSpec*) {
        _index += sizeof(void*); _packed = false; return true; }

    virtual bool actStructElem(int i, int size, FormatSpec** elems);
    virtual bool actFixedArray(const Vector<int>& sizes, FormatSpec* fmt);
    virtual bool actVarArray(StructSpec*,
                             const Vector<int>&, FormatSpec*);

    static void initClass();

  protected:
    int alignField(int formatIndex, int num_elems, FormatSpec** elems);

  private:
    int _index;  // current index into the data structure
    bool _packed; // true if a format is already packed
};

__UTILS_END_NAMESPACE

#endif
