///////////////////////////////////////////////////////////////////////////////
//
//                               CustomSpec.h
//
// Defines classes to assist in creating custom format specifiers
//
// Classes defined for export:
//     CustomSpec - a general framework for custom format specifiers.
//            Users register callbacks to respond to types of actions
//     CustomPtrSpec - a subclass specifically for entities that are pointers
//     ApplyFormatAction - A callback action used to respond to an action 
//            type
//     ApplyFormatRoutine - Convenience callback to use a function pointer
//     
// Templates defined for export
//     ClassApplyFormatAction<T> - convenience template for an action 
//         which executes a member function of class T
//
///////////////////////////////////////////////////////////////////////////////

#ifndef utils_custom_spec_h
#define utils_custom_spec_h

#include <utils/formatting/Format.h>
#include <utils/Managed.h>

__UTILS_BEGIN_NAMESPACE

template <class T> class Dict;

class ApplyFormatAction : public Managed {
  public:
    ApplyFormatAction(bool external=false) { _external = external; }
    bool isExternal() const { return _external; }

    virtual bool execute(FormatAction*) = 0;

  private:
    bool _external;   // true if this is an "external" action, i.e., not
    // one defined in a constructor
};

template <class T>
class ClassApplyFormatAction :  public ApplyFormatAction {
  public:
    ClassApplyFormatAction(T* obj, bool (T::*func)(FormatAction*),
                             bool external=false)
        : ApplyFormatAction(external)
    {
        _obj = obj; _func = func;
    }
    virtual bool execute(FormatAction* action) {
        return (_obj->*_func)(action);
    }

  private:
    T* _obj;      // the object we are working with
    bool (T::*_func)(FormatAction*);  // _obj's method to call
};

class ApplyFormatRoutine : public ApplyFormatAction {
  public:
    ApplyFormatRoutine(bool (*func)(FormatAction*),
                         bool external = false)
        : ApplyFormatAction(external) { _func = func; }
    virtual bool execute(FormatAction* action) { return (*_func)(action); }
  private:
    bool (*_func)(FormatAction*);   // the function to call
};

class CustomSpec : public FormatSpec {
    UTILS_BASE_HEADER(CustomSpec);

  public:
    CustomSpec();
    virtual ~CustomSpec();

    virtual bool act(FormatAction*);
    
    bool addFormatAction(Type action_type, ApplyFormatAction* imp);
    bool addFormatRoutine(Type action_type, bool (*)(FormatAction*),
                            bool external=false);
    void copyExternalsTo(CustomSpec* other);
    bool canApplyAction(FormatAction* action);
    virtual bool applyAction(FormatAction* action);
    
    static void initClass();

  private:
    /* the dictionary which translates FormatAction types to
       ApplyFormatAction*'s */
    Dict<ApplyFormatAction*>* _actions;
};

__UTILS_END_NAMESPACE

#endif
