///////////////////////////////////////////////////////////////////////////////
//
//                              BufferSizeAction.h
//
// Defines the action which calculates size of buffer needed to hold the 
// flattened data of a data structure specified by a format.  For efficiency
// reasons the size returned might include some alignment padding, but
// the buffer size needed to hold the flattened data will never be larger
// than the returned value
//
// Classes defined for export:
//   BufferSizeAction
//
///////////////////////////////////////////////////////////////////////////////

#ifndef utils_buffer_size_action_h
#define utils_buffer_size_action_h

#include <utils/formatting/DataAction.h>

__UTILS_BEGIN_NAMESPACE

class BufferSizeAction : public DataAction {
    UTILS_BASE_HEADER(BufferSizeAction);
  public:
    BufferSizeAction(void* data = NULL);

    void resetBufSize() { _buf_size = 0; }
    int getBufSize() const { return _buf_size; }
    void addBufSize(int s) { _buf_size += s; }

    // act on fixed primitives
    virtual bool actInt() {
        _buf_size += sizeof(int); DataAction::actInt(); return true; }
    virtual bool actShort() {
        _buf_size += sizeof(short); DataAction::actShort(); return true; }
    virtual bool actLong() {
        _buf_size += sizeof(long); DataAction::actLong(); return true; }
    virtual bool actFloat() {
        _buf_size += sizeof(float); DataAction::actFloat(); return true; }
    virtual bool actDouble() {
        _buf_size += sizeof(double); DataAction::actDouble(); return true; }
    virtual bool actChar() {
        _buf_size += sizeof(char); DataAction::actChar(); return true; }
    virtual bool actLength(int size) {
        _buf_size += size; DataAction::actLength(size); return true; }

    virtual bool actString();
    virtual bool actPtr(FormatSpec*);

    virtual bool actFixedArray(const Vector<int>& sizes, FormatSpec* fmt);
    virtual bool actVarArray(StructSpec*,
                             const Vector<int>&, FormatSpec*);

    static int size(FormatSpec*, void*);
    static void initClass();

  private:
    int _buf_size;  // current accumulated buffer size
};

__UTILS_END_NAMESPACE

#endif
