/****************************************************************************

                               SymbolTable.cc

 Defines classes and templates for managing a symbol table

 Classes defined for export:
   SymbolTable - the symbol table class
   Symbol - the actual entry into the symbol table
   SymbolManager - class for memory management of symbols
   ManagedSymManager - class for memory management of symbols of reference
     counted data
   StringSymManager - class for memory management of symbols with string
     data

 Templates defined for export
   SymbolDeleter - template to create memory managment classes for symbols
     that need to be deleted

****************************************************************************/

#ifndef UTILS_SYMBOL_TABLE_H
#define UTILS_SYMBOL_TABLE_H

#include <utils/StringDict.h>

#ifdef __cplusplus

#include <utils/Managed.h>

__UTILS_BEGIN_NAMESPACE

// SymbolManager - class for memory management of symbols
class SymbolManager : public Managed {
  public:
    virtual ~SymbolManager() {}
    virtual void link(void*) {}
    virtual void unlink(void*) {}

    virtual bool isLast() const { return false; }
    virtual const char* getTypeName() const { return "SymbolManager"; }
};

// SymbolDeleter - template to create memory managment classes for symbols
template <class T> class SymbolDeleter : public SymbolManager {
  public:
    virtual ~SymbolDeleter() {}
    virtual void unlink(void* data) { delete (T*) data; }
};

// ManagedSymManager - class for memory management of symbols with data
//   derived from Managed
class ManagedSymManager : public SymbolManager {
  public:
    virtual ~ManagedSymManager() {}
    virtual void link(void* data) { if (data) ((Managed*) data)->ref(); }
    virtual void unlink(void* data) { if (data) ((Managed*) data)->unref(); }
};

// StringSymManager - class for memory management of symbols with string data
class StringSymManager : public SymbolManager {
 public:
  virtual void unlink(void* data) { delete [] (char*) data; }
};

// Symbol - the actual entry into the symbol table
struct Symbol {
    void* data;                 // the data
    bool valid;               // true if valid data
    SymbolManager* cleaner;   // used to cleanup and manage the data
};

//   SymbolTable - the symbol table class
class SymbolTable : public StringDict<Symbol*> {
  public:
    SymbolTable(int num_buckets = 250);

    virtual ~SymbolTable();

    bool set(const char* key, const void* data,
             SymbolManager* cleaner = (SymbolManager*) NULL,
             bool overwrite = false);
    bool remove(const char* key);
    void clear();

    bool valid(const char* key) {
        Symbol* symbol;
        return (find(key, symbol) && symbol->valid);
    }
    void* get(const char* key) const;
    bool get(const char* key, void*& value) const;

    // some convenience functions that assume that strings are stored 
    // in the symbol table
    int getInt(const char* name) const;
    float getFloat(const char* name) const;
    double getDouble(const char* name) const;
    const char* getString(const char* name) const
        { return (const char*) get(name); }

    static void init();

    // convenience static for managing string data
    static StringSymManager* stringManager;
    // convenience static for managing Managed derived data
    static ManagedSymManager* managedManager;
};

__UTILS_END_NAMESPACE

typedef __UTILS::SymbolTable *UTL_SymbolTable;

extern "C" {

#else 
typedef void *UTL_SymbolTable;
#endif

typedef void (*SYM_ManageFunc)(void*, void*);
typedef void (*SYM_CallbackFunc)(const char*, void*, void*);

extern UTL_SymbolTable SYM_new(int num_buckets);
extern void SYM_delete(UTL_SymbolTable);
extern bool SYM_set(UTL_SymbolTable, const char*, void* data,
                    SYM_ManageFunc linker, SYM_ManageFunc unlinker,
                    void* cb_data, bool overwrite);
extern bool SYM_remove(UTL_SymbolTable, const char*);
extern void SYM_clear(UTL_SymbolTable);
extern void* SYM_get(UTL_SymbolTable, const char*);
extern bool SYM_probe(UTL_SymbolTable table, const char* name, void** res);
extern int SYM_getInt(UTL_SymbolTable, const char* name);
extern float SYM_getFloat(UTL_SymbolTable, const char* name);
extern double SYM_getDouble(UTL_SymbolTable, const char* name);
extern const char* SYM_getString(UTL_SymbolTable, const char* name);
extern void SYM_applyToAll(UTL_SymbolTable, SYM_CallbackFunc, void*);

#ifdef __cplusplus
}
#endif

#endif
