///////////////////////////////////////////////////////////////////////////////
//
//                               Output.h
//
// Defines a class that outputs to either a file or to a buffer.
// The class can open files by name, and ance an output destination is 
// established, the class provides methods for writing out strings and 
// numbers, as well as conveniently indenting the output file
//
// Classes defined for export:
//     Output - the output class
//
///////////////////////////////////////////////////////////////////////////////

#ifndef UTILS_OUTPUT_h
#define UTILS_OUTPUT_h

#include <stdio.h>
#include <utils/Basic.h>

__UTILS_BEGIN_NAMESPACE
    
class String;
class Name;

class Output {
  public:
    // Constructor (default Output writes to stdout)
    Output();
    Output(const Output&);

    // Destructor closes file if Output opened it.
    ~Output();

    // Sets file pointer to write to
    void		setFilePointer(FILE *newFP);

    // Returns file pointer writing to, or NULL if writing to buffer
    FILE *		getFilePointer() const;

    // Opens named file, sets file pointer to result. Returns false on error
    bool		openFile(const char *fileName, bool append=false);

    // Closes file if opened with openFile. (Does nothing if not.)
    void		closeFile();

    // Sets up buffer to write to and size and offset in buffer at which to 
    // begin writing.
    void		setBuffer(void *bufPointer, int initSize,
                          bool (*resizeProc)(void*&, int&,
                                               char*&, void*) = 0L,
                          void* callback_data = NULL, int offset = 0);

    // Returns pointer to buffer and the total bytes written in the buffer
    // from the start of the buffer (not from offset!)
    // Returns false if not writing into buffer.
    bool		getBuffer(void *&bufPointer, int &nBytes) const;

    // The actual number of bytes allocated to the buffer may be larger
    // than the number of bytes written. This returns that actual number.
    int		getBufferSize() const { return _buf_size; }
    
    // Resets buffer for output again; output starts over at beginning
    void		resetBuffer();

    // Increments/decrements indentation level by amount (default 1).
    // Each indentation level is 4 spaces.
    void		incrementIndent(int amount = 1)
	{ _indent_level += amount; }
    void		decrementIndent(int amount = 1)
	{ _indent_level -= amount; }

    // Writes item of particular type to current file pointer/buffer
    void		write(char	     c);
    void		write(const char    *s);
    void		write(const String &s);
    void		write(const Name   &n);
    void		write(int	     i);
    void		write(unsigned int   i);
    void		write(short	     s);
    void		write(unsigned short s);
    void		write(float	     f);
    void		write(double	     d);
    void                write(unsigned char*, int size);

    void flush();

    // Writes indentation to file/buffer based on current indentation level
    void		indent();

    // Resets things for writing to a new file or changing files
    void		reset();

    // Prevents header and extra white space from being written.
    // Useful for producing compact strings of data.
    void		setCompact(bool flag)		{ _compact = flag; }
    bool		isCompact() const		{ return _compact; }

    static bool standardResize(void*&, int&, char*&, void*);
  private:
    bool resizeBuffer();

  private:
    FILE		*_fp;		// File writing to
    bool		_to_buffer;	// true if writing to buffer
    void		*_buffer;	// Buffer writing to
    char		*_cur_buf;	// Current pointer in buffer
    int		_buf_size;	// Maximum buffer size
    int			_indent_level;	// Current indentation level
    bool              _compact;       // true if eliminating white-space
    bool              _opened_here;   // true if Output opened file
    bool (*_resize_proc)(void*&, int&, char*&, void*);
    void* _callback_data;   // opaque data passed to _resize_proc

    // Returns true if writing into memory buffer rather than file
    bool		isToBuffer() const { return _to_buffer; }

    // Returns number of bytes in current buffer
    int	  bytesInBuf() const { return (_cur_buf - (char *) _buffer); }
};

__UTILS_END_NAMESPACE

#endif

