///////////////////////////////////////////////////////////////////////////////
//
//                                 List.h
//
// This file defines the class for a simple linked list 
// abstract data type.  
//
// This code began its life as C code gotten from Chris Fedor for TCA, it
// mutated into C++ code and from there to the basis for a templated list.
// Its varied pedigree is what results in some of the redundancy in how
// you can iterate through a list, either through passing in a function
// to be called for each element or through using a ListIteratorBase
//
// Classes defined for export:
//    ListBase - a linked list of void*
//    ListIteratorBase - iterator for a linked list of void*
//
// Templates defined for export:
//    List<T> - list of T*
//    ListIterator<T> - iterator of a list of T*
//    ManagedList<T> - list of T* which deletes each element when the list
//                     is deleted
//
///////////////////////////////////////////////////////////////////////////////

#ifndef UTILS_LIST_H
#define UTILS_LIST_H

#include <utils/Basic.h>

__UTILS_BEGIN_NAMESPACE

struct ListElem;
class ListIteratorBase;

// ListBase - a linked list of void*
class ListBase {
  public:
    ListBase();
    ListBase(void*);
    ListBase(void*, void*);
    ~ListBase();
    
    void prepend(void*);
    void append(void*);
    void insertAfter(void*, void*);
    void* pop();
    void* remove(int (*)(void*, void*), void*);
    bool remove(void*);
    void removeAll(int (*)(void*, void*), void*);
    void removeAll(void*);
    void clear();
    bool member(void*);
    void* find(int (*)(void*, void*), void*);
    bool iterate(int (*)(void*, void*), void*);
    bool iterateFromLast(int (*)(void*, void*), void*);
    ListBase* copy();
    bool equal(ListBase*);
    void insertUnique(void*);
    void* first() const;
    void* last() const;
    bool empty() const { return _length == 0; }

    int length() { return _length; }

    static void clearCellList();
    
  private:
    void initialize();

  private:
    int _length;         // the number of elements in the list
    ListElem* _first;  // the first element in the list
    ListElem* _last;   // the last element in the list 

    friend class ListIteratorBase;
};

// ListIteratorBase - iterator for a linked list of void*
class ListIteratorBase {
 public:
  ListIteratorBase(ListBase& list);

  void* first();
  void* next();
  void removeCurrent();
  ListElem* getCurrent() const;
  void setCurrent(ListElem*);
  void insertAfterCurrent(void*);
  void insertBeforeCurrent(void*);

 private:
  ListBase& _list;   // the list we are iterating over
  ListElem* _next;   // the current element
};

// List<T> - list of T*
template <class T> class List : public ListBase {
  public:
    List() : ListBase() {}
    List(T* item1) : ListBase((void*) item1) {}
    List(T* item1, T* item2) : ListBase((void*) item1, (void*) item2) {}
    
    void prepend(T* item) { ListBase::prepend((void*) item); }
    void append(T* item) { ListBase::append((void*) item); }
    void insertAfter(T* item, const T* ref) {
        ListBase::insertAfter((void*) item,(void*) ref);
    }
    T* pop() { return (T*) ListBase::pop(); }
    T* remove(int (*match)(void*, T*), void* param) {
        return (T*) ListBase::remove((int (*)(void*, void*)) match, param);
    }
    bool remove(const T* item) { return ListBase::remove((void*) item); }
    void removeAll(int (*match)(void*, T*), void* param) {
        ListBase::removeAll((int (*)(void*, void*)) match, param);
    }
    void removeAll(const T* item) { ListBase::removeAll((void*) item); }
    bool member(const T* item) { return ListBase::member((void*) item); }
    T* find(int (*match)(void*, T*), void* param) {
        return (T*) ListBase::find((int (*)(void*, void*)) match, param);
    }
    bool iterate(int (*func)(void*, T*), void* param) {
        return ListBase::iterate((int (*)(void*, void*)) func, param);
    }
    bool iterateFromLast(int (*func)(void*, T*), void* param) {
        return ListBase::iterateFromLast((int (*)(void*, void*)) func, param);
    }
    List* copy() { return (List<T>*) ListBase::copy(); }
    bool equal(List* other) { return ListBase::equal(other); }
    void insertUnique(T* item) { ListBase::insertUnique((void*) item); }
    T* first() const { return (T*) ListBase::first(); }
    T* last() const { return (T*) ListBase::last(); }
};

// ListIterator<T> - iterator of a list of T*
template <class T> class ListIterator : public ListIteratorBase {
 public:
  ListIterator(List<T>& list) : ListIteratorBase(list) {}

  T* first() { return (T*) ListIteratorBase::first(); }
  T* next() { return (T*) ListIteratorBase::next(); }
  void insertBeforeCurrent(T* item) {
    ListIteratorBase::insertBeforeCurrent((void*) item);
  }
  void insertAfterCurrent(T* item) {
    ListIteratorBase::insertAfterCurrent((void*) item);
  }
};

template <class T> class ManagedList : public List<T> {
 public:
  ~ManagedList() {
    clear();
  }

  void clear() {
    ListIterator<T> iter(*this);
    for (T* cur=iter.first(); cur; cur=iter.next())
      delete cur;
    List<T>::clear();
  }
};

__UTILS_END_NAMESPACE

#endif
