///////////////////////////////////////////////////////////////////////////////
//
//                               Linear.h
//
// Declares classes for creating and using simple linear entities
//
// Classes declared for export:
//     Rotation - a 3D rotation class
//     Vec2f - a 2D floating point vector class
//     Vec2s - a 2D short vector class
//     Vec3f - a 3D floating point vector class
//     Vec3d - a 3D double precision floating point vector class
//     Vec4f - a 4D floating point vector class
//     Line - 3D line class
//     LinearVector - Templatized N-d vector class (implements the above VecXX classes.)
//     Transform - a 4x4 homogenous transform matrix class
//     Mat2f - 2D matrix class
//
///////////////////////////////////////////////////////////////////////////////

#ifndef UTILS_LINEAR_H
#define UTILS_LINEAR_H

#include <stdio.h>
#include <utils/Basic.h>

#ifdef HAVE_IOSTREAM
#include <iostream>
#endif

#include <utils/Vec3d.h>

#include <assert.h>

using namespace std;

#include "utils/message.h"


__UTILS_BEGIN_NAMESPACE

class Transform;
class Line;
class ConfigFile;




/////////////////////////////////////////////////
// Generic templatized vector

template <class T, int size>
class LinearVector
{
  // Constructor/destructor
 public:
        LinearVector() {
                int i;
                for (i = 0; i < size; i++) {
                        _data[i] = 0;
                }
        }
        LinearVector(const T *v) {
                int i;
                for (i = 0; i < size; i++) {
                        _data[i] = v[i];
                }
        }

        LinearVector(const LinearVector<T, size> &v) {
                int i;
                for (i = 0; i < size; i++) {
                        _data[i] = v._data[i];
                }
        }

        /*
         * Common case - constructing a 2 or 3 vector
         */

        LinearVector(const T &x, const T &y) {
                assert(size == 2);
                _data[0] = x;
                _data[1] = y;
        }

        LinearVector(const T &x, const T &y, const T &z) {
                assert(size == 3);
                _data[0] = x;
                _data[1] = y;
                _data[2] = z;
        }

   
  // Methods
 public:
        const LinearVector<T, size> &operator =(const LinearVector<T, size> &v) {
                int i;
                for (i = 0; i < size; i++) {
                        _data[i] = v._data[i];
                }
                return (*this);
        }

        LinearVector<T, size> cross(const LinearVector<T, size> &v) const {
                LinearVector<T, size> retval;
                assert(size == 3);
                retval._data[0] = (_data[1]*v._data[2]) - (_data[2]*v._data[1]);
                retval._data[1] = (_data[2]*v._data[0]) - (_data[0]*v._data[2]);
                retval._data[2] = (_data[0]*v._data[1]) - (_data[1]*v._data[0]);
                return retval;
        }
        double            dot( const LinearVector<T, size> &v ) const {
                int i;
                double retval = 0;
                for (i = 0; i < size; i++) {
                        retval += _data[i] * v._data[i];
                }
                return retval;
        }
  bool              equals( const LinearVector<T, size> &v, T tolerance ) const {
          int i;
          T tmp;
          for (i = 0; i < size; i++) {
                  tmp = _data[i] - v._data[i];
                  if ((tmp > tolerance)
                      || (-tmp > tolerance)) {
                          return false;
                  }
          }
          return true;
  }

  const T* getValue() const { return _data; }
  T* getValue() { return _data; }

  void              getValue( T *v ) {
          int i;
          for (i = 0; i < size; i++) {
                  v[i] = _data[i];
          }
  }
  

  double            length() const {
          int i; 
          double retval = 0;
          for (i = 0; i < size; i++) {
                  retval += _data[i] * _data[i];
          }
          return sqrt(retval);
  }

  void              negate() {
          int i;
          for (i = 0; i < size; i++) {
                  _data[i] = -_data[i];
          }
  }
  void              normalize() {
          double f = length();
          int i;
          for (i = 0; i < size; i++) {
                  _data[i] /= f;
          }
  }
   
  void              setValue( const T *v ) {
          int i;
          for (i = 0; i < size; i++) {
                  _data[i] = v[i];
          }
  }
  

  const T&                operator []( int i ) const {
          assert(i >= 0 && i < size);
          return _data[i];
  }

  T&          operator []( int i ) {
          assert (i >= 0 && i < size);
          return _data[i];
  }


  void             operator *=( const T &d ) {
          int i;
          for (i = 0; i < size; i++) {
                  _data[i] *= d;
          }
  }
  void             operator /=( const T &d ) {
          int i;
          for (i = 0; i < size; i++) {
                  _data[i] /= d;
          }
  }

  void                operator +=( const LinearVector<T, size> &u ) {
          int i;
          for (i = 0; i < size; i++) {
                  _data[i] += u._data[i];
          }          
  }
  void                operator -=( const LinearVector<T, size> &u ) {
          int i;
          for (i = 0; i < size; i++) {
                  _data[i] -= u._data[i];
          }          
  }

  LinearVector<T, size>            operator -() const {
          LinearVector<T, size> retval;
          int i;
          for (i = 0; i < size; i++) {
                  retval._data[i] = -_data[i];
          }          
          return retval;
  }
   
  LinearVector<T, size>           operator *(const T &d ) const {
          LinearVector<T, size> retval;
          int i;
          for (i = 0; i < size; i++) {
                  retval._data[i] = _data[i] * d;
          }          
          return retval;
  }
  
  LinearVector<T, size>     operator /( const T &d ) const {
          LinearVector<T, size> retval;
          int i;
          for (i = 0; i < size; i++) {
                  retval._data[i] = _data[i] / d;
          }          
          return retval;
  }
   
  LinearVector<T, size>          operator +( const LinearVector<T, size> &v ) const {
          LinearVector<T, size> retval;
          int i;
          for (i = 0; i < size; i++) {
                  retval._data[i] = _data[i] + v._data[i];
          }          
          return retval;
          
  }

  LinearVector<T, size>          operator -( const LinearVector<T, size> &v ) const {
          LinearVector<T, size> retval;
          int i;
          for (i = 0; i < size; i++) {
                  retval._data[i] = _data[i] - v._data[i];
          }          
          return retval;
  }
  bool               operator ==( const LinearVector<T, size> &v ) {
          int i;
          for (i = 0; i < size; i++) {
                  if (v._data[i] != _data[i]) {
                          return false;
                  }
          }          
          return true;
  }
  bool               operator !=( const LinearVector<T, size> &v ) {
          return !(*this == v);
  }

// Variables
 protected:
  T                _data[size];
};


typedef LinearVector<float, 2> Vec2f;
// remove to make sure nobody is using it
// typedef LinearVector<float, 3> Vec3f;
typedef LinearVector<float, 4> Vec4f;
typedef LinearVector<double, 2> Vec2d;
typedef LinearVector<double, 4> Vec4d;

///////////////////////////////////////////////////////
// class Rotation
//

// ### Note that although this class is implemented as a quaternion,
// the interface is *not* according to normal quaternion notation.  In
// particular, Rotation * Vec3f means "apply as rotation", and not do
// what is normally considered quaterion * vector multiplication.
//
// Also, the implementation ranges from moderately inefficient to
// extremely inefficient.

class Rotation
{
  // Constructor/destructor
 public:
  Rotation();
  Rotation( const double v[4] );
  Rotation( double q0, double q1, double q2, double q3 );
  Rotation( const Transform &m );
  Rotation( const Vec3d &axis, double radians );
  Rotation( const Vec3d &rotateFrom, const Vec3d &rotateTo );

  // Create a rotation from roll, pitch, yaw.  Roll is about the X
  // axis, pitch about the Y axis, yaw about the Z axis.  Before
  // you use this, check that this is your convention.  This relates
  // to the Navlab11 robot convention that Z is down, X forward, Y right.
  //
  // Note that when comparing different uses of RPY conventions you
  // will often find that the signs are negated, the application order
  // reversed, or both.  This is a *different* issue from the mapping
  // of roll, pitch, yaw to X,Y,Z.
  //
  // Two properties of three angle representations that are relevant
  // are:
  //  1] If you negate the angles and reverse the application order,
  //     you get the inverse transform, and less obviously,
  //  2] If you reverse the application order and change between
  //     rotating about fixed or moving axes, the result is the same.
  //     Fixed axes means that each rotation is done about the
  //     orthogonal axes in the pre-mapping coordinate frame.  Moving
  //     axes means that each rotation is done about the corresponding
  //     axis in the partially transformed coordinates.  In the
  //     navigation literature, the transform is usually described as
  //     being done on moving axes, whereas in our code we implement
  //     by transforming around fixes axes, as that seems simpler.
  //     You end up with the same transform for the same RPY either
  //     way, it's just how you construct the transform that differs.
  //
  // If someone says that they translate, yaw, pitch, roll, then they
  // are using the moving axes construction.  If they say they roll,
  // pitch, yaw, then translate, then they are using the fixed axes
  // construction.  Unless the signs are also reversed, in which case
  // they are using the inverse transform of the other moving/fixed
  // axis construction.
  //
  // Adding to the merriment is a property of the matrix transform
  // notation: the transform closest to the vector is applied first.
  // This has two implications:
  //  1] If row vectors are used instead of column vectors, this *also*
  //     reverses the order of transformation in the notation, since
  //     the transform is now on the right of the vector.
  //  2] When column vectors are used (as we do here), the order of
  //     application is right-to-left, the *reverse* of the order they
  //     are written in.
  //
  // You will see people using homogenous coordinates say:
  //    "first we translate, then we yaw, pitch and roll about moving axes,
  //     like this: trans * yaw * pitch * roll * vector"
  //
  // But what they are really doing is rolling, pitching and yawing
  // about fixed axes, then translating.  That is, the rotation
  // transforms are necessarily about fixed axes (if the code works)
  // because the true application order is the opposite of what they
  // say.
  //
  // Be warned that although there is some agreement that the term
  // "Euler angles" is used with respect to the moving axis
  // construction, others use the term "Euler angles", but then go on
  // to describe the fixed axis construction.
  //
  // If someone was confused they could also define roll, pitch and
  // yaw about the appropriate robot axes, but then apply
  // "roll", "pitch" and "yaw" in other orders, like 
  // "roll * yaw * pitch".  While this is a consistent and usable
  // 3-angle convention, it isn't RPY.  One of the significant
  // properties of RPY is that yaw is compass heading (modulo possible
  // sign change and/or offset due to choice of axes.)  A change in
  // roll or pitch does not affect yaw, so yaw must be applied last
  // (if around fixed axes), or first (if around moving axes.)  That
  // is, yaw must be applied when the world and robot yaw axes are
  // aligned.
  //
  // So basically you have lots of great rationalizations for
  // reversing the application order and changing the signs until
  // things come out right...
  // 
  Rotation (double roll, double pitch, double yaw);

// Methods
 public:
  const double*         getValue() const;
  void                 getValue( double &q0, double &q1,
                                 double &q2, double &q3 ) const;
  Rotation&          setValue( double q0, double q1, double q2, double q3 );
  void                 getValue( Vec3d &axis, double &radians ) const;
  void                 getValue( Transform &matrix ) const;
  Rotation&          setValue( const double q[4] );
  Rotation&          setValue( const Transform &m );
  Rotation&          setValue( const Vec3d &axis, double radians );
  Rotation&          setValue( const Vec3d &rotateFrom,
                               const Vec3d &rotateTo );

  Rotation&          operator *=( const Rotation &q );
   
  friend int           operator ==( const Rotation &q1,
                                    const Rotation &q2 );
  friend int           operator !=( const Rotation &q1,
                                    const Rotation &q2 );
  friend Rotation    operator *( const Rotation &q1,
                                 const Rotation &q2 );

  // ### NOT normal quaternion multiplication.  Rotates vector by this
  // rotation.
  friend Vec3d operator *(const Rotation &q, const Vec3d &vec) {
    Vec3d res;
    q.multVec(vec, res);
    return res;
  }
   
  bool               equals( const Rotation &r, double tolerance ) const;
  Rotation&          invert();
  Rotation           inverse() const;

  // ### NOT normal quaternion multiplication.  Rotates vector by this
  // rotation.
  void                 multVec( const Vec3d &src, Vec3d &dst ) const;
   
  static Rotation    slerp( const Rotation &rot0,
                            const Rotation &rot1, double t );
  static Rotation    identity();

  double&               operator []( int i );
  const double&         operator []( int i ) const;

 private:
  void normalize();

// Variables
 protected:
  double        quaternion[4];
};

///////////////////////////////////////////////////////
// class Line

class Line {
  // Constructor/destructor
 public:
  Line() {}
   
  // Construct a line from two points lying on the line.  If you
  // want to construct a line from a position and a direction, use
  // Line(p, p + d).
  // Line is directed from p0 to p1.
  Line(const Vec3d &p0, const Vec3d &p1);
   
   // Set values as indicated by corresponding constructor
  void                 setValue(const Vec3d &p0, const Vec3d &p1);
   
  // Accessors
  const Vec3d&       getPosition() const     { return pos; }
  const Vec3d&       getDirection() const    { return dir; }
   
  // Methods
 public:
   
  // Variables
 protected:
  // Parametric description:
  //  l(t) = pos + t * dir
  Vec3d pos;
  Vec3d dir;
   
  friend class Transform;
};

///////////////////////////////////////////////////////
// class Transform

typedef double TransMat[4][4];

class Transform
{
  // Constructor/destructor
 public:
  Transform();
  Transform( double a11, double a12, double a13, double a14,
             double a21, double a22, double a23, double a24,
             double a31, double a32, double a33, double a34,
             double a41, double a42, double a43, double a44 );
  Transform( TransMat m );

  Transform(Vec3d translate, Rotation rotate);
      
// Methods
 public:
  void set(ConfigFile& transform);
  void                 setValue( const TransMat &m );
  void                 setValue(double a11, double a12, double a13, double a14,
                                double a21, double a22, double a23, double a24, 
                                double a31, double a32, double a33, double a34, 
                                double a41, double a42, double a43, double a44);
  void                 getValue( TransMat &m ) const;
  const TransMat&         getValue() const;

  // roll about x axis, pitch about x axis, yaw about z axis.  See
  // Rotation::Rotation(r, p, y) for discussion of RPY conventions.
  void toRPY(double& roll, double& pitch, double& yaw);

  void                 makeIdentity();
  static Transform      identity();

  void                 setRotate( const Rotation &q );
  void                 setScale( double s );
  void                 setScale( const Vec3d &s );
  void                 setTranslate( const Vec3d &t );
  void                 setTransform( const Vec3d &t, const Rotation &r );
  void                 getTransform( Vec3d &t, Rotation &r ) const;

  double                        det3( int r1, int r2, int r3,
                                     int c1, int c2, int c3 ) const;
  double                        det3() const;
  double                        det4() const;
  Transform             inverse() const;
   
  Transform&            multRight( const Transform &m );
  Transform&            multLeft( const Transform &m );

  void                 multVecMatrix( const Vec3d &src,
                                      Vec3d &dst ) const;
  void                 multMatrixVec( const Vec3d &src,
                                      Vec3d &dst ) const;

  void                 multMatrixDir( const Vec3d &src,
                                      Vec3d &dst ) const;
  void                 multMatrixLine( const Line &src,
                                       Line &dst ) const;
   
  void                 print( FILE *fp ) const;
   
  operator             double*();
  operator             TransMat&();

  double*               operator []( int i );
  const double*         operator []( int i ) const;

  Transform&            operator =( const TransMat &m );
  Transform&            operator =( const Transform &m );
  Transform&            operator =( const Rotation &q );
   
  Transform&            operator *=( const Transform &m );

  friend Vec3d operator *(const Transform &trans, const Vec3d &v) {
    Vec3d res;
    trans.multMatrixVec(v, res);
    return res;
  }

  friend Transform      operator *( const Transform &m1, const Transform &m2 );

  friend int           operator ==( const Transform & m1, const Transform &m2 );
  friend int           operator !=( const Transform & m1, const Transform &m2 );

  bool               equals( const Transform &m, double tolerance ) const;

  friend Transform      operator*(double f, const Transform& m);
  friend Transform      operator*(const Transform& m, double f)  { return f*m; }
  friend Transform      operator/(double f, const Transform& m)  {
    return (1.0f/f)*m; }
  friend Transform      operator/(const Transform& m, double f)  {
    return (1.0f/f)*m; }
  friend Transform      operator+(const Transform& m1, const Transform& m2);
  friend Transform      operator-(const Transform& m1, const Transform& m2);
   
  // Variables
 protected:
  void                 getAdjoint(Transform&) const;
  TransMat matrix;
};


//// Mat2d class:

typedef double Mat2dData[2][2];

class Mat2d {
public:
  // the data.
  Mat2dData data;

  // Data is initially zero.
  Mat2d () {
    data[0][0] = 0;
    data[0][1] = 0;
    data[1][0] = 0;
    data[1][1] = 0;
  }

  // Initialize from array in row-major format.
  Mat2d (const Mat2dData &mat) {
    data[0][0] = mat[0][0];
    data[0][1] = mat[0][1];
    data[1][0] = mat[1][0];
    data[1][1] = mat[1][1];
  }

  Mat2d (double a11, double a12, double a21, double a22) {
    data[0][0] = a11;
    data[0][1] = a12;
    data[1][0] = a21;
    data[1][1] = a22;
  }

  Mat2d (double mat[2][2]) {
    data[0][0] = mat[0][0];
    data[0][1] = mat[0][1];
    data[1][0] = mat[1][0];
    data[1][1] = mat[1][1];
  }

  static Mat2d identity () {
    Mat2d res;
    res.data[0][0] = 1;
    res.data[1][1] = 1;
    return res;
  }

  double det () const {
    return data[0][0] * data[1][1] - data[1][0] * data[0][1];
  }

  Mat2d inverse () const {
    double d = det();
    if (d==0) {
      fprintf(stdout, "singular matrix to Mat2d::inverse.\n");
      abort();
    }
    double d_inv = 1/d;
    Mat2d res;
    res.data[0][0] = d_inv * data[1][1];
    res.data[0][1] = d_inv * -data[0][1];
    res.data[1][0] = d_inv * -data[1][0];
    res.data[1][1] = d_inv * data[0][0];
    return res;
  };

  Mat2d transpose () const {
    Mat2d res;
    res.data[0][0] = data[0][0];
    res.data[1][1] = data[1][1];
    res.data[1][0] = data[0][1];
    res.data[0][1] = data[1][0];
    return res;
  };
   
  Mat2d operator *(const Mat2d &m2) const {
    Mat2d res;
    res.data[0][0] = data[0][0] * m2.data[0][0]
      + data[0][1] * m2.data[1][0];

    res.data[0][1] = data[0][0] * m2.data[0][1]
      + data[0][1] * m2.data[1][1];

    res.data[1][0] = data[1][0] * m2.data[0][0]
      + data[1][1] * m2.data[1][0];

    res.data[1][1] = data[1][0] * m2.data[0][1]
      + data[1][1] * m2.data[1][1];
    return res;
  };

  utils::Vec2f operator *(const utils::Vec2f &vec) const {
    utils::Vec2f retval;
    retval[0] = data[0][0] * vec[0] + data[0][1] * vec[1];
    retval[1] = data[1][0] * vec[0] + data[1][1] * vec[1];
    return retval;
  };

  Mat2d operator *(const double k) const {
    Mat2d res;
    res.data[0][0] = data[0][0] * k;
    res.data[0][1] = data[0][1] * k;
    res.data[1][0] = data[1][0] * k;
    res.data[1][1] = data[1][1] * k;
    return res;
  };

  Mat2d operator +(const Mat2d& m2) {
    Mat2d res;
    res.data[0][0] = data[0][0] + m2.data[0][0];
    res.data[0][1] = data[0][1] + m2.data[0][1];
    res.data[1][0] = data[1][0] + m2.data[1][0];
    res.data[1][1] = data[1][1] + m2.data[1][1];
    return res;
  };
};


#ifdef HAVE_IOSTREAM
// prints out the Mat2d as a 2x2 matrix
std::ostream& operator<<(std::ostream& it, const Mat2d &printing);

// prints out a vector
std::ostream& operator<<(std::ostream& it, const Vec2d& printing);

// prints out the transform as a four by four matrix
std::ostream& operator<<(std::ostream& it, const Transform &printing);

// prints as {x, y}
std::ostream& operator<<(std::ostream& it, const Vec2f &printing);
#endif

__UTILS_END_NAMESPACE

#endif
