///////////////////////////////////////////////////////////////////////////////
//
//                               Dict.h
//
// Declares a dictionary template, i.e., it associates unsigned long key's 
// with values of any type T.  The base implementation goes from unsigned
// long keys to void*
//
// Classes implemented for export:
//     DictBase - the base dictionary class
//     DictEntry - an entry in a base dictionary
//
// Templates implemented for export:
//     Dict<T> - the template wrapper for DictBase
//    
///////////////////////////////////////////////////////////////////////////////

#ifndef mf_dict_h
#define mf_dict_h

#include <utils/Basic.h>

__UTILS_BEGIN_NAMESPACE

class DictEntry {
  private:
    unsigned long		key;
    void *		value;
    DictEntry *	next;
    DictEntry(unsigned long k, void *v)	{ key = k; value = v; };

    friend class DictBase;
    friend class DictIteratorBase;
};

class DictBase {
  public:
    DictBase( int entries = 251 );
    ~DictBase();
    void	clear();
    bool	enter(unsigned long key, void *value);
    bool	find(unsigned long key, void *&value) const;
    bool	remove(unsigned long key);
    bool	isEmpty();
    void	applyToAll(void (*rtn)(unsigned long key, void *value) );
    void	applyToAll(void (*rtn)(unsigned long key, void *value,
                                       void *data), 
                           void *data );

  private:
    DictEntry *&	findEntry(unsigned long key) const;

  private:
    int	_table_size;
    DictEntry **_buckets;

    friend class DictIteratorBase;
};

template <class T>
class Dict : public DictBase {
  public:
    // create a dictionary with entries buckets
    Dict(int entries = 251) : DictBase(entries) {}

    // Calls given routine (passing value) for each entry in dictionary.
    //  The order of entries is not guaranteed to mean anything.
    void	applyToAll(void (*rtn)(unsigned long key, T value) ) {
        DictBase::applyToAll((void (*)(unsigned long, void*)) rtn);
    }
    
    // Like above, but with a data entry for callback usage
    void applyToAll(void (*rtn)(unsigned long key, T value, void *data), 
                    void *data ) {
        DictBase::applyToAll((void (*)(unsigned long, void*, void*)) rtn,
                               data);
    }
    
    // Add an entry to the dictionary
    bool enter(unsigned long key, T value) {
        return DictBase::enter(key, (void*) value);
    }

    // Find an entry in the dictionary
    bool find(unsigned long key, T& value) {
        void* v = (void*) value;
        bool res = DictBase::find(key, v);
        value = (T) v;
        return res;
    }
};

// DictIteratorBase - iterator for a linked list of void*
class DictIteratorBase {
 public:
  DictIteratorBase(DictBase& dict);

  void* first();
  void* next();

 private:
  DictBase& _dict;   // the dict we are iterating over
  DictEntry* _next;   // the current element
  int _index;
};

// DictIterator<T> - iterator of a dict of T
template <class T> class DictIterator : public DictIteratorBase {
 public:
  DictIterator(Dict<T>& dict) : DictIteratorBase(dict) {}

  T first() { return (T) DictIteratorBase::first(); }
  T next() { return (T) DictIteratorBase::next(); }
};

template <class T> class ManagedDict : public Dict<T*> {
 public:
  ~ManagedDict() {
    clear();
  }

  void clear() {
    DictIterator<T*> iter(*this);
    for (T* cur=iter.first(); cur; cur=iter.next())
      delete cur;
    Dict<T*>::clear();
  }
};

__UTILS_END_NAMESPACE

#endif
