/***************************************************************************

                                ConfigElem.h

  Defines abstract base classes of elements of a ConfigFile (see ConfigFile.h
  and ConfigFile.cc).

  Classes defined for export:
    ConfigElem - the abstract base class of all ConfigFile elements
    PrimElem - the abstract base class of all primitive atomic elements, such
      as integers, floats, doubles, booleans, characters, strings, etc.

***************************************************************************/

#ifndef UTILS_CONFIG_ELEM_H
#define UTILS_CONFIG_ELEM_H

#include <utils/BaseSub.h>
#include <utils/Vector.h>
#include <utils/String.h>

__UTILS_BEGIN_NAMESPACE

class Input;
class Output;
class ConfigWatch;

// ConfigElem - the abstract base class of all ConfigFile elements
class ConfigElem : public Managed {
    UTILS_BASE_ABSTRACT_HEADER(ConfigElem);
public:
    ConfigElem();
    virtual ~ConfigElem();

    virtual void clear() = 0;
    virtual bool parseVal(const char* val) = 0;
    virtual void writeData(Output& output) = 0;
    virtual bool readData(Input& input) = 0;
    virtual bool setExpectedSize(int size) { return false; }
    virtual int getExpectedSize() const { return -1; }
    virtual int getValue(Type dest_type, void* data, int max_num=1) const = 0;
    virtual bool setValue(Type dest_type, void* data, int num_values) = 0;

    virtual int numValues() const { return 1; }
    virtual unsigned char* getData() = 0;
    virtual bool copy(ConfigElem* src) = 0;

    ConfigElem* clone() const;

    int getLineNum() const { return _line_num; }
    void setLineNum(int num) { _line_num = num; }

    ConfigWatch* watcher() const { return _watcher; }
    void setWatcher(ConfigWatch* watcher);
    void invokeSetWatch();

    static void initClass();

protected:
    static void input_error(const char* err_msg, Input& input);
    static void get_error(const char* msg, const ConfigElem* elem);

private:
    int _line_num;   // current line number
    ConfigWatch* _watcher;
};

// PrimElem - the abstract base class of all primitive atomic elements
class PrimElem : public ConfigElem {
public:
    PrimElem() { _expected_size = -1; _num_values = 0;}
    
    virtual bool parseVal(const char* val);
    virtual bool parsePrim(const char* val) = 0;
    virtual void writeData(Output& output);
    virtual bool readData(Input& input);
    virtual bool setValue(Type dest_type, void* data, int num_values);
    virtual int numValues() const { return _num_values; }
    virtual bool setExpectedSize(int size) {
        _expected_size = size;
        return true;
    }
    virtual int getExpectedSize() const { return _expected_size; }

    virtual void clear() = 0;
    virtual void writePrim(Output& input, int index) = 0;
    virtual bool setPrim(Type, void*, int index) = 0;

 protected:
    void setNumValues(int n) { _num_values = n; }

private:
    int _expected_size;  // expected number of values before reading (-1 for
                         // no expectations)
    int _num_values;     // the actual number of values after reading
};

///////////////////////////////////////////////////////////////////////////
// StringElem - holds an array of strings
class StringElem : public PrimElem
{
    UTILS_BASE_HEADER(StringElem);
public:
    StringElem();
    virtual ~StringElem();

    virtual void clear();
    virtual bool parsePrim(const char* val);
    virtual void writePrim(Output& output, int index);
    virtual bool setPrim(Type, void* data, int index);
    virtual int getValue(Type dest_type, void* data, int max_num=1) const;
    virtual bool copy(ConfigElem* src);

    const char* get(int i) const { return _data[i]->getString(); }
    void append(const char* v) { _data.append(new String(v)); }
    virtual unsigned char* getData() {
      return (unsigned char*) _data.getData();
    }

    static void initClass();

private:
    Vector<String*> _data;    // the data
};

__UTILS_END_NAMESPACE

#endif
